/* $Id: vircam_tfits.c,v 1.13 2007/10/25 17:34:01 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 17:34:01 $
 * $Revision: 1.13 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <math.h>
#include <string.h>

#include <cpl.h>
#include "vircam_utils.h"
#include "vircam_tfits.h"

/**
    \defgroup vircam_tfits vircam_tfits
    \ingroup supportroutines

    \brief
    These are methods for manipulating the vircam_tfits object

    \author
    Jim Lewis, CASU
*/

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_load
    \par Purpose:
        Load an input table into a vircam_tfits object
    \par Description:
        A table from an input cpl_frame is loaded into a vircam_tfits object.
	Only the table for a given extension number is loaded. The rest
	of the object properties are initialised
    \par Language:
        C
    \param table
        The input cpl_frame object
    \param nexten
        The image extension that you want to load.
    \return
        The output vircam_tfits object.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_tfits *vircam_tfits_load(cpl_frame *table, int nexten) {
    vir_tfits *p;
    cpl_table *tab;
    int nf;
    const char *fctid = "vircam_tfits_load";

    /* Check for nonsense input */

    if (table == NULL)
	return(NULL);

    /* See if you can load the table */

    tab = cpl_table_load(cpl_frame_get_filename(table),nexten,0);
    if (tab == NULL) {
        cpl_msg_error(fctid,"Unable to load %s -- %s\n",
                      cpl_frame_get_filename(table),cpl_error_get_message());
        cpl_error_reset();
        return(NULL);
    }

    /* Get the vir_tfits structure */

    p = cpl_malloc(sizeof(vir_tfits));

    /* Load stuff in */

    p->table = tab;
    p->nexten = nexten;
    p->phu = NULL;
    p->ehu = NULL;
    p->fname = cpl_strdup(cpl_frame_get_filename(table));
    p->status = VIR_OK;
   
    /* Get the extension header and the extension name */

    (void)vircam_tfits_get_ehu(p);
    if (cpl_propertylist_has(p->ehu,"EXTNAME")) {
        p->extname = cpl_strdup(cpl_propertylist_get_string(p->ehu,"EXTNAME"));
    } else {
        nf = 11 + (int)log10((double)nexten);
        p->extname = cpl_malloc(nf);
        (void)snprintf(p->extname,nf,"DET1.CHIP%d",nexten);
    }
    nf = strlen(p->extname) + strlen(p->fname) + 3;
    p->fullname = cpl_malloc(nf);
    (void)snprintf(p->fullname,nf,"%s[%s]",p->fname,p->extname);

    /* Get out of here */

    return(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_extract
    \par Purpose:
        Extract selected rows from a vir_tfits table
    \par Description:
        A vir_tfits structure is passed in containing a table that has
	had some rows selected. A new vir_tfits table is returned with only
	the selected rows.
    \par Language:
        C
    \param in
        The input vir_tfits object
    \return
        The output vir_tfits object.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_tfits *vircam_tfits_extract(vir_tfits *in) {
    vir_tfits *p;

    /* Check for nonsense input */

    if (in == NULL)
	return(NULL);

    /* Get the vir_tfits structure */

    p = cpl_malloc(sizeof(vir_tfits));

    /* Load stuff in */

    p->table = cpl_table_extract_selected(vircam_tfits_get_table(in));
    p->nexten = vircam_tfits_get_nexten(in);
    p->phu = NULL;
    p->ehu = NULL;
    p->fname = cpl_strdup(vircam_tfits_get_filename(in));
   
    /* Get out of here */

    return(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_load_list
    \par Purpose:
        Load a input tables into a vircam_tfits object array
    \par Description:
        Tables from an input cpl_frameset are loaded into a list of 
	vircam_tfits objects. Only the tables for a given extension number are
	loaded. The rest of the object properties are initialised
    \par Language:
        C
    \param f
        The input cpl_frameset object
    \param exten
        The image extension that you want to load.
    \return
        The output vircam_tfits object list.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_tfits **vircam_tfits_load_list(cpl_frameset *f, int exten) {
    int i;
    vir_tfits **p;

    /* Check for nonsense input */

    if (f == NULL)
	return(NULL);

    /* Get some workspace */

    p = cpl_malloc(cpl_frameset_get_size(f)*sizeof(vir_tfits *));
    
    /* Now load each of the frames... */

    for (i = 0; i < cpl_frameset_get_size(f); i++) {
	p[i] = vircam_tfits_load(cpl_frameset_get_frame(f,i),exten);
	if (p[i] == NULL) {
	    vircam_tfits_delete_list(p,i-1);
	    return(NULL);
	}
    }

    /* Now return the array */

    return(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_delete
    \par Purpose:
        Free all the workspace associated with a vircam_tfits object
    \par Description:
        Free all the workspace associated with a vircam_tfits object
    \par Language:
        C
    \param p
        The input vir_tfits object
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_tfits_delete(vir_tfits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return;

    /* Free up workspace if it's been used */

    freetable(p->table);
    freepropertylist(p->phu);
    freepropertylist(p->ehu);
    freespace(p->fname);
    freespace(p->extname);
    freespace(p->fullname);
    cpl_free(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_delete_list
    \par Purpose:
        Free all the workspace associated with a list of vircam_tfits objects
    \par Description:
        Free all the workspace associated with a list of vircam_tfits objects
    \par Language:
        C
    \param p
        The input list of vir_tfits objects
    \param n
        The number of vir_tfits objects in the above array
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_tfits_delete_list(vir_tfits **p, int n) {
    int i;

    /* Check for nonsense input */

    if (p == NULL)
	return;

    /* Free up workspace if it's been used */

    for (i = 0; i < n; i++)
	vircam_tfits_delete(p[i]);
    freespace(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_table
    \par Purpose:
        Get the CPL table from the vir_tfits object
    \par Description:
        Return the CPL table from the input vir_tfits object. This table is
	suitable for use in all cpl_table routines.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \return 
        The cpl_image object. NULL if there was an error.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_table *vircam_tfits_get_table(vir_tfits *p) {
    
    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* Return it */

    return(p->table);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_nexten
    \par Purpose:
        Get the FITS extension number for the current image in a vir_tfits 
	object
    \par Description:
        Get the FITS extension number for the current image in a vir_tfits 
	object
    \par Language:
        C
    \param p
        The input vir_tfit object
    \return 
        The extension number (-1 in case of error)
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_tfits_get_nexten(vir_tfits *p) {
    
    /* Check for nonsense input */

    if (p == NULL)
	return(-1);

    /* Return it */

    return(p->nexten);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_phu
    \par Purpose:
        Get the propertylist for the primary header for a given vir_tfits 
	image.
    \par Description:
        Get the propertylist for the primary header for a given vir_tfits 
	image. This should only need to be read once and then can be used 
	to add things to the primary header.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \return 
        The propertylist represeting the primary header of the input table
	(NULL if there is an error).
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_propertylist *vircam_tfits_get_phu(vir_tfits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* If the propertylist hasn't already been loaded, then do it now */

    if (p->phu == NULL) 
        p->phu = cpl_propertylist_load(p->fname,0);
    
    /* Return it */

    return(p->phu);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_ehu
    \par Purpose:
        Get the propertylist for the extension header for a given vir_tfits 
	image.
    \par Description:
        Get the propertylist for the extension header for a given vir_tfits 
	image. This is the extension that is relevant of the image.
	This should only need to be read once and then can be used to add
	things to the primary header.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \return 
        The propertylist represeting the extension header of the input table
	(NULL if there is an error).
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_propertylist *vircam_tfits_get_ehu(vir_tfits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* If the propertylist hasn't already been loaded, then do it now */

    if (p->ehu == NULL) 
        p->ehu = cpl_propertylist_load(p->fname,p->nexten);
    
    /* Return it */

    return(p->ehu);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_filename
    \par Purpose:
        Get the filename from which the current vir_tfits object originated
    \par Description:
        Get the filename from which the current vir_tfits object originated. If
	this is null, then the image didn't originate in an FITS file.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \return 
        The name of the file from which this table originated
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern char *vircam_tfits_get_filename(vir_tfits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* Return it */

    return(p->fname);
}
  
/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_fullname
    \par Purpose:
        Get the fullname of the FITS extension from which the current
        vir_tfits object originated
    \par Description:
        Get the fullname of the FITS extension  from which the current
        vir_tfits object originated. If this is null, then the image didn't
        originate in an FITS file.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \return
        The fullname name of the file from which this image originated
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern char *vircam_tfits_get_fullname(vir_tfits *p) {

    /* Check for nonsense input */

    if (p == NULL)
        return(NULL);

    /* Return it */

    return(p->fullname);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_get_status
    \par Purpose:
        Get the error status of the current object.
    \par Description:
        Get the error status of the current object.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \return 
        The error status
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_tfits_get_status(vir_tfits *p) {
    
    /* Check for nonsense input */

    if (p == NULL)
	return(VIR_FATAL);

    /* Return it */
   
    return(p->status);
}
  
/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_set_error
    \par Purpose:
        Set the error status and message for an object
    \par Description:
        The input status is checked to see if there has been a problem
	with the current object. If there has been, then the status is
	stored away and any error message from the cpl_error system is
	copied down.
    \par Language:
        C
    \param p
        The input vir_tfits object
    \param status
        The input error status
    \return 
        A flag to say whether the input status was fatal
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_tfits_set_error(vir_tfits *p, int status) {

    /* Check for nonsense input */

    if (p == NULL)
	return(0);

    /* Get out of here if the status is OK */

    if (status == VIR_OK)
	return(0);

    /* Set the error message if there was an error */

    p->status = status;
    
    /* Reset the cpl error flag */

    cpl_error_reset();
    if (status == VIR_FATAL) 
	return(1);
    else
	return(0);
}
    
/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_tfits_wrap
    \par Purpose:
        Wrap an table in a vir_tfits wrapper
    \par Description:
        The input table is inserted into a vir_tfits wrapper. A model
        vir_tfits object may be provided to give the new object 
        headers. If the phu and ehu parameters are not null then they will
        be used as the propertylists for the new object. If not, then
        an attempt will be made to copy the propertylists from the model.
    \par Language:
        C
    \param tab
        The input cpl_table
    \param model
        The input vir_tfits model object
    \param phu
        The input propertylist for the primary header for the new object.
    \param ehu
        The input propertylist for the extension header for the new object.
    \return
        The new vir_tfits structure.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_tfits *vircam_tfits_wrap(cpl_table *tab, vir_tfits *model,
				    cpl_propertylist *phu,
				    cpl_propertylist *ehu) {
    vir_tfits *p;

    /* Check for nonsense input */

    if (tab == NULL)
        return(NULL);

    /* Get the vir_fits structure */

    p = cpl_malloc(sizeof(vir_tfits));

    /* Load stuff in */

    p->table = tab;
    p->nexten = -1;
    if (phu != NULL)
	p->phu = phu;
    else if (model != NULL)
        p->phu = cpl_propertylist_duplicate(vircam_tfits_get_phu(model));
    else
	p->phu = NULL;
    if (ehu != NULL)
        p->ehu = ehu;
    else if (model != NULL)
        p->ehu = cpl_propertylist_duplicate(vircam_tfits_get_ehu(model));
    else
	p->ehu = NULL;
    p->fname = NULL;
    p->status = VIR_OK;
    p->extname = NULL;
    p->fullname = NULL;

    /* Get out of here */

    return(p);
}

/**@}*/

/*

$Log: vircam_tfits.c,v $
Revision 1.13  2007/10/25 17:34:01  jim
Modified to remove lint warnings

Revision 1.12  2007/10/19 09:25:10  jim
Fixed problems with missing includes

Revision 1.11  2007/10/15 12:50:28  jim
Modified for compatibility with cpl_4.0

Revision 1.10  2007/03/01 12:42:42  jim
Modified slightly after code checking

Revision 1.9  2007/02/25 06:32:40  jim
Fixed typo

Revision 1.8  2007/02/20 21:13:13  jim
Better error reporting in case of load failure

Revision 1.7  2006/07/04 09:19:05  jim
replaced all sprintf statements with snprintf

Revision 1.6  2006/06/19 11:17:58  jim
Fixed bug causing a memory overwrite

Revision 1.5  2006/06/09 11:26:26  jim
Small changes to keep lint happy

Revision 1.4  2006/05/24 13:34:36  jim
Added vircam_tfits_wrap

Revision 1.3  2006/04/20 11:21:21  jim
Added _fullname method

Revision 1.2  2006/03/22 13:58:32  jim
Cosmetic fixes to keep lint happy

Revision 1.1  2006/03/15 14:35:17  jim
new file


*/
