/* $Id: vircam_pfits.c,v 1.26 2007/11/22 12:35:33 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/11/22 12:35:33 $
 * $Revision: 1.26 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <string.h>

#include "vircam_utils.h"
#include "vircam_pfits.h"

static int vircam_pfits_get_float(const cpl_propertylist *plist, 
				  const char *key, float *fval);
static int vircam_pfits_get_double(const cpl_propertylist *plist, 
				   const char *key, double *fval);

/**
   \defgroup vircam_pfits vircam_pfits
   \ingroup supportroutines

   \brief
   These are support routines used for extracting information from FITS 
   headers.

   \author Jim Lewis, CASU
*/

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of crval1
    \param    plist       property list to read from
    \param    crval1      requested value of crval1
    \retval VIR_OK if everything went right \retval VIR_FATAL if 
        there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_crval1(const cpl_propertylist *plist, 
				   double *crval1) {

    return(vircam_pfits_get_double(plist,"CRVAL1",crval1));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of crpix1
    \param    plist       property list to read from
    \param    crpix1      requested value of crpix1
    \retval VIR_OK if everything went right \retval VIR_FATAL if 
        there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_crpix1(const cpl_propertylist *plist, 
				   double *crpix1) {

    return(vircam_pfits_get_double(plist,"CRPIX1",crpix1));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of crval2
    \param    plist       property list to read from
    \param    crval2      requested value of crval2
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_crval2(const cpl_propertylist *plist, 
				   double *crval2) {

    return(vircam_pfits_get_double(plist,"CRVAL2",crval2));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of crpix2
    \param    plist       property list to read from
    \param    crpix2      requested value of crpix2
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_crpix2(const cpl_propertylist *plist, 
				   double *crpix2) {

    return(vircam_pfits_get_double(plist,"CRPIX2",crpix2));
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of cd1_1
    \param    plist       property list to read from
    \param    cd11        requested value of cd1_1
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_cd11(const cpl_propertylist *plist, double *cd11) {

    return(vircam_pfits_get_double(plist,"CD1_1",cd11));
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of cd1_2
    \param    plist       property list to read from
    \param    cd12        requested value of cd1_2
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_cd12(const cpl_propertylist *plist, double *cd12) {

    return(vircam_pfits_get_double(plist,"CD1_2",cd12));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of cd2_1
    \param    plist       property list to read from
    \param    cd21        requested value of cd2_1
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_cd21(const cpl_propertylist *plist, double *cd21) {

    return(vircam_pfits_get_double(plist,"CD2_1",cd21));
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of cd2_2
    \param    plist       property list to read from
    \param    cd22        requested value of cd2_2
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_cd22(const cpl_propertylist *plist, double *cd22) {

    return(vircam_pfits_get_double(plist,"CD2_2",cd22));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of pv2_1
    \param    plist       property list to read from
    \param    pv21        requested value of pv2_1
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_pv21(const cpl_propertylist *plist, double *pv21) {

    return(vircam_pfits_get_double(plist,"PV2_1",pv21));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of pv2_3
    \param    plist       property list to read from
    \param    pv23        requested value of pv2_3
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_pv23(const cpl_propertylist *plist, double *pv23) {

    return(vircam_pfits_get_double(plist,"PV2_3",pv23));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of exposure time
    \param    plist       property list to read from
    \param    exptime     requested value of exptime
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_exptime(const cpl_propertylist *plist, 
				    float *exptime) {

    return(vircam_pfits_get_float(plist,"EXPTIME",exptime));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of mindit time
    \param    plist       property list to read from
    \param    mindit      requested value of mindit time
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_mindit(const cpl_propertylist *plist, 
				   float *mindit) {

    return(vircam_pfits_get_float(plist,"ESO DET MINDIT",mindit));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of dit delay time
    \param    plist       property list to read from
    \param    ditdelay    requested value of dit delay time
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_ditdelay(const cpl_propertylist *plist, 
				     float *ditdelay) {

    return(vircam_pfits_get_float(plist,"ESO DET DITDELAY",ditdelay));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of naxis1
    \param    plist       property list to read from
    \param    naxis1      requested value of naxis1
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_naxis1(const cpl_propertylist *plist, 
				   long *naxis1) {
    int val;

    val = cpl_propertylist_get_int(plist,"NAXIS1");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*naxis1 = (long)val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*naxis1 = 0;
	return(VIR_FATAL);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of naxis2
    \param    plist       property list to read from
    \param    naxis2      requested value of naxis2
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_naxis2(const cpl_propertylist *plist, 
				   long *naxis2) {
    int val;

    val = cpl_propertylist_get_int(plist,"NAXIS2");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*naxis2 = (long)val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*naxis2 = 0;
	return(VIR_FATAL);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of chipno
    \param    plist       property list to read from
    \param    chipno      requested value of chipno
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_chipno(const cpl_propertylist *plist, 
				   int *chipno) {
    int val;

    val = cpl_propertylist_get_int(plist,"ESO DET CHIP NO");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*chipno = val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*chipno = 0;
	return(VIR_FATAL);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the X jitter offset
    \param    plist       property list to read from
    \param    xoff        requested value of the x offset
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_jxoff(const cpl_propertylist *plist, 
				  float *xoff) {

    return(vircam_pfits_get_float(plist,"ESO DRS XOFFDITHER",xoff));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the Y jitter offset
    \param    plist       property list to read from
    \param    yoff        requested value of the y offset
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_jyoff(const cpl_propertylist *plist, 
				  float *yoff) {

    return(vircam_pfits_get_float(plist,"ESO DRS YOFFDITHER",yoff));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the airmass
    \param    plist       property list to read from
    \param    airmass     requested value of the airmass
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_airmass(const cpl_propertylist *plist, 
				    float *airmass) {

    return(vircam_pfits_get_float(plist,"ESO TEL AIRM START",airmass));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the number of microsteps in a sequence
    \param    plist       property list to read from
    \param    nusteps     requested value of the number of microsteps
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_nusteps(const cpl_propertylist *plist, 
				    int *nusteps) {
    int val;

    val = cpl_propertylist_get_int(plist,"NUSTEP");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*nusteps = val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*nusteps = 0;
	return(VIR_FATAL);
    }

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the first run number in the current microstep
              sequence
    \param    plist       property list to read from
    \param    ustepnum   requested value of the microstep sequence number
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_ustepnum(const cpl_propertylist *plist, 
				     int *ustepnum) {
    int val;

    val = cpl_propertylist_get_int(plist,"USTEPNUM");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*ustepnum = val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*ustepnum = 0;
	return(VIR_FATAL);
    }

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the first run number in the current jitter
              sequence
    \param    plist       property list to read from
    \param    jitternum   requested value of the jitter sequence number
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_jitternum(const cpl_propertylist *plist, 
				      int *jitternum) {
    int val;

    val = cpl_propertylist_get_int(plist,"JITTRNUM");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*jitternum = val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*jitternum = 0;
	return(VIR_FATAL);
    }

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of NDIT
    \param    plist       property list to read from
    \param    ndit        requested value of ndit
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_ndit(const cpl_propertylist *plist, 
				 int *ndit) {
    int val;

    val = cpl_propertylist_get_int(plist,"ESO DET NDIT");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*ndit = val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*ndit = 1;
	return(VIR_FATAL);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of DET_LIVE
    \param    plist       property list to read from
    \param    detlive     requested value of DET_LIVE
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_detlive(const cpl_propertylist *plist, 
				    int *detlive) {
    int val;

    val = cpl_propertylist_get_bool(plist,"ESO DET CHIP LIVE");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*detlive = val;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*detlive = 1;
	return(VIR_FATAL);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the name of the current filter.
    \param    plist       property list to read from
    \param    filt        requested name of the filter
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_filter(const cpl_propertylist *plist, char *filt) {

    strcpy(filt,cpl_propertylist_get_string(plist,"ESO INS FILT1 NAME"));
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
        return(VIR_OK);
    } else {
	cpl_error_reset();
	return(VIR_FATAL);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the saturation level for this detector
    \param    plist       property list to read from
    \param    saturation  requested name of the filter
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_saturation(const cpl_propertylist *plist, 
				       float *saturation) {
    int ival;

    ival = cpl_propertylist_get_int(plist,"ESO DET SATURATION");
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
	*saturation = (float)ival;
        return(VIR_OK);
    } else {
	cpl_error_reset();
	*saturation = 65535.0;
	return(VIR_FATAL);
    }


}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the detector gain
    \param    plist       property list to read from
    \param    gain        requested value of the gain
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_gain(const cpl_propertylist *plist, float *gain) {

    return(vircam_pfits_get_float(plist,"ESO DRS GAIN",gain));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get the value of the modified Julian date
    \param    plist       property list to read from
    \param    mjd        requested value of the mjd
    \retval VIR_OK if everything went right \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

extern int vircam_pfits_get_mjd(const cpl_propertylist *plist, double *mjd) {

    return(vircam_pfits_get_double(plist,"MJD-OBS",mjd));

}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get a floating point value
    \param    plist       property list to read from
    \param    key         the keyword for the property
    \param    fval        the output floating point value
    \retval VIR_OK if everything went right 
    \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

static int vircam_pfits_get_float(const cpl_propertylist *plist, 
				  const char *key, float *fval) {
    cpl_type type;
    const char *fctid = "vircam_pfits_get_float";

    /* Get the type of this keyword */

    type = cpl_propertylist_get_type(plist,key);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
	*fval = 0.0;
	cpl_error_reset();
	return(VIR_FATAL);
    }

    /* Switch of the property type */

    switch (type) {
    case CPL_TYPE_FLOAT:
	*fval = cpl_propertylist_get_float(plist,key);
	break;
    case CPL_TYPE_DOUBLE:
	*fval = (float)cpl_propertylist_get_double(plist,key);
	break;
    default:
	*fval = 0.0;
	cpl_msg_error(fctid,"Keyword %s is not floating point in header",key);
	return(VIR_FATAL);
    }
    return(VIR_OK);
}

/*---------------------------------------------------------------------------*/
/**
    \brief    Get a double precision value
    \param    plist       property list to read from
    \param    key         the keyword for the property
    \param    fval        the output double precision value
    \retval VIR_OK if everything went right 
    \retval VIR_FATAL if there was an error
 */
/*---------------------------------------------------------------------------*/

static int vircam_pfits_get_double(const cpl_propertylist *plist, 
				   const char *key, double *fval) {
    cpl_type type;
    const char *fctid = "vircam_pfits_get_float";

    /* Get the type of this keyword */

    type = cpl_propertylist_get_type(plist,key);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
	*fval = 0.0;
	cpl_error_reset();
	return(VIR_FATAL);
    }

    /* Switch of the property type */

    switch (type) {
    case CPL_TYPE_FLOAT:
	*fval = (double)cpl_propertylist_get_float(plist,key);
	break;
    case CPL_TYPE_DOUBLE:
	*fval = cpl_propertylist_get_double(plist,key);
	break;
    default:
	*fval = 0.0;
	cpl_msg_error(fctid,"Keyword %s is not floating point in header",key);
	return(VIR_FATAL);
    }
    return(VIR_OK);
}

/**@}*/

/* 

$Log: vircam_pfits.c,v $
Revision 1.26  2007/11/22 12:35:33  jim
Modified vircam_pfits_get_saturation so that the value of the
saturation is read as an integer, but passed back as a float

Revision 1.25  2007/11/20 09:39:39  jim
added vircam_pfits_get_mjd

Revision 1.24  2007/11/14 10:45:23  jim
Added vircam_pfits_get_saturation

Revision 1.23  2007/10/25 17:34:01  jim
Modified to remove lint warnings

Revision 1.22  2007/10/19 09:25:10  jim
Fixed problems with missing includes

Revision 1.21  2007/03/23 10:53:22  jim
Fixed little documentation errors

Revision 1.20  2007/03/01 12:42:42  jim
Modified slightly after code checking

Revision 1.19  2006/07/17 09:33:00  jim
Changed keyword for detlive routine

Revision 1.18  2006/07/11 14:52:13  jim
Added vircam_pfits_get_detlive

Revision 1.17  2006/06/20 19:00:06  jim
Added vircam_pfits_get_ndit

Revision 1.16  2006/06/13 14:08:12  jim
Added vircam_pfits_get_gain

Revision 1.15  2006/06/09 11:26:26  jim
Small changes to keep lint happy

Revision 1.14  2006/05/24 13:35:23  jim
Added _get_nusteps, _get_ustepnum, _get_jitternum and _get_filter

Revision 1.13  2006/05/17 12:05:48  jim
Add vircam_pfits_get_airmass

Revision 1.12  2006/05/15 13:15:11  jim
Added vircam_pfits_j[x,y]off

Revision 1.11  2006/04/21 15:17:48  jim
Added vircam_pfits_get_float and vircam_pfits_get_double static routines

Revision 1.10  2006/03/01 10:59:25  jim
Added vircam_pfits_get_ditdelay

Revision 1.9  2006/02/18 11:48:03  jim
added routines to get mindit and chip number

Revision 1.8  2006/01/23 10:30:49  jim
Mainly documentation mods

Revision 1.7  2005/12/14 22:17:33  jim
Updated docs

Revision 1.6  2005/11/25 15:33:22  jim
Some code fixes to keep splint happy

Revision 1.5  2005/11/25 09:56:15  jim
Tidied up some more documentation

Revision 1.4  2005/11/07 13:15:16  jim
Fixed lots of bugs and added some error checking

Revision 1.3  2005/11/07 11:23:41  jim
fixed call to cpl_error_get_code

Revision 1.2  2005/11/03 13:28:50  jim
All sorts of changes to tighten up error handling

Revision 1.1.1.1  2005/08/05 08:29:09  jim
Initial import


*/
