/* $Id: vircam_mask.c,v 1.10 2007/10/19 09:25:10 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/19 09:25:10 $
 * $Revision: 1.10 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <cpl.h>
#include "vircam_utils.h"
#include "vircam_fits.h"
#include "vircam_mask.h"
#include "vircam_dfs.h"

static unsigned char *vircam_mask_getbpm(vir_fits *bpmimage);
static unsigned char *vircam_mask_conf2bpm(vir_fits *cpmimage);

/**
    \defgroup vircam_mask vircam_mask
    \ingroup supportroutines

    \brief
    These are methods for manipulating the vircam_mask bad pixel mask objects

    \author
    Jim Lewis, CASU
*/

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_define
    \par Purpose:
        Define a mask type from a given input frameset
    \par Description
        An input frameset is searched for either a master bad pixel mask
        or a master confidence map. (BPM is searched for first). If one of
        them is found then the frame is returned along with a flag defining
        the mask type.
    \par Language:
        C
    \param framelist
        The input frame list
    \param labels
        The input frame list labels
    \param nlab
        The number of labels
    \return 
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_mask *vircam_mask_define(cpl_frameset *framelist, int *labels, 
				    int nlab) {
    cpl_frame *master_mask;
    int masktype;
    vir_mask *m;
    char *fctid = "vircam_define_mask";

    /* What kind of mask are we defining here? */

    if ((master_mask = vircam_frameset_subgroup_1(framelist,labels,nlab,
						  VIRCAM_CAL_BPM)) == NULL) {
	if ((master_mask = vircam_frameset_subgroup_1(framelist,labels,
						      nlab,VIRCAM_CAL_CONF)) == NULL) {
            cpl_msg_info(fctid,
			 "No master pixel mask found -- all pixels considered good");
	    masktype = MASK_NONE;
        } else {
	    masktype = MASK_CPM;
	}
    } else {
	masktype = MASK_BPM;
    }

    /* If a master mask is defined, then check to see if the file is 
       accessible. If it isn't then continue on as though you don't have one */

    if (master_mask != NULL) {
        if (access(cpl_frame_get_filename(master_mask),R_OK) != 0) {
	    cpl_msg_warning(fctid,"File %s is not read accessible",
			    cpl_frame_get_filename(master_mask));
	    masktype = MASK_NONE;
	    freeframe(master_mask);
	}
    }

    /* Get the vir_mask structure... */

    m = cpl_malloc(sizeof(*m));

    /* Initialise a few things */

    m->master_mask = master_mask;
    m->mask_image = NULL;
    m->masktype = masktype;
    m->nx = 0;
    m->ny = 0;
    m->mdata = NULL;

    /* Now return it */

    return(m);    
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_load
    \par Purpose:
        Load the image from an input mask object
    \par Description:
        The image from an input vir_mask object is loaded
    \par Language:
        C
    \param m
        The input mask object
    \param nexten
        The image extension that you want to load.
    \param nx
        The X dimension of the data array (in case the image is undefined)
    \param ny
        The Y dimension of the data array (in case the image is undefined)
    \return
        The usual vircam status variable
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_mask_load(vir_mask *m, int nexten, int nx, int ny) {

    /* Check for nonsense input */

    if (m == NULL)
	return(VIR_FATAL);

    /* Look to see if the sizes make sense */

    if (nx <= 0 && ny <= 0 && m->masktype == MASK_NONE)
	return(VIR_FATAL);

    /* If the mask image has already been loaded then free it up. */

    if (m->mask_image != NULL) {
	vircam_fits_delete(m->mask_image);
	freespace(m->mdata);
    }

    /* Load up the image if there is one. */

    if (m->masktype != MASK_NONE) {
        m->mask_image = vircam_fits_load(m->master_mask,CPL_TYPE_INT,nexten);
        if (m->mask_image == NULL)
	    return(VIR_FATAL);
	m->nx = cpl_image_get_size_x(vircam_fits_get_image(m->mask_image));
	m->ny = cpl_image_get_size_y(vircam_fits_get_image(m->mask_image));
    } else {
	m->nx = nx;
	m->ny = ny;
    }

    /* Return the status */

    return(VIR_OK);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_delete
    \par Purpose:
        Delete internal workspace associated with a mask object, then delete
	the object itself.
    \par Description:
        All of the internal workspace in a mask object is freed. The the 
	memory holding the mask object itself is freed.
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_mask_delete(vir_mask *m) {

    if (m == NULL)
	return;
    vircam_mask_clear(m);
    freeframe(m->master_mask); 
    freespace(m);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_clear
    \par Purpose:
        Delete internal workspace associated with a mask object, but keep
	the object itself intact.
    \par Description:
        All of the internal workspace in a mask object is freed. The the 
	memory holding the mask object itself is left alone.
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_mask_clear(vir_mask *m) {
    if (m == NULL)
	return;
    freespace(m->mdata);
    freefits(m->mask_image);
    m->nx = 0;
    m->ny = 0;
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_force
    \par Purpose:
        Forces a predefined mask out of the structure to be replaced by
	a MASK_NONE type.
    \par Description:
        This routine is used as a catch-all in case there is a problem in 
	loading an extension of a mask with vircam_mask_load. In that case
	calling this routine will force out the old mask definition and
	replace it with a safe MASK_NONE option.
    \par Language:
        C
    \param m
        The input vir_mask object
    \param nx
        The X size of the mask array
    \param ny
        The Y size of the mask array
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_mask_force(vir_mask *m, int nx, int ny) {
    if (m == NULL) 
	return;
    freespace(m->mdata);
    freefits(m->mask_image);
    freeframe(m->master_mask);
    m->masktype = MASK_NONE;
    m->nx = nx;
    m->ny = ny;
}
    
/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_get_fits
    \par Purpose:
        Get the vir_fits structure from a mask object
    \par Description:
        Get the vir_fits structure from a mask object
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        The vir_fits structure from the mask object
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_fits *vircam_mask_get_fits(vir_mask *m) {
    return(m->mask_image);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_get_filename
    \par Purpose:
        Get the filename from a vir_mask structure
    \par Description:
        Get the filename from a vir_mask structure
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        The char pointer to the file name
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern const char *vircam_mask_get_filename(vir_mask *m) {
    if (m->master_mask != NULL) {
	return(cpl_frame_get_filename(m->master_mask));
    } else {
        return(NULL);
    }
}


/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_get_size_x
    \par Purpose:
        Get the x size of the mask data array
    \par Description:
        Get the x size of the mask data array
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        The X size of the mask data array
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_mask_get_size_x(vir_mask *m) {
    return(m->nx);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_get_size_y
    \par Purpose:
        Get the y size of the mask data array
    \par Description:
        Get the y size of the mask data array
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        The Y size of the mask data array
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_mask_get_size_y(vir_mask *m) {
    return(m->ny);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_get_type
    \par Purpose:
        Get the mask type
    \par Description:
        Get the mask type
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        The mask type. See vircam_mask.h for more details
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_mask_get_type(vir_mask *m) {
    return(m->masktype);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_get_data
    \par Purpose:
        Get the mask data array
    \par Description:
        Get the mask data array
    \par Language:
        C
    \param m
        The input vir_mask object
    \return
        The bad pixel mask data array
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern unsigned char *vircam_mask_get_data(vir_mask *m) {
    unsigned char *bpm;
    long npix;

    /* Has this already been done? */

    if (m->mdata != NULL)
	return(m->mdata);

    /* Get the bpm depending on what type of input you have */

    switch (m->masktype) {
    case MASK_NONE:
	npix = (m->nx)*(m->ny);
        bpm = cpl_calloc(npix,sizeof(*bpm));
        break;
    case MASK_BPM:
        bpm = vircam_mask_getbpm(vircam_mask_get_fits(m));
        break;
    case MASK_CPM:
        bpm = vircam_mask_conf2bpm(vircam_mask_get_fits(m));
        break;
    default:
	npix = (m->nx)*(m->ny);
        bpm = cpl_calloc(npix,sizeof(*bpm));
        break;
    }
    m->mdata = bpm;
    return(bpm);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_getbpm
    \par Purpose:
        Get the data from a bad pixel mask frame.
    \par Description:
        A bad pixel map frame is read and the data transfered to a byte
        data array. This is needed because CPL doesn't have an unsigned
        char data type. The returned array needs to be deallocated when you
        are finished with it.
    \par Language:
        C
    \param bpmimage
        The input frame with the bad pixel mask
    \return
        The output bad pixel mask data. This needs to be deallocated when
        you are finished with it.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

static unsigned char *vircam_mask_getbpm(vir_fits *bpmimage) {
    long npts,i;
    int *bpmdata;
    cpl_image *b;
    unsigned char *bpm;

    /* Load the bad pixel map data */

    b = vircam_fits_get_image(bpmimage);
    npts = cpl_image_get_size_x(b)*cpl_image_get_size_y(b);
    bpmdata = cpl_image_get_data(b);

    /* Get some space for the bad pixel mask and define it */

    bpm = cpl_malloc(npts*sizeof(*bpm));
    for (i = 0; i < npts; i++)
        bpm[i] = (unsigned char)bpmdata[i];

    /* Tidy and exit */

    return(bpm);
}
	
/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_mask_conf2bpm
    \par Purpose:
        Convert confidence map data to a bad pixel mask
    \par Description:
        A confidence map is given and a bad pixel mask is created. The
        BPM is set at pixels where the input confidence is zero. The output
        BPM needs to be deallocated when you're finished with it
    \par Language:
        C
    \param cpmimage
        The input confidence map image
    \return
        The output bad pixel mask data array. This need to be deallocated
        when you're finished with it
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

static unsigned char *vircam_mask_conf2bpm(vir_fits *cpmimage) {
    long npts,i;
    int *cpmdata;
    cpl_image *c;
    unsigned char *bpm;

    /* Load the confidence map image and get its data */

    c = vircam_fits_get_image(cpmimage);
    npts = cpl_image_get_size_x(c)*cpl_image_get_size_y(c);
    cpmdata = cpl_image_get_data(c);

    /* Get some space for the bad pixel mask and define it where the
       confidence map is zero */

    bpm = cpl_malloc(npts*sizeof(*bpm));
    for (i = 0; i < npts; i++)
        bpm[i] = (cpmdata[i] == 0);

    /* Tidy and exit */

    return(bpm);
}

/**@}*/

/*

$Log: vircam_mask.c,v $
Revision 1.10  2007/10/19 09:25:10  jim
Fixed problems with missing includes

Revision 1.9  2007/07/18 15:34:56  jim
Added vircam_mask_force and vircam_mask_get_filename. Also made changes to
the way that vircam_mask_load and vircam_mask_define deal with corrupt or
missing mask extensions

Revision 1.8  2007/04/04 10:34:55  jim
Modified to use new dfs tags

Revision 1.7  2007/03/23 10:53:22  jim
Fixed little documentation errors

Revision 1.6  2007/03/01 12:42:42  jim
Modified slightly after code checking

Revision 1.5  2006/06/09 11:26:26  jim
Small changes to keep lint happy

Revision 1.4  2006/05/04 10:05:07  jim
Fixed bug where the cpl_frame was being deleted by vircam_mask_clear. This
needs to remain until the whole structure is deleted.

Revision 1.3  2006/05/02 13:25:47  jim
removed include to xmemory

Revision 1.2  2006/03/22 12:13:08  jim
Fixed comments

Revision 1.1  2006/03/22 11:37:58  jim
new file


*/
