/* $Id: vircam_imcore.c,v 1.21 2007/05/02 09:13:57 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/05/02 09:13:57 $
 * $Revision: 1.21 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "vircam_mods.h"
#include "vircam_utils.h"
#include "vircam_fits.h"
#include "catalogue/imcore.h"

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \ingroup reductionmodules
    \brief Generate object catalogues from input images

    \par Name:
        vircam_imcore
    \par Purpose:
        Generate object catalogues from input images
    \par Description:
        A frame and its confidence map are given. Detection thresholds and
	various other parameters are also given. Output is a table with all
	the extracted objects with object classifications included.
    \par Language:
        C
    \param infile
        The input frame with the image to be analysed
    \param conf
        The input frame with the confidence map
    \param ipix
        The minimum allowable size of an object
    \param threshold
        The detection threshold in sigma above sky
    \param icrowd
        If set then the deblending software will be used
    \param rcore
        The core radius in pixels
    \param nbsize
        The smoothing box size for background map estimation
    \param cattyp
        The type of catalogue to be produced
    \param filtfwhm
        The FWHM of the smoothing kernel in the detection algorithm
    \param outtab
        The output table of object
    \param status
        The input/output status that has the same return value a below
    \retval VIR_OK 
        if everything is ok
    \retval VIR_WARN,VIR_FATAL
        errors in the called routines
    \par QC headers:
        The following values will go into the extension propertylist
        - \b SATURATION
	    Saturation level in ADU
	- \b MEAN_SKY
	    Mean sky brightness in ADU
	- \b SKY_NOISE
	    Pixel noise at sky level in ADU
        - \b IMAGE_SIZE
	    The average FWHM of stellar objects in pixels
        - \b ELLIPTICITY
	    The average stellar ellipticity (1 - b/a)
	- \b APERTURE_CORR
	    The stellar aperture correction for 1x core flux
        - \b NOISE_OBJ
	    The number of noise objects
    \par DRS headers:
        The following values will go into the extension propertylist
	- \b THRESHOL
	    The detection threshold in ADU
        - \b MINPIX 
	    The minimum number of pixels per image
	- \b CROWDED
	    Flag for crowded field analysis
	- \b RCORE
	    The core radius for default profile fit in pixels
        - \b FILTFWHM
	    The FWHM of the smoothing kernel in the detection algorithm
        - \b SEEING
	    The average FWHM of stellar objects in pixels
	- \b CLASSIFD
	    A flag to say that the catalgoue has been classified
	- \b SKYLEVEL
	    Mean sky brightness in ADU
        - \b SKYNOISE
	    Pixel noise at sky level in ADU
    \par Other headers:
        The following values will go into the extension propertylist
        - \b APCORxx
	    A series of aperture correction values for each of the core
	    radius apertures.
        - \b SYMBOLx
	    A series of keywords to be used by GAIA for plotting ellipses
    \author
        Jim Lewis, CASU

 */
/*---------------------------------------------------------------------------*/

extern int vircam_imcore(vir_fits *infile, vir_fits *conf, int ipix,
                         float threshold, int icrowd, float rcore, int nbsize,
                         int cattyp, float filtfwhm, vir_tfits **outtab, 
			 int *status) {
    int retval;
    const char *fctid = "vircam_imcore";
    cpl_propertylist *plist;
    vir_fits *in,*c;

    /* Inherited status */

    *outtab = NULL;
    if (*status != VIR_OK)
	return(*status);

    /* Copy the input */

    cattype = cattyp;
    in = vircam_fits_duplicate(infile);
    c = vircam_fits_duplicate(conf);

    /* Call the main processing routine and get the catalogue */

    retval = imcore_conf(in,c,ipix,threshold,icrowd,rcore,nbsize,cattype,
			 filtfwhm,outtab);
    vircam_fits_delete(in);
    vircam_fits_delete(c);
    if (retval != VIR_OK)
	FATAL_ERROR;
    if (cpl_table_get_nrow(vircam_tfits_get_table(*outtab)) == 0) {
	cpl_msg_warning(fctid,"No objects found in %s",
			vircam_fits_get_fullname(infile));
	freetfits(*outtab);
	WARN_RETURN
    }

    /* Get the property list from the input frame */

    plist = vircam_fits_get_phu(infile);
    if (plist == NULL) {
	cpl_msg_error(fctid,"Unable to open propertylist %s",
		      vircam_fits_get_filename(infile));
        FATAL_ERROR
    }

    /* Do the classification */

    retval = classify(*outtab,plist,cattype);
    if (retval != VIR_OK) 
        WARN_RETURN

    GOOD_STATUS
}

/**@}*/

/*

$Log: vircam_imcore.c,v $
Revision 1.21  2007/05/02 09:13:57  jim
Changed api so that outtab is now a vir_tfits structure

Revision 1.20  2007/03/29 12:19:39  jim
Little changes to improve documentation

Revision 1.19  2007/03/01 12:42:41  jim
Modified slightly after code checking

Revision 1.18  2006/11/27 12:06:27  jim
Tarted up some docs

Revision 1.17  2006/08/07 14:20:31  jim
Now duplicates input so that imcore doesn't write over it

Revision 1.16  2006/08/01 11:27:53  jim
Modifications to imcore background estimation and to add ability to
specify the smoothing kernel width

Revision 1.15  2006/07/11 14:52:47  jim
Fixed possible memory leak

Revision 1.14  2006/06/08 14:49:35  jim
If the imcore_conf call generates a table with no rows, then it proceeds
no further

Revision 1.13  2006/06/08 09:03:49  jim
Initialise outtab and extra to be NULL

Revision 1.12  2006/03/23 21:18:49  jim
Minor changes mainly to comment headers

Revision 1.11  2006/03/22 13:58:32  jim
Cosmetic fixes to keep lint happy

Revision 1.10  2006/03/01 10:31:28  jim
Now uses new vir_fits objects

Revision 1.9  2006/02/22 10:08:58  jim
Added status variable to call

Revision 1.8  2006/02/18 11:48:55  jim
*** empty log message ***

Revision 1.7  2006/01/23 10:30:49  jim
Mainly documentation mods

Revision 1.6  2005/12/14 22:17:33  jim
Updated docs

Revision 1.5  2005/11/25 09:56:15  jim
Tidied up some more documentation

Revision 1.4  2005/11/03 15:16:28  jim
Lots of changes mainly to strengthen error reporting

Revision 1.3  2005/11/03 13:28:49  jim
All sorts of changes to tighten up error handling

Revision 1.2  2005/09/22 08:40:41  jim
Fixed some bugs in imcore and added classification Removed some unnecessary
files


*/
