/* $Id: vircam_flatcor.c,v 1.18 2007/10/25 17:34:00 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 17:34:00 $
 * $Revision: 1.18 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "vircam_mods.h"
#include "vircam_utils.h"
#include "vircam_fits.h"

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \ingroup reductionmodules
    \brief Correct input data for flat field response

    \par Name:
        vircam_flatcor
    \par Purpose:
        Correct input data for flat field response
    \par Description:
        Two images are given -- one is the data for an observation, the
        other is for a mean flat frame. The latter is divided into
        the former. The result overwrites the input data array for the object
        image. 
    \par Language:
        C
    \param infile 
        The input data image (overwritten by result).
    \param flatsrc 
        The input flat image
    \param status 
        An input/output status that is the same as the returned values below. 
    \retval VIR_OK 
        if everything is ok
    \retval VIR_WARN 
        if there is a zero divide or if the output property list is NULL.
    \retval VIR_FATAL 
        if image data fails to load
    \par QC headers:
        None
    \par DRS headers:
        The following DRS keywords are written to the infile extension header
        - \b FLATCOR
            The name of the originating FITS file for the flat image
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_flatcor(vir_fits *infile, vir_fits *flatsrc, int *status) {
    cpl_error_code cpl_retval;
    cpl_image *i,*f;
    cpl_propertylist *oplist;
    const char *fctid = "vircam_flatcor";

    /* Inherited status */

    if (*status != VIR_OK)
	return(*status);

    /* Get the images and check the dimensions of each */

    i = vircam_fits_get_image(infile);
    f = vircam_fits_get_image(flatsrc);
    if (vircam_compare_dims(i,f) != VIR_OK) {
	cpl_msg_error(fctid,"Object and flat data array dimensions don't match");
	FATAL_ERROR
    }

    /* Use the cpl image routine to do the arithmetic */
	
    cpl_retval = cpl_image_divide(i,f);
    switch (cpl_retval) {
    case CPL_ERROR_NONE:
	break;
    case CPL_ERROR_DIVISION_BY_ZERO:
	WARN_CONTINUE
	break;
    default:
	FATAL_ERROR
    }

    /* Now put some stuff in the DRS extension... */

    oplist = vircam_fits_get_ehu(infile);
    if (oplist != NULL) {
        cpl_propertylist_update_string(oplist,"ESO DRS FLATCOR",
				       vircam_fits_get_fullname(flatsrc));
        cpl_propertylist_set_comment(oplist,"ESO DRS FLATCOR",
				     "Image used in flat correction");
    } else 
	WARN_CONTINUE

    /* Get out of here */

    GOOD_STATUS
}


/**@}*/

/*

$Log: vircam_flatcor.c,v $
Revision 1.18  2007/10/25 17:34:00  jim
Modified to remove lint warnings

Revision 1.17  2007/03/29 12:19:39  jim
Little changes to improve documentation

Revision 1.16  2007/03/01 12:42:41  jim
Modified slightly after code checking

Revision 1.15  2006/04/20 11:18:23  jim
Now adds an extension name to the error messages rather than an extension number

Revision 1.14  2006/03/23 21:18:47  jim
Minor changes mainly to comment headers

Revision 1.13  2006/03/22 13:36:50  jim
cosmetic changes to keep lint happy

Revision 1.12  2006/03/22 11:39:23  jim
fixed stupid bug

Revision 1.11  2006/03/17 13:55:19  jim
Added comments to DRS keywords

Revision 1.10  2006/03/15 10:43:41  jim
Fixed a few things

Revision 1.9  2006/03/08 14:32:21  jim
Lots of little modifications

Revision 1.8  2006/03/06 13:49:08  jim
Modified so that the DRS keywords are written directly to the extension header
for the input image

Revision 1.7  2006/03/01 10:31:28  jim
Now uses new vir_fits objects

Revision 1.6  2006/01/23 10:30:49  jim
Mainly documentation mods

Revision 1.5  2006/01/03 10:30:04  jim
Given inherited status

Revision 1.4  2006/01/03 10:11:28  jim
Modified to be slightly higher level than originally written. Now write
info to an output property list

Revision 1.3  2005/12/14 22:17:33  jim
Updated docs

Revision 1.2  2005/11/25 09:56:14  jim
Tidied up some more documentation

Revision 1.1  2005/09/13 13:33:58  jim
Forgot to add these


*/
