/* $Id: vircam_dfs.c,v 1.32 2007/11/20 09:37:56 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/11/20 09:37:56 $
 * $Revision: 1.32 $
 * $Name:  $
 */


/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <cpl.h>

#include "vircam_dfs.h"
#include "vircam_utils.h"

#define PACKSZ 1024

/**
   \defgroup vircam_dfs vircam_dfs
   \ingroup supportroutines

   \brief
   These are methods for manipulating keywords in data product headers

   \author
   Jim Lewis, CASU

*/

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_dfs_set_groups
    \par Purpose:
        Set the group of each frame in a frameset as RAW or CALIB according
        to DO.CATG tag given to it.
    \par Description:
        A frameset is given. Each frame in the frameset is classified as
	RAW or CALIB depending upon the dfs tag it's been given.
    \par Language:
        C
    \param set     
        The input frameset
    \retval VIR_OK 
        If everything is ok
    \retval VIR_FATAL 
        If NULL frameset
    \author
        Jim Lewis, CASU

 */
/*---------------------------------------------------------------------------*/

extern int vircam_dfs_set_groups(cpl_frameset *set) {
    cpl_frame *cur_frame;
    const char *tag;
    const char *fctid = "vircam_dfs_set_groups";
    int nframes,i;

    /* Check entries */

    if (set == NULL) 
	return(VIR_FATAL); 
    
    /* Initialize */

    nframes = cpl_frameset_get_size(set);

    /* Loop on frames */

    for (i = 0; i < nframes; i++) {
        cur_frame = cpl_frameset_get_frame(set,i);
        tag = cpl_frame_get_tag(cur_frame);
       
        /* RAW frames */

        if (!strcmp(tag,VIRCAM_CROSSTALK_RAW) ||
	    !strcmp(tag,VIRCAM_DARK_RAW)  ||
	    !strcmp(tag,VIRCAM_DARKCUR_RAW)  ||
	    !strcmp(tag,VIRCAM_NOISE_FLAT_RAW) ||
	    !strcmp(tag,VIRCAM_NOISE_DARK_RAW) ||
            !strcmp(tag,VIRCAM_DOME_RAW) ||
	    !strcmp(tag,VIRCAM_SCI_OBJECT_RAW) || 
	    !strcmp(tag,VIRCAM_SCI_OBJECT_EXT_RAW) || 
	    !strcmp(tag,VIRCAM_TEST_SCIENCE_RAW) || 
	    !strcmp(tag,VIRCAM_LIN_DOME_RAW) ||
	    !strcmp(tag,VIRCAM_LIN_DOME_CHECK) ||
	    !strcmp(tag,VIRCAM_LIN_DARK_RAW) ||
	    !strcmp(tag,VIRCAM_LIN_DARK_CHECK) ||
	    !strcmp(tag,VIRCAM_ILLUM_RAW) ||
	    !strcmp(tag,VIRCAM_PERSIST_OBJ_RAW) ||
	    !strcmp(tag,VIRCAM_PERSIST_DARK_RAW) ||
	    !strcmp(tag,VIRCAM_RESET_RAW) ||
	    !strcmp(tag,VIRCAM_STD_OBJECT_RAW) ||
	    !strcmp(tag,VIRCAM_TWI_RAW))
            cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);

        /* CALIB frames */

        else if (!strcmp(tag,VIRCAM_CAL_RESET) ||
	         !strcmp(tag,VIRCAM_CAL_DARK) ||
	         !strcmp(tag,VIRCAM_CAL_DOME_FLAT) ||
	         !strcmp(tag,VIRCAM_CAL_TWILIGHT_FLAT) ||
	         !strcmp(tag,VIRCAM_CAL_CONF) ||
	         !strcmp(tag,VIRCAM_CAL_BPM) ||
		 !strcmp(tag,VIRCAM_CAL_FRINGE) ||
	         !strcmp(tag,VIRCAM_CAL_CHANTAB) ||
	         !strcmp(tag,VIRCAM_CAL_READGAINFILE) ||
	         !strcmp(tag,VIRCAM_CAL_CHANTAB_INIT) ||
	         !strcmp(tag,VIRCAM_CAL_PHOTTAB) ||
	         !strcmp(tag,VIRCAM_CAL_STDTAB) ||
		 !strcmp(tag,VIRCAM_CAL_OBJCAT) ||
		 !strcmp(tag,VIRCAM_CAL_MSTDTAB) ||
		 !strcmp(tag,VIRCAM_CAL_2MASS) ||
		 !strcmp(tag,VIRCAM_CAL_REFCAT) ||
	         !strcmp(tag,VIRCAM_REF_RESET) ||
	         !strcmp(tag,VIRCAM_REF_DARK) ||
	         !strcmp(tag,VIRCAM_REF_DOME_FLAT) ||
	         !strcmp(tag,VIRCAM_REF_TWILIGHT_FLAT))
            cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_CALIB);
	else 
	    cpl_msg_info(fctid,"No such tag as %s in frame %s",tag,
			 cpl_frame_get_filename(cur_frame));
    }
    return(VIR_OK);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_dfs_set_product_primary_header
    \par Purpose:
        Set the dfs product header keywords in the primary header
    \par Description:
        Set the dfs product header keywords in the primary header
    \par Language:
        C
    \param plist     
        The propertylist to which the keywords should be appended
    \param frame
        The output product frame
    \param frameset
        The input recipe frameset
    \param parlist
        The parameter list of the input recipe
    \param recipeid
        The name of the input recipe
    \param dict
        The dictionary ID
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_dfs_set_product_primary_header(cpl_propertylist *plist,
						  cpl_frame *frame,
						  cpl_frameset *frameset,
						  cpl_parameterlist *parlist,
						  char *recipeid, 
						  const char *dict) {
    const char *fctid="vircam_dfs_set_product_primary_header";
    char package[PACKSZ],*catg;

    /* Get the package name and version */

    (void)snprintf(package,PACKSZ,"%s/%s",PACKAGE,PACKAGE_VERSION);

    /* Add the data-flow keywords */

    if (cpl_dfs_setup_product_header(plist,frame,frameset,parlist,recipeid,
				     package,dict) != CPL_ERROR_NONE) {
	cpl_msg_warning(fctid,
			"Problem with the product primary DFS compliance");
	cpl_error_reset();
    }

    /* Get a copy of the PRO CATG */

    catg = cpl_strdup(cpl_propertylist_get_string(plist,"ESO PRO CATG"));

    /* Remove the stuff that isn't supposed to be here */

    cpl_propertylist_erase_regexp(plist,"ESO PRO",0);

    /* Now add the PRO CATG back in */

    cpl_propertylist_update_string(plist,"ESO PRO CATG",catg);
    cpl_propertylist_set_comment(plist,"ESO PRO CATG",
				 "Category of pipeline product frame");
    cpl_free(catg);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_dfs_set_product_exten_header
    \par Purpose:
        Set the dfs product header keywords in the primary header
    \par Description:
        Set the dfs product header keywords in the primary header
    \par Language:
        C
    \param plist     
        The propertylist to which the keywords should be appended
    \param frame
        The output product frame
    \param frameset
        The input recipe frameset
    \param parlist
        The parameter list of the input recipe
    \param recipeid
        The name of the input recipe
    \param dict
        The dictionary ID
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_dfs_set_product_exten_header(cpl_propertylist *plist,
						cpl_frame *frame,
						cpl_frameset *frameset,
						cpl_parameterlist *parlist,
						char *recipeid, 
						const char *dict) {
    const char *fctid="vircam_dfs_set_product_exten_header";
    char package[PACKSZ];

    /* Get the package name and version */

    (void)snprintf(package,PACKSZ,"%s/%s",PACKAGE,PACKAGE_VERSION);

    /* Add the data-flow keywords */

    if (cpl_dfs_setup_product_header(plist,frame,frameset,parlist,recipeid,
				     package,dict) != CPL_ERROR_NONE) {
	cpl_msg_warning(fctid,"Problem with the product extn DFS compliance");
	cpl_error_reset();
    }

    /* Remove the stuff that isn't supposed to be here */

    cpl_propertylist_erase(plist,"INSTRUME");
    cpl_propertylist_erase(plist,"PIPEFILE");
    cpl_propertylist_erase_regexp(plist,"ESO INS",0);
    cpl_propertylist_erase_regexp(plist,"ESO OBS",0);
    cpl_propertylist_erase_regexp(plist,"ESO TEL",0);

}

/**@}*/

/*

$Log: vircam_dfs.c,v $
Revision 1.32  2007/11/20 09:37:56  jim
Added VIRCAM_LIN_DOME_CHECK and VIRCAM_LIN_DARK_CHECK

Revision 1.31  2007/10/19 06:55:06  jim
Modifications made to use new method for directing the recipes to the
standard catalogues using the sof

Revision 1.30  2007/08/23 09:00:59  jim
Removed annoying error messages in the product header routines which used
to tell you if it couldn't find stuff to erase.

Revision 1.29  2007/04/23 12:50:22  jim
Added VIRCAM_TEST_SCIENCE_RAW

Revision 1.28  2007/04/04 10:34:42  jim
Complete rewrite of the dfs tags

Revision 1.27  2007/03/23 10:53:22  jim
Fixed little documentation errors

Revision 1.26  2007/03/01 12:42:41  jim
Modified slightly after code checking

Revision 1.25  2007/02/15 11:53:45  jim
Added type CHANNEL_TABLE_INIT

Revision 1.24  2007/02/05 13:54:00  jim
Added REFERENCE tags

Revision 1.23  2007/01/17 23:53:43  jim
Modified to remove INS, OBS and TEL structures from extension headers

Revision 1.22  2006/12/06 12:58:57  jim
Added MASTER_FRINGE

Revision 1.21  2006/11/28 20:57:05  jim
Added ILLUM types

Revision 1.20  2006/10/05 09:22:59  jim
Small modifications to a couple of cpl calls to bring them into line with
cpl v3.0

Revision 1.19  2006/07/04 09:19:04  jim
replaced all sprintf statements with snprintf

Revision 1.18  2006/06/14 14:13:58  jim
fixed minor doc problem

Revision 1.17  2006/06/13 21:26:02  jim
Added VIRCAM_CALIB_READNOISE_FILE

Revision 1.16  2006/05/24 13:33:56  jim
Added VIRCAM_PHOTTAB

Revision 1.15  2006/05/18 12:32:23  jim
Added channel table and object catalogue to calib list

Revision 1.14  2006/04/27 09:46:27  jim
Modified to conform to new dictionary

Revision 1.13  2006/04/26 09:39:35  jim
Fixes the pipeline id to the header

Revision 1.12  2006/04/25 13:47:59  jim
Modified calls to vircam_dfs_set_product_*_header routines

Revision 1.11  2006/04/20 11:19:22  jim
*** empty log message ***

Revision 1.10  2006/03/22 13:14:24  jim
Sorted data categories

Revision 1.9  2006/02/27 13:53:32  jim
Added new categories to vircam_dfs_set_groups

Revision 1.8  2006/02/18 11:45:01  jim
Added vircam_dfs_set_product_primary_header and
vircam_dfs_set_product_exten_header

Revision 1.7  2006/01/23 10:30:49  jim
Mainly documentation mods

Revision 1.6  2005/12/14 22:17:33  jim
Updated docs

Revision 1.5  2005/11/29 14:56:36  jim
Added a few new categories

Revision 1.4  2005/11/25 09:56:14  jim
Tidied up some more documentation

Revision 1.3  2005/11/03 13:28:48  jim
All sorts of changes to tighten up error handling

Revision 1.2  2005/09/20 15:07:46  jim
Fixed a few bugs and added a few things

Revision 1.1.1.1  2005/08/05 08:29:09  jim
Initial import


*/
