/*

$Id: imcore_conf.c,v 1.17 2007/05/03 11:15:34 jim Exp $

*/


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <cpl.h>
#include "../vircam_fits.h"
#include "../vircam_pfits.h"
#include "../vircam_wcsutils.h"

#include "ap.h"
#include "util.h"
#include "imcore.h"
#include "floatmath.h"

#define FATAL_ERR(_a) {freetable(tab); cpl_msg_error(fctid,_a); tidy(); return(VIR_FATAL);}

#define NW 5

static float *smoothed = NULL;
static float *smoothedc = NULL;
static unsigned char *mflag = NULL;
static float *indata = NULL;
static int *confdata = NULL;
static ap_t ap;
static int freeconf = 0;

static float weights[NW*NW];
static float weightc[NW*NW];
static long nx;
static long ny;

static void crweights(float);
static void convolve(int);
static void tidy(void);

extern int imcore_conf(vir_fits *infile, vir_fits *conf, int ipix, 
		       float threshold, int icrowd, float rcore, int nbsize, 
		       int cattyp, float filtfwhm, vir_tfits **outcat) {

    int i,retval,mulpix,j,*currentc,nw2,status;
    float fconst,nullval,skymed,skysig,thresh,xintmin,offset;
    float *current,isatbc,isat,junk;
    long npix,nxc,nyc,npts;
    cpl_image *map,*cmap;
    cpl_propertylist *plist,*extra;
    const char *fctid = "imcore_conf";

    /* Initialise output */

    *outcat = NULL;

    /* Useful constants */

    fconst = 1.0/M_LN2;
    nullval = 0.0;
    cattype = cattyp;
    nobjects = 0;

    /* Open input image */

    map = vircam_fits_get_image(infile);
    if ((indata = cpl_image_get_data_float(map)) == NULL) 
	FATAL_ERR("Error getting image data");
    nx = (long)cpl_image_get_size_x(map);
    ny = (long)cpl_image_get_size_y(map);
    npts = nx*ny;
    if (vircam_pfits_get_gain(vircam_fits_get_ehu(infile),&gain) != VIR_OK)
	gain = 5.0;

    /* Open the associated confidence map, if it exists */

    if (conf != NULL) {
	cmap = vircam_fits_get_image(conf);
	if ((confdata = cpl_image_get_data(cmap)) == NULL)
	    FATAL_ERR("Error getting confidence map data");
        nxc = (long)cpl_image_get_size_x(cmap);
        nyc = (long)cpl_image_get_size_y(cmap);
        if ((nx != nxc) || (ny != nyc))
	    FATAL_ERR("Input image and confidence dimensions don't match");
	freeconf = 0;
    } else {
	confdata = cpl_malloc(npts*sizeof(*confdata));
	for (i = 0; i < npts; i++) 
	    confdata[i] = 100;
	freeconf = 1;
	cmap = NULL;
    }
    
    /* Get mflag array for flagging saturated pixels */

    npix = nx*ny;
    mflag = cpl_calloc(npix,sizeof(*mflag));

    /* Open the ap structure and define some stuff in it */

    ap.lsiz = nx;
    ap.csiz = ny;
    ap.inframe = map;
    ap.conframe = cmap;
    ap.xtnum = vircam_fits_get_nexten(infile);
    apinit(&ap);
    ap.indata = indata;
    ap.confdata = confdata;
    ap.multiply = 1;
    ap.ipnop = ipix;
    ap.mflag = mflag;
    ap.rcore = rcore;
    ap.filtfwhm = filtfwhm;
    ap.icrowd = icrowd;
    ap.fconst = fconst;

    /* Open the output catalogue FITS table */

    tabinit(&ap);

    /* Compute a saturation level before background correction. */

    retval = imcore_backstats(&ap,nullval,1,&skymed,&skysig,&isatbc);
    if (retval != VIR_OK) 
	FATAL_ERR("Error calculating saturation level");

    /* Flag up regions where the value is above the saturation level
       determined above. */

    for (i = 0; i < npix ; i++)
        if (indata[i] > isatbc)
            mflag[i] = MF_SATURATED;
        else if (indata[i] < STUPID_VALUE)
            mflag[i] = MF_STUPID_VALUE;

    /* Compute the background variation and remove it from the data*/

    retval = imcore_background(&ap,nbsize,nullval);
    if (retval != VIR_OK) 
	FATAL_ERR("Error calculating background");

    /* Compute a saturation level after background correction. */

    retval = imcore_backstats(&ap,nullval,1,&skymed,&skysig,&isat);
    if (retval != VIR_OK) 
	FATAL_ERR("Error calculating saturation");


    /* Compute background statistics */

    retval = imcore_backstats(&ap,nullval,0,&skymed,&skysig,&junk);
    if (retval != VIR_OK) 
	FATAL_ERR("Error calculating background stats");

    /* Get the propertly list for the input file and add some info*/

    plist = vircam_fits_get_ehu(infile);
    cpl_propertylist_update_float(plist,"ESO DRS SKYLEVEL",skymed);
    cpl_propertylist_set_comment(plist,"ESO DRS SKYLEVEL",
				 "[adu] Median sky brightness");
    cpl_propertylist_update_float(plist,"ESO DRS SKYNOISE",skysig);
    cpl_propertylist_set_comment(plist,"ESO DRS SKYNOISE",
				 "[adu] Pixel noise at sky level");

    /* Take mean sky level out of data */

    for (i = 0; i < nx*ny; i++)
	indata[i] -= skymed;
    
    /* Work out isophotal detection threshold levels */

    thresh = threshold*skysig;
    
    /* Minimum intensity for consideration */

    xintmin = 1.5*thresh*((float)ipix);

    /* Minimum size for considering multiple images */

    mulpix = MAX(8,2*ipix);

    /* Actual areal profile levels: T, 2T, 4T, 8T,...but written wrt T
       i.e. threshold as a power of 2 */

    offset = logf(thresh)*fconst;

    /* Get a bit of workspace for buffers */

    smoothed = cpl_malloc(nx*sizeof(*smoothed));
    smoothedc = cpl_malloc(nx*sizeof(*smoothedc));

    /* Define a few things more things in ap structure */

    ap.mulpix = mulpix;
    ap.areal_offset = offset;
    ap.thresh = thresh;
    ap.xintmin = xintmin;
    ap.sigma = skysig;
    ap.background = skymed;
    ap.saturation = (float)isat;

    /* Now update mflag array */

    for (i = 0; i < npix; i++) {
        if ((indata[i]*confdata[i]*0.01 > 3.0*skysig) && 
	    (mflag[i] != MF_SATURATED)) 
	    mflag[i] = MF_UNSATPIX;
    }

    /* Set the weights */

    crweights(filtfwhm);
    nw2 = NW/2;

    /* Right, now for the extraction loop.  Begin by defining a group of
       three rows of data and confidence */

    for (j = nw2; j < ny-nw2; j++) {
        current = indata + j*nx;
        currentc = confdata + j*nx;
	convolve(j);
   
        /* Do the detection now */

        apline(&ap,current,currentc,smoothed,smoothedc,j,NULL);

        /* Make sure we're not overruning the stacks */

        if (ap.ibstack > (ap.maxbl - ap.lsiz))
	    apfu(&ap);
	if (ap.ipstack > (ap.maxpa*3/4))
            for (i = 0; i < ap.maxpa*3/8; i++)
		apfu(&ap);

	/* See if there are any images to terminate */

	if (ap.ipstack > 1)
	    terminate(&ap);
    }

    /* Post process. First truncate the cpl_table to the correct size and then
       work out an estimate of the seeing */

    cpl_table_set_size(tab,nobjects);
    retval = do_seeing(&ap);
    if (retval != VIR_OK)
	FATAL_ERR("Error working out seeing");
    tabclose(&ap);

    /* Create a property list with extra parameters. First QC parameters */

    extra = cpl_propertylist_duplicate(vircam_fits_get_ehu(infile));
    cpl_propertylist_update_float(extra,"ESO QC SATURATION",ap.saturation);
    cpl_propertylist_set_comment(extra,"ESO QC SATURATION",
				 "[adu] Saturation level");
    cpl_propertylist_update_float(extra,"ESO QC MEAN_SKY",ap.background);
    cpl_propertylist_set_comment(extra,"ESO QC MEAN_SKY",
				 "[adu] Median sky brightness");
    cpl_propertylist_update_float(extra,"ESO QC SKY_NOISE",ap.sigma);
    cpl_propertylist_set_comment(extra,"ESO QC SKY_NOISE",
				 "[adu] Pixel noise at sky level");

    /* Now DRS parameters */

    cpl_propertylist_update_float(extra,"ESO DRS THRESHOL",ap.thresh);
    cpl_propertylist_set_comment(extra,"ESO DRS THRESHOL",
				 "[adu] Isophotal analysis threshold");
    cpl_propertylist_update_int(extra,"ESO DRS MINPIX",ap.ipnop);
    cpl_propertylist_set_comment(extra,"ESO DRS MINPIX",
				 "[pixels] Minimum size for images");
    cpl_propertylist_update_int(extra,"ESO DRS CROWDED",ap.icrowd);
    cpl_propertylist_set_comment(extra,"ESO DRS CROWDED",
				 "Crowded field analysis flag");
    cpl_propertylist_update_float(extra,"ESO DRS RCORE",ap.rcore);
    cpl_propertylist_set_comment(extra,"ESO DRS RCORE",
				 "[pixels] Core radius for default profile fit");
    cpl_propertylist_update_float(extra,"ESO DRS SEEING",ap.fwhm);
    cpl_propertylist_set_comment(extra,"ESO DRS SEEING",
				 "[pixels] Average FWHM");
    cpl_propertylist_update_float(extra,"ESO DRS FILTFWHM",ap.filtfwhm);
    cpl_propertylist_set_comment(extra,"ESO DRS FILTFWHM",
				 "[pixels] FWHM of smoothing kernel");
    cpl_propertylist_update_int(extra,"ESO DRS XCOL",imcore_xcol);
    cpl_propertylist_set_comment(extra,"ESO DRS XCOL","Column for X position");
    cpl_propertylist_update_int(extra,"ESO DRS YCOL",imcore_ycol);
    cpl_propertylist_set_comment(extra,"ESO DRS YCOL","Column for Y position");
    
    /* Now wrap all this stuff up and send it back */

    plist = cpl_propertylist_duplicate(vircam_fits_get_phu(infile));
    status = VIR_OK;
    (void)vircam_tabwcs(extra,imcore_xcol,imcore_ycol,&status);
    *outcat = vircam_tfits_wrap(tab,NULL,plist,extra);

    /* Tidy and exit */  

    tidy();
    return(VIR_OK);
}

static void crweights(float filtfwhm) {
    int i,j,nw2,n;
    double gsigsq,di,dj;
    float renorm;

    /* Get the kernel size */

    nw2 = NW/2;
    
    /* Set the normalisation constants */

    gsigsq = 1.0/(2.0*pow(MAX(1.0,(double)filtfwhm)/2.35,2.0));
    renorm = 0.0;

    /* Now work out the weights */

    n = -1;
    for (i = -nw2; i <= nw2; i++) {
	di = (double)i;
	di *= gsigsq*di;
	for (j = -nw2; j <= nw2; j++) {
	    dj = (double)j;
	    dj *= gsigsq*dj;
	    n++;
	    weights[n] = (float)exp(-(di + dj));
	    renorm += weights[n];
	}
    }

    /* Now normalise the weights */

    n = -1;
    for (i = -nw2; i <= nw2; i++) {
	for (j = -nw2; j <= nw2; j++) {
	    n++;
	    weights[n] /= renorm;
	    weightc[n] = 0.01*weights[n];
	}
    }
}

static void convolve(int ir) {
    int i,nw2,ix,jx,jy,n;
    float *idata;
    int *cdata;
    

    /* Zero the summations */

    for (i = 0; i < nx; i++) {
	smoothed[i] = 0.0;
	smoothedc[i] = 0.0;
    }

    /* Now big is the smoothing kernel? */

    nw2 = NW/2;

    /* Now loop for each column */

    for (ix = nw2; ix < nx-nw2; ix++) {
	n = -1;
	for (jy = ir-nw2; jy <= ir+nw2; jy++) {
	    idata = indata + jy*nx;
	    cdata = confdata + jy*nx;
	    for (jx = ix-nw2; jx <= ix+nw2; jx++) {
		n++;
		smoothed[ix] += weights[n]*idata[jx];
		smoothedc[ix] += weightc[n]*idata[jx]*cdata[jx];
	    }
	}
    }
}

static void tidy(void) {

    if (freeconf) 
	freespace(confdata);
    freespace(smoothed);
    freespace(smoothedc);
    freespace(mflag);
    apclose(&ap);
}

/* 

$Log: imcore_conf.c,v $
Revision 1.17  2007/05/03 11:15:34  jim
Fixed little problem with table wcs

Revision 1.16  2007/05/02 09:11:35  jim
Modified to allow for inclusion of table WCS keywords into FITS header

Revision 1.15  2007/03/01 12:38:26  jim
Small modifications after a bit of code checking

Revision 1.14  2006/11/27 11:56:59  jim
Changed cpl_propertylist_append to cpl_propertylist_update calls

Revision 1.13  2006/08/01 11:27:54  jim
Modifications to imcore background estimation and to add ability to
specify the smoothing kernel width

Revision 1.12  2006/07/11 14:51:02  jim
Fixed small bug in the range of the main loop

Revision 1.11  2006/07/03 09:33:19  jim
Fixed a few things to keep the compiler happy

Revision 1.10  2006/06/30 21:31:09  jim
MOdifications to background routines and smoothing kernel

Revision 1.9  2006/06/13 14:06:10  jim
Gets gain estimate from header

Revision 1.8  2006/06/06 13:04:22  jim
Fixed apline so that it now takes confidence map info correctly

Revision 1.7  2006/05/30 12:13:53  jim
Initialised nobjects

Revision 1.6  2006/05/18 12:35:01  jim
Fixed bug for header writing

Revision 1.5  2006/03/15 10:43:42  jim
Fixed a few things

Revision 1.4  2006/03/01 10:31:29  jim
Now uses new vir_fits objects

Revision 1.3  2005/09/22 08:40:42  jim
Fixed some bugs in imcore and added classification Removed some unnecessary
files

Revision 1.2  2005/09/20 15:07:47  jim
Fixed a few bugs and added a few things

Revision 1.1  2005/09/13 13:25:29  jim
Initial entry after modifications to make cpl compliant


*/
