/*

$Id: imcore_background.c,v 1.8 2007/12/19 13:16:50 jim Exp $

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <cpl.h>

#include "floatmath.h"
#include "util.h"
#include "imcore.h"

static int **hist = NULL;
static int *nnp = NULL;
static int npvx;
static int npvy;
static void tidy(void);
static void sortit (float [], int);

extern int imcore_background(ap_t *ap, int nbsize, float nullval) {
    float fracx,fracy,skymed,sigma,skymedc,sigmac,avsky,fnbsize,dely,delx;
    float t1,t2,dsky,*map,**bvals,*work;
    int ifracx,ifracy,nbsizx,nbsizy,nbx,nby,npixstripe,l,i,ll;
    int isquare,ilev,j,iclip,mcpix,iloop,irej,nbsizo2,kk,k,iby,ibyp1,ibx,ibxp1;
    int *shist,*conf;
    unsigned char *mflag;
    long nx,ny;

    /* Set up some variables */

    map = ap->indata;
    conf = ap->confdata;
    mflag = ap->mflag;
    nx = ap->lsiz;
    ny = ap->csiz;

    /* check to see if nbsize is close to exact divisor */

    fracx = ((float)nx)/((float)nbsize);
    fracy = ((float)ny)/((float)nbsize);
    ifracx = (int)(fracx + 0.1);
    ifracy = (int)(fracy + 0.1);
    nbsizx = nx/ifracx;
    nbsizy = ny/ifracy;
    nbsize = MAX(NINT(0.9*nbsize), MIN(nbsize, MIN(nbsizx,nbsizy)));
    nbsize = MIN(nx,MIN(ny,nbsize)); /* trap for small maps */

    /* Divide the map into partitions */

    nbx = nx/nbsize;
    nby = ny/nbsize;
    npixstripe = nbsize*nx;
    npvx = nbx;
    npvy = nby;

    /* Get histogram workspace if you can */

    hist = cpl_malloc(nbx*sizeof(int *));
    for (l = 0; l < nbx; l++) 
	hist[l] = cpl_malloc(MAXHIST*sizeof(int));

    /* Same for background values array */

    bvals = cpl_malloc(nby*sizeof(float *));
    for (l = 0; l < nby; l++) 
	bvals[l] = cpl_malloc(nbx*sizeof(float));

    /* Store some of this away for use later */

    ap->backmap.nbx = nbx;
    ap->backmap.nby = nby;
    ap->backmap.nbsize = nbsize;
    ap->backmap.bvals = bvals;

    /* Finally a counter array */

    nnp = cpl_malloc(nbx*sizeof(int));

    /* Loop for each row of background squares. Start by initialising
       the accumulators and histograms */

    for (l = 0; l < nby; l++) {
	memset((char *)nnp,0,nbx*sizeof(*nnp));
        for (i = 0; i < nbx; i++) 
   	    memset((char *)hist[i],0,MAXHIST*sizeof(int));

        /* Skim through the data in this stripe. Find out which square each
	   belongs to and add it it to the relevant histogram */

	ll = l*npixstripe;
        for (i = 0; i < npixstripe; i++) {
	    if (map[ll+i] != nullval && conf[ll+i] > 0 &&
		mflag[ll+i] != MF_SATURATED) {
		isquare = (int)((float)(i % nx)/(float)nbsize);
		isquare = MIN(nbx-1,MAX(0,isquare));
		ilev = MIN(MAXHISTVAL,MAX(MINHISTVAL,NINT(map[i+ll])));
		hist[isquare][ilev-MINHISTVAL] += 1;
		nnp[isquare] += 1;
	    }
	}
  
        /* but only do background estimation if enough pixels ----------- */

	for (j = 0; j < nbx; j++) {
	    if (nnp[j] > 0.25*nbsize*nbsize){
		shist = hist[j];
		imcore_medsig(shist,MAXHIST,MINHISTVAL-1,nnp[j],&skymed,&sigma);

                /* do an iterative 3-sigma upper clip to give a more robust
		   estimator */

		iclip = MAXHISTVAL;
		mcpix = nnp[j];
		skymedc = skymed;
		sigmac = sigma;
		for(iloop = 0; iloop < 3; iloop++) {
		    irej = 0;
 	            for(i = NINT(skymedc+3.0*sigmac); i <= iclip; i++)
                        irej += shist[i-MINHISTVAL];
		    if (irej == 0)
			break;
		    iclip = NINT(skymedc+3.0*sigmac) - 1;
		    mcpix = mcpix - irej;
		    imcore_medsig(shist,MAXHIST,MINHISTVAL-1,mcpix,&skymedc,
				  &sigmac);
		}
		bvals[l][j] = skymedc;
	    } else {
		bvals[l][j] = -1000.0;
	    }
	}
    }

    /* filter raw background values */

    bfilt(bvals,nbx,nby);

    /* compute average sky level */

    work = cpl_malloc(nbx*nby*sizeof(*work));
    k = 0;
    for(l = 0; l < nby; l++)
        for(j = 0; j < nbx; j++) 
	    work[k++] = bvals[l][j];
    sortit(work,nbx*nby);
    avsky = work[(nbx*nby)/2];
    freespace(work);

    /* ok now correct map for background variations and put avsky back on */

    nbsizo2 = nbsize/2;
    fnbsize = 1.0/((float)nbsize);
    for (k = 0; k < ny; k++) {
	kk = k*nx;

	/* Nearest background pixel vertically */

        iby = (k + 1 + nbsizo2)/nbsize;
	ibyp1 = iby + 1;
	iby = MIN(nby,MAX(1,iby));
	ibyp1 = MIN(nby,ibyp1);
	dely = (k + 1 - nbsize*iby + nbsizo2)*fnbsize;

	for (j = 0; j < nx; j++) {
	    if (map[kk+j] == nullval) 
		continue;

            /* nearest background pixel across */

	    ibx = (j + 1 + nbsizo2)/nbsize;
	    ibxp1 = ibx + 1;
	    ibx = MIN(nbx,MAX(1,ibx));
	    ibxp1 = MIN(nbx,ibxp1);
	    delx = (j + 1 - nbsize*ibx + nbsizo2)*fnbsize;

            /* bilinear interpolation to find background */
	    
	    t1 = (1.0 - dely)*bvals[iby-1][ibx-1] + dely*bvals[ibyp1-1][ibx-1];
	    t2 = (1.0 - dely)*bvals[iby-1][ibxp1-1] + dely*bvals[ibyp1-1][ibxp1-1];
	    dsky = avsky - (1.0 - delx)*t1 - delx*t2;
            map[kk+j] += dsky;
	}
    }

    /* Free some workspace */

    tidy();
    return(VIR_OK);
}

extern int imcore_backstats(ap_t *ap, float nullval, int satonly, 
			    float *skymed, float *skysig, float *sat) {
    int ilev,iclip,iloop,i,*ihist,isat,iter,*conf;
    long mpix,npix,k,mcpix,irej,lpix,nx,ny;
    float sigsq,skymedc,sigmac,*map,sata,fac;

    /* Get some info from the ap structure */

    map = ap->indata;
    conf = ap->confdata;
    nx = ap->lsiz;
    ny = ap->csiz;

    /* Check to make sure there are some non-zero values here */

    ilev = 1;
    for (i = 0; i < nx*ny; i++) {
	if (map[i] != 0.0 && conf[i] > 0) {
	    ilev = 0;
	    break;
	}
    }
    if (ilev == 1) {
	*skymed = 0.0;
	*skysig = 0.0;
	*sat = 0.0;
	return(VIR_WARN);
    }

    /* First, get some workspace for the background histogram */

    ihist = cpl_calloc(MAXHIST,sizeof(*ihist));

    /* Loop for up to 10 iterations. For each iteration we multiply the 
       input data by a successively higher power of 1 in order to 
       try and deal with data that has very small noise estimates */

    fac = 0.5;
    for (iter = 0; iter <=9; iter++) {
	fac *= 2.0;
	for (k = 0; k < MAXHIST; k++)
	    ihist[k] = 0;

	/* Now form the histogram of all pixel intensities */

	mpix = 0;
	isat = 0;
	npix = nx*ny;
	for (k = 0; k < npix; k++) {
	    if (map[k] != nullval && conf[k] > 0) {
		ilev = MIN(MAXHISTVAL,MAX(MINHISTVAL,NINT(fac*map[k])));
		ihist[ilev - MINHISTVAL] += 1;
		isat = MAX(isat,ilev);
		mpix++;
	    }
	}
	sata = MIN(MAXHISTVAL,MAX(MINSATURATE,0.9*((float)isat))/fac);
	lpix = ihist[isat - MINHISTVAL];
	while (lpix < mpix/1000 && isat > MINHISTVAL) {
	    isat--;
	    lpix += ihist[isat - MINHISTVAL];
	}
	*sat = ((float)isat)/fac;
	*sat = MIN(MAXHISTVAL,MAX(MINSATURATE,MAX(0.95*(*sat),sata)));

	/* If all you want is the saturation level, then get out of here...*/

	if (satonly) {
	    freespace(ihist);
	    return(VIR_OK);
	}

	/* Now find the median and sigma */

	imcore_medsig(ihist,MAXHIST,MINHISTVAL-1,mpix,skymed,skysig);
	sigsq = (*skysig)*(*skysig);

	/* Do an iterative 3-sigma upper clip to give a more robust 
	   estimator */

	iclip = MAXHISTVAL;
	mcpix = mpix;
	skymedc = *skymed;
	sigmac = *skysig;
	for (iloop = 0; iloop < 3; iloop++) {
	    irej = 0;
	    for (i = NINT(skymedc+3.0*sigmac); i <= iclip; i++)
		irej += ihist[i - MINHISTVAL];
	    if (irej == 0)
		break;
	    iclip = NINT(skymedc+3.0*sigmac)-1;
	    mcpix = mcpix-irej;
	    imcore_medsig(ihist,MAXHIST,MINHISTVAL-1,mcpix,&skymedc,&sigmac);
	}
	if (sigmac > 2.5)
	    break;
    }

    /* Set the final answer */

    *skymed = skymedc/fac;
    *skysig = sigmac/fac;
    freespace(ihist);
    return(VIR_OK);
}   

extern void imcore_backest(ap_t *ap, float x, float y, float *skylev, 
			   float *skyrms) {
    int i,j,nbx,nby,nbsize,nbsizo2,iby,ibyp1,ibx,ibxp1;
    float **bvals,fnbsize,dely,delx,t1,t2;

    /* Define some local variables */

    nbx = ap->backmap.nbx;
    nby = ap->backmap.nby;
    nbsize = ap->backmap.nbsize;
    bvals = ap->backmap.bvals;

    /* Get closest pixel to the input location */

    i = NINT(x);
    j = NINT(y);

    /* Now, work out where in the map to do the interpolation */

    nbsizo2 = nbsize/2;
    fnbsize = 1.0/((float)nbsize);
    iby = (j + nbsizo2)/nbsize;
    ibyp1 = iby + 1;
    iby = MIN(nby,MAX(1,iby));
    ibyp1 = MIN(nby,ibyp1);
    dely = (j  - nbsize*iby + nbsizo2)*fnbsize;
    ibx = (i + nbsizo2)/nbsize;
    ibxp1 = ibx + 1;
    ibx = MIN(nbx,MAX(1,ibx));
    ibxp1 = MIN(nbx,ibxp1);
    delx = (i - nbsize*ibx + nbsizo2)*fnbsize;

    /* Now do a linear interpolation to find the background. Calculate MAD of
       the four adjacent background cells as an estimate of the RMS */

    t1 = (1.0 - dely)*bvals[iby-1][ibx-1] + dely*bvals[ibyp1-1][ibx-1];
    t2 = (1.0 - dely)*bvals[iby-1][ibxp1-1] + dely*bvals[ibyp1-1][ibxp1-1];
    *skylev = (1.0 - delx)*t1 + delx*t2;
    *skyrms = 0.25*(fabsf(bvals[iby-1][ibx-1] - *skylev) +
		    fabsf(bvals[ibyp1-1][ibx-1] - *skylev) +
		    fabsf(bvals[iby-1][ibxp1-1] - *skylev) +
		    fabsf(bvals[ibyp1-1][ibxp1-1] - *skylev));
}

extern void imcore_medsig(int *shist, int nh, int ist, int itarg, float *med, 
			  float *sig) {
    int isum, medata;
    float ffrac,sigmed;
 
    /* median */ 

    isum = 0;
    medata = ist;
    while (isum <= (itarg+1)/2 && (medata-MINHISTVAL) < nh) {
	medata++;
	isum += shist[medata-MINHISTVAL];
    }
    if (shist[medata-MINHISTVAL] == 0) {
	ffrac = 0.0;
    } else {
	ffrac = (float)(isum - (itarg+1)/2)/(float)shist[medata-MINHISTVAL];
    }
    *med = (float)medata - ffrac + 0.5;

    /* sigma */

    isum = 0;
    medata = ist;
    while (isum <= (itarg+3)/4 && (medata-MINHISTVAL) < nh) {
	medata++;
	isum += shist[medata-MINHISTVAL];
    }
    if (shist[medata-MINHISTVAL] == 0) {
	ffrac = 0.0;
    } else {
	ffrac = (float)(isum - (itarg+3)/4)/(float)shist[medata-MINHISTVAL];
    }
    sigmed = (float)medata - ffrac + 0.5;
    *sig = 1.48*(*med - sigmed);
    *sig = MAX(0.5,*sig);
}

static void sortit (float ia[], int n) {
    int i, j, ii, jj, ifin;
    float it;
 
    jj = 4;
    while (jj < n) 
	jj = 2 * jj;
    jj = MIN(n,(3 * jj)/4 - 1);
    while (jj > 1) {
        jj = jj/2;
        ifin = n - jj;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + jj;
            if (ia[i] <= ia[j]) 
		continue;
            it = ia[j];
            do {
                ia[j] = ia[i];
                j = i;
                i = i - jj;
                if (i < 0) 
		    break;
            } while (ia[i] > it);
            ia[j] = it;
        }
    }
    return;
}


static void tidy(void) {
    int i;
 
    freespace(nnp);
    if (hist != NULL) {
        for (i = 0; i < npvx; i++) 
            freespace(hist[i]);
    }
    freespace(hist);
    return;
}

/*

$Log: imcore_background.c,v $
Revision 1.8  2007/12/19 13:16:50  jim
Modified the interation step size

Revision 1.7  2007/04/23 12:51:27  jim
Traps for 0 in confidence map

Revision 1.6  2007/03/01 12:38:26  jim
Small modifications after a bit of code checking

Revision 1.5  2006/08/01 11:27:54  jim
Modifications to imcore background estimation and to add ability to
specify the smoothing kernel width

Revision 1.4  2006/07/11 14:50:27  jim
Modified to do arithmetic in double precision for stats

Revision 1.3  2006/06/30 21:31:09  jim
MOdifications to background routines and smoothing kernel

Revision 1.2  2006/04/20 11:15:19  jim
A few minor bugs fixed

Revision 1.1  2005/09/13 13:25:28  jim
Initial entry after modifications to make cpl compliant


*/

