/*

$Id: classify.c,v 1.9 2007/05/02 09:11:34 jim Exp $

*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <strings.h>

#include <cpl.h>
#include <vircam_utils.h>
#include <vircam_pfits.h>
#include "classify.h"

#define MAXHIST   111
#define STEP      0.05
#define NSAMPLE   150
#define MAXLOOP   5
#define BLIMDEF   15.0;
#define FLIMDEF   11.0;
#define CMINDEF   7.5
#define CMAXDEF   15.0
#define NAREAL    8
#define PI        3.14159265358979323846

#define COREMAG(A,B,C) 2.5*log10((double)(max(C,A-B)))

/* Make the data arrays and header values global */

static long nrows;
static float thresh,skylevel,skynoise,rcore,exptime;

/* Derived values */

static int poor;
static float sigell,fitell,elllim,sigellf,fitellf;
static float blim,flim,cmin,cmax;
static float fit1,fit2,fit3,fit4,fit5;
static float fit_final,sigma_final;
static float *lower1,*lower2,*lower3,*upper1,*upper2,*upper3,*uppere;
static float avsig1,avsig2,avsig3,wt1,wt2,wt3;

/* Classification values */

static int nstar,ngal,njunk,ncmp;

/* Values for the data quality and aperture corrections */

static float avsat,corlim,cormin,apcpkht,apcor,apcor1,apcor2,apcor3,apcor4;
static float apcor5,apcor6,apcor7;

/* Data arrays */

static float *workspace = NULL;
static cpl_table *catcopy = NULL;

static float *core_flux,*core1_flux,*core2_flux,*core3_flux,*core4_flux;
static float *core5_flux,*peak_height,*peak_mag,*ellipticity,*iso_flux;
static float *total_flux,*cls,*sig;


/* Column definitions */

#define NCOL 10
static const char *cols32[NCOL] = {"Core_flux","Core1_flux","Core2_flux",
				   "Core3_flux","Core4_flux","Peak_height",
				   "Ellipticity","Isophotal_flux","Total_flux",
				 "Core5_flux"};

static const char *cols80[NCOL] = {"Aper_flux_3","Aper_flux_1","Aper_flux_4",
				   "Aper_flux_5","Aper_flux_6","Peak_height",
				   "Ellipticity","Isophotal_flux","Hall_flux",
				   "Aper_flux_7"};

static int ncols;


/* Subroutine prototypes */

static void anhist(float *, int, float *, float *);
static void boundries(float *, float *, float *, float, float, float, float,
                      int, float, float, float *, float *, float *, float *);
static void boundpk(float *, float *, float, float, float *, float *,
                    float *, float *);
static void classify_run(void);
static void classstats(float *, float *, int, float *, float *);
static void classstats_ap(float *, float *);
static void classstats_el(void);
static void classstats_ellf(float);
static void classstats_final(void);
static void medstat(float *, int, float *, float *);
static void sort1(float *, int);
static void sort2(float *, float *, int);

extern int classify(vir_tfits *catalogue, cpl_propertylist *plist ,
		    int cattype) {
    float fwhm,*work,*areal[NAREAL];
    float pkht,ell,core,ap,delap,area,junk,arg;
    char *cols[NCOL],colname[32];
    cpl_propertylist *extra;
    cpl_table *cat;
    const char *fctid = "vircam_classify";
    int i,n,iap,i1,i2;

    /* Get some DQC info from the extra propertylist generated by imcore */

    extra = vircam_tfits_get_ehu(catalogue);
    thresh = cpl_propertylist_get_float(extra,"ESO DRS THRESHOL");
    skylevel = cpl_propertylist_get_float(extra,"ESO QC MEAN_SKY");
    skynoise = cpl_propertylist_get_float(extra,"ESO QC SKY_NOISE");
    rcore = cpl_propertylist_get_float(extra,"ESO DRS RCORE");
    fwhm = cpl_propertylist_get_float(extra,"ESO DRS SEEING");

    /* Get the exposure time */

    if (vircam_pfits_get_exptime(plist,&exptime) != VIR_OK) {
	cpl_msg_warning(fctid,"Unable to get expsoure time!");
	exptime = 10.0;
    }
    
    /* Get the number of columns and decide which column labels to use */

    cat = vircam_tfits_get_table(catalogue);
    ncols = cpl_table_get_ncol(cat);
    switch (cattype) {
    case 1:
	for (i = 0; i < NCOL; i++)
	    cols[i] = (char *)cols32[i];
	break;
    case 2:
	for (i = 0; i < NCOL; i++)
	    cols[i] = (char *)cols80[i];
	break;
    default:
	cpl_msg_error(fctid,"Don't recognise catalogues with %d columns: cattype == %d",ncols,cattype);
	return(VIR_FATAL);
    }

    /* Make a copy of the table as you are going to muck about with the
       column values. Get the column data */

    catcopy = cpl_table_duplicate(cat);
    nrows = cpl_table_get_nrow(cat);
    core_flux = cpl_table_get_data_float(catcopy,cols[0]);
    core1_flux = cpl_table_get_data_float(catcopy,cols[1]);
    core2_flux = cpl_table_get_data_float(catcopy,cols[2]);
    core3_flux = cpl_table_get_data_float(catcopy,cols[3]);
    core4_flux = cpl_table_get_data_float(catcopy,cols[4]);
    peak_height = cpl_table_get_data_float(catcopy,cols[5]);
    ellipticity = cpl_table_get_data_float(catcopy,cols[6]);
    iso_flux = cpl_table_get_data_float(catcopy,cols[7]);
    total_flux = cpl_table_get_data_float(catcopy,cols[8]);
    core5_flux = cpl_table_get_data_float(catcopy,cols[9]);
    cls = cpl_table_get_data_float(cat,"Classification");
    sig = cpl_table_get_data_float(cat,"Statistic");

    /* Get some workspace */

    workspace = cpl_malloc(2*nrows*sizeof(float));
    peak_mag = workspace;
    work = workspace + nrows;
    
    /* Convert fluxes to "magnitudes" */

    for (i = 0; i < nrows; i++) {
	core_flux[i] = COREMAG(core_flux[i],0.0,1.0);
	core1_flux[i] = COREMAG(core1_flux[i],0.0,1.0);
	core2_flux[i] = COREMAG(core2_flux[i],0.0,1.0);
	core3_flux[i] = COREMAG(core3_flux[i],0.0,1.0);
	core4_flux[i] = COREMAG(core4_flux[i],0.0,1.0);
        core5_flux[i] = COREMAG(core5_flux[i],0.0,1.0);
	iso_flux[i] = COREMAG(iso_flux[i],0.0,1.0);
	total_flux[i] = COREMAG(total_flux[i],0.0,1.0);
	peak_mag[i] = COREMAG(peak_height[i],skynoise,0.1);
    }

    /*  Now get the areal profile information. You'll need this in a sec */

    for (i = 0; i < NAREAL; i++) {
        sprintf(colname,"Areal_%d_profile",i+1);
	areal[i] = cpl_table_get_data_float(catcopy,colname);
    }

    /* What is the seeing like? */

    poor = 0;
    if (fwhm > max(5.0,rcore*sqrt(2.0)))
        poor = 1;

    /* Ok, now call the routine that does all the work */

    classify_run();

    /* Right, now get a better estimate of the seeing */

    n = 0;
    for (i = 0; i < nrows; i++) {
        pkht = peak_height[i];
        ell = ellipticity[i];
	core = core_flux[i];
        if (cls[i] == -1.0 && ell < elllim && core < corlim && 
            pkht > 10.0*thresh) { 
	    ap = log(0.5*pkht/thresh)/log(2.0) + 1.0;
            iap = (int)ap;
            delap = ap - (float)iap;
            if (iap > 0 && iap < NAREAL && areal[1][i] > 0.0) {
	        i1 = (iap-1)*nrows + i;
                i2 = iap*nrows + i;
                area = areal[iap-1][i]*(1.0 - delap) + areal[iap][i]*delap;
                work[n++] = 2.0*sqrt(area/PI);
            }
        }
    }
    if (n > 2) { 
        medstat(work,n,&fwhm,&junk);
       
        /* Allow for finite pixel size */

        arg = 0.25*PI*fwhm*fwhm - 1;
        fwhm = 2.0*sqrt(max(0.0,arg/PI));
       
    } else
        fwhm = -1.0;

    /* Tidy up a bit */

    freespace(workspace);
    freetable(catcopy);

    /* Write header results into extra property list. First the QC */

    cpl_propertylist_update_float(extra,"ESO QC IMAGE_SIZE",fwhm);
    cpl_propertylist_set_comment(extra,"ESO QC IMAGE_SIZE",
				 "[pixels] Average FWHM of stellar objects");
    cpl_propertylist_update_float(extra,"ESO QC ELLIPTICITY",fitell);
    cpl_propertylist_set_comment(extra,"ESO QC ELLIPTICITY",
				 "Average stellar ellipticity (1-b/a)");
    switch (cattype) {
    case 1:
        cpl_propertylist_update_float(extra,"ESO QC APERTURE_CORR",apcor);
        cpl_propertylist_set_comment(extra,"ESO QC APERTURE_CORR",
				     "Stellar ap-corr 1x core flux");
	break;
    case 2:
        cpl_propertylist_update_float(extra,"ESO QC APERTURE_CORR",apcor3);
        cpl_propertylist_set_comment(extra,"ESO QC APERTURE_CORR",
				     "Stellar ap-corr 1x core flux");
	break;
    }
    cpl_propertylist_update_int(extra,"ESO QC NOISE_OBJ",njunk);
    cpl_propertylist_set_comment(extra,"ESO QC NOISE_OBJ",
				 "Number of noise objects");
    cpl_propertylist_update_float(extra,"ESO QC SATURATION",avsat);
    
    /* Now some helpful DRS keywords */

    cpl_propertylist_update_bool(extra,"ESO DRS CLASSIFD",1);
    cpl_propertylist_set_comment(extra,"ESO DRS CLASSIFD",
				 "Catalogue has been classified");

    /* Now the aperture correction keywords */

    switch (cattype) {
    case 1:
        cpl_propertylist_update_float(extra,"APCORPK",apcpkht);
        cpl_propertylist_set_comment(extra,"APCORPK","Stellar aperture correction - peak height");
        cpl_propertylist_update_float(extra,"APCOR1",apcor1);
        cpl_propertylist_set_comment(extra,"APCOR1","Stellar aperture correction - 1/2x core flux");
        cpl_propertylist_update_float(extra,"APCOR",apcor);
        cpl_propertylist_set_comment(extra,"APCOR","Stellar aperture correction - 1x core flux");
        cpl_propertylist_update_float(extra,"APCOR2",apcor2);
        cpl_propertylist_set_comment(extra,"APCOR2","Stellar aperture correction - sqrt(2)x core flux");
        cpl_propertylist_update_float(extra,"APCOR3",apcor3);
        cpl_propertylist_set_comment(extra,"APCOR3","Stellar aperture correction - 2x core flux");
        cpl_propertylist_update_float(extra,"APCOR4",apcor4);
        cpl_propertylist_set_comment(extra,"APCOR4","Stellar aperture correction - 2*sqrt(2)x core flux");
        cpl_propertylist_update_float(extra,"APCOR5",apcor5);
        cpl_propertylist_set_comment(extra,"APCOR5","Stellar aperture correction - 4x core flux");
	break;
    case 2:
        cpl_propertylist_update_float(extra,"APCORPK",apcpkht);
        cpl_propertylist_set_comment(extra,"APCORPK","Stellar aperture correction - peak height");
        cpl_propertylist_update_float(extra,"APCOR1",apcor1);
        cpl_propertylist_set_comment(extra,"APCOR1","Stellar aperture correction - 1/2x core flux");
        cpl_propertylist_update_float(extra,"APCOR2",apcor2);
        cpl_propertylist_set_comment(extra,"APCOR2","Stellar aperture correction - core/sqrt(2) flux");
        cpl_propertylist_update_float(extra,"APCOR3",apcor3);
        cpl_propertylist_set_comment(extra,"APCOR3","Stellar aperture correction - 1x core flux");
        cpl_propertylist_update_float(extra,"APCOR4",apcor4);
        cpl_propertylist_set_comment(extra,"APCOR4","Stellar aperture correction - sqrt(2)x core flux");
        cpl_propertylist_update_float(extra,"APCOR5",apcor5);
        cpl_propertylist_set_comment(extra,"APCOR5","Stellar aperture correction - 2x core flux");
        cpl_propertylist_update_float(extra,"APCOR6",apcor6);
        cpl_propertylist_set_comment(extra,"APCOR6","Stellar aperture correction - 2*sqrt(2)x core flux");
        cpl_propertylist_update_float(extra,"APCOR7",apcor7);
        cpl_propertylist_set_comment(extra,"APCOR7","Stellar aperture correction - 4x core flux");

	break;
    }

    /* Write header information to help GAIA */

    cpl_propertylist_update_string(extra,"SYMBOL1","{Ellipticity Position_angle Areal_1_profile Classification} {el");
    cpl_propertylist_update_string(extra,"SYMBOL2","lipse blue (1.0-$Ellipticity) $Position_angle+90 {} $Classific");
    cpl_propertylist_update_string(extra,"SYMBOL3","ation==1} {sqrt($Areal_1_profile*(1.0-$Ellipticity)/3.142)} : {");
    cpl_propertylist_update_string(extra,"SYMBOL4","Ellipticity Position_angle Areal_1_profile Classification} {el");
    cpl_propertylist_update_string(extra,"SYMBOL5","lipse red (1.0-$Ellipticity) $Position_angle+90 {} $Classific");
    cpl_propertylist_update_string(extra,"SYMBOL6","ation==-1} {sqrt($Areal_1_profile*(1.0-$Ellipticity)/3.142)} :");
    cpl_propertylist_update_string(extra,"SYMBOL7","{Ellipticity Position_angle Areal_1_profile Classification} {el");
    cpl_propertylist_update_string(extra,"SYMBOL8","lipse green (1.0-$Ellipticity) $Position_angle+90 {} $Classifi");
    cpl_propertylist_update_string(extra,"SYMBOL9","cation==0} {sqrt($Areal_1_profile*(1.0-$Ellipticity)/3.142)}");

    /* Get out of here */

    return(VIR_OK);
}


static void anhist(float *data, int n, float *medval, float *sigma) {
    int i,*histo,ilev,imax,ismax;
    float *sval,hmax,smax,hlim,ratio;

    /* Get some workspace for the histogram */

    histo = cpl_calloc(MAXHIST,sizeof(int));
    sval = cpl_calloc(MAXHIST,sizeof(float));

    /* Sort data into the histogram */

    for (i = 0; i < n; i++) {
        ilev = vircam_nint(data[i]/STEP);
        if (ilev >= -10 && ilev <= 100) {
 	    ilev += 10;
            histo[ilev] += 1;
        }
    }

    /* Now find the maximum of the histogram and its position... */

    hmax = 0.0;
    imax = 0;
    for (i = 0; i < MAXHIST; i++) {
        if (histo[i] > hmax) {
 	    hmax = (float)histo[i];
            imax = i;
        }
    }

    /* Trap for hmax == 0 */

    if (hmax == 0.0) {
        if (n >= 10) {
	    *medval = data[(n+1)/2-1];
            *sigma = 1.48*0.5*(data[(3*n+3)/4-1] - data[(n+3)/4-1]);
        } else {
  	    *medval = 0.0;
	    *sigma = 1.0;
        }
        return;
    }

    /* Now do three point running average to see if there are other local
       maxima */

    smax = 0.0;
    ismax = 0;
    for (i = 1; i < MAXHIST-1; i++) {
        sval[i] = (histo[i-1] + histo[i] + histo[i+1])/3.0;
        if (sval[i] > smax) {
  	    smax = sval[i];
            ismax = i;
        }
    }
    if (ismax < imax) {
        imax = ismax;
        hmax = (float)histo[imax];
    }

    /* Now check for lower local maxima */

    for (i = imax-1; i > 0; i--) {
        if (sval[i] >= sval[i+1] && sval[i] >= sval[i-1]) {
	    if (sval[i] > 0.5*smax)
  	        ismax = i;
        }
    }
    if (ismax < imax) {
	imax = ismax;
        hmax = (float)histo[imax];
    }
    
    /* Now work out where the peak is */
    
    *medval = min((float)(imax-10)*STEP,data[(n+1)/2-1]);
    hlim = vircam_nint(0.5*hmax);
    i = 1;
    while (histo[imax-i] > hlim && imax-i > 1)
        i++;
    ratio = hmax/max(1.0,(float)histo[imax-i]);
    *sigma = (float)i*STEP/(sqrt(2.0)*max(1.0,log(ratio)));
    *sigma = max(*sigma,0.5*STEP);

    /* Tidy and exit */
    
    freespace(histo);
    freespace(sval);
}


static void boundries(float *core1, float *core2, float *core3, float medval1,
                      float sigma1, float medval2, float sigma2, int small, 
                      float area1, float area2, float *wt, float *avsig, 
                      float *lower, float *upper) {
    int i,n;
    float c1,c2,dc,*work,xnoise,xmag,xflux,ratio,asign,junk;

    /* Get a workspace */

    work = cpl_malloc(nrows*sizeof(float));

    /* Initialise the lower boundry */

    lower[0] = cmin;
    lower[1] = cmax;
    asign = ((small == 1) ? -1.0 : 1.0);
    
    /* Now collect the data */

    n = 0;
    for (i = 0; i < nrows; i++) {
        c1 = core1[i];
        if (! poor) {
            c2 = core2[i];
            dc = asign*(c2 - c1);
            if (dc > medval1 - 3.0*sigma1 && c1 < 12.0)
 	        work[n++] = dc - medval1;
	} else {
            c2 = core3[i];
            dc = c2 - c1;
            if (dc > medval2 - 3.0*sigma2 && c1 < 12.0)
 	        work[n++] = dc - medval2;
        }
    }
 
    /* Find the median */

    medstat(work,n,avsig,&junk);
    freespace(work);

    /* Work out sigma levels for both types of seeing */

    if (! poor) {
        *wt = min(5.0,max(1.0,*avsig/sigma1));
        xnoise = sqrt(area1)*skynoise;
    } else {
        *wt = min(2.5,max(1.0,*avsig/sigma2));
        xnoise = sqrt(area2)*skynoise;
    }

    /* Now work out the boundries */

    for (i = 0; i < NSAMPLE; i++) {
        xmag = 5.0 + (float)(i+1)*0.1;
        xflux = pow(10.0,(double)(0.4*xmag));
        ratio = COREMAG(1.0+xnoise/xflux,0.0,0.0);
        if (! poor) {
 	    lower[i] = medval1 - 3.0*sqrt(sigma1*sigma1 + ratio*ratio);
            upper[i] = medval1 + 3.0*sqrt(sigma1*sigma1 + 0.5*ratio*ratio);
        } else {
      	    lower[i] = medval2 - 3.0*sqrt(sigma2*sigma2 + ratio*ratio);
            upper[i] = medval2 + 3.0*sqrt(sigma2*sigma2 + 0.5*ratio*ratio);
        }
    }
    upper[0] = ((poor == 0) ? medval1 : medval2);
    upper[1] = upper[0];
}

static void boundpk(float *core, float *pkht, float medval, float sigma, 
                    float *wt, float *avsig, float *lower, float *upper) {
    int i,n;
    float c,p,*work,xnoise,xmag,pmag,xflux,pflux,ratio,junk;

    /* Get the space for the boundry lines and a workspace */

    work = cpl_malloc(nrows*sizeof(float));

    /* Collect the data */

    n = 0;
    for (i = 0; i < nrows; i++) {
	c = core[i];
	p = pkht[i];
        if (c - p > medval - 3.0*sigma && c < 12)
  	    work[n++] = c - p - medval;
    }

    /* Find the median */

    medstat(work,n,avsig,&junk);
    freespace(work);
    *wt = min(5.0,max(1.0,*avsig/sigma));

    /* Now work out boundries */

    xnoise = sqrt(PI*rcore*rcore)*skynoise;
    for (i = 0; i < NSAMPLE; i++) {
        xmag = 5.0 + (float)(i+1)*0.1;
        pmag = xmag - medval;
        xflux = pow(10.0,(double)(0.4*xmag));
        pflux = pow(10.0,(double)(0.4*pmag));
        ratio = 2.5*log10((double)(1.0+max(xnoise/xflux,skynoise/pflux)));
        lower[i] = medval - 3.0*sqrt(sigma*sigma + ratio*ratio);
        upper[i] = medval + 3.0*sqrt(sigma*sigma + 0.5*ratio*ratio);
    }
    upper[0] = medval;
    upper[1] = upper[0];
}


static void classify_run(void) {
    float fluxlim,ell,pk,pkht,core,sig1,sig2,sig3,denom,w1,w2,w3;
    float core_small,core_large,core_midd,statistic,statcut,sigtot;
    float fit0,sigma0,xnoise,xmag,ratio,xflux,ratell,ratscl,ellbound;
    float *lower,*upper,sigma1,sigma2,sigma3,sigma4,sigma5;
    int i,iarg;

    /* Update faint limit to cope with short exposures */

    blim = BLIMDEF;
    flim = FLIMDEF;
    fluxlim = 2.5*log10((double)(5.0*sqrt(PI*rcore*rcore)*skynoise));
    flim = min(flim,max(6.0,fluxlim+3.0));
    corlim = min(blim,max(12.5,fluxlim+7.0));
    cormin = min(blim,max(12.5,fluxlim+6.5));

    /* Work out min and max core flux */

    cmin = CMINDEF;
    cmax = CMAXDEF;
    for (i = 0; i < nrows; i++) {
	xflux = core_flux[i];
	cmin = min(cmin,xflux);
	cmax = max(cmax,xflux);
    }
    cmin = max(fluxlim-0.5,cmin);
    cmax += 0.1;
    cmax = min(cmax,20.0);

    /* Work out ellipticity stats for likely stellar objects */

    classstats_el();

    /* Ok, get the classification statistics for each of the tests.  First
       the core flux vs 1/2*core flux */

    classstats(core_flux,core1_flux,1,&fit1,&sigma1);

    /* Core flux vs 2*core flux */

    classstats(core_flux,core3_flux,0,&fit2,&sigma2);

    /* Core flux vs sqrt(2)*core flux */

    classstats(core_flux,core2_flux,0,&fit4,&sigma4);

    /* Core flux vs 2*sqrt(2)*core flux */

    classstats(core_flux,core4_flux,0,&fit5,&sigma5);

    /* Core flux vs Peak height */

    classstats(core_flux,peak_mag,1,&fit3,&sigma3);

    /* Faint end ellipticity */

    classstats_ellf(fluxlim);

    /* Get workspace for the boundry arrays */
   
    lower1 = cpl_malloc(NSAMPLE*sizeof(float));
    lower2 = cpl_malloc(NSAMPLE*sizeof(float));
    lower3 = cpl_malloc(NSAMPLE*sizeof(float));
    upper1 = cpl_malloc(NSAMPLE*sizeof(float));
    upper2 = cpl_malloc(NSAMPLE*sizeof(float));
    upper3 = cpl_malloc(NSAMPLE*sizeof(float));

    /* Right, work out the boundries for the classification tests 
       First core vs sqrt(2)*core or core vs 0.5*core depending upon
       the seeing */

    boundries(core_flux,core1_flux,core2_flux,fit1,sigma1,fit4,sigma4,
              1,PI*rcore*rcore,2.0*PI*rcore*rcore,&wt1,&avsig1,lower1,
              upper1);

    /* Now core vs 2*core or core vs 2*sqrt(2)*core */

    boundries(core_flux,core3_flux,core4_flux,fit2,sigma2,fit5,sigma5,
              0,4.0*PI*rcore*rcore,8.0*PI*rcore*rcore,&wt2,&avsig2,lower2,
	      upper2);

    /* Now core vs peak height */

    boundpk(core_flux,peak_mag,fit3,sigma3,&wt3,&avsig3,lower3,upper3); 

     
    /* Do final classification statistics and find the saturation limit */

    classstats_final();

    /* Define final boundries */

    lower = cpl_malloc(NSAMPLE*sizeof(float));
    upper = cpl_malloc(NSAMPLE*sizeof(float));
    uppere = cpl_malloc(NSAMPLE*sizeof(float));
    xnoise = sqrt(PI*rcore*rcore)*skynoise;
    ratell = xnoise/pow(10.0,0.4*(fluxlim+1.5));
    ratell = COREMAG(1.0+ratell,0.0,0.0);
    ratscl = (pow((fitellf + 2.0*sigellf - fitell),2.0) - 4.0*sigell*sigell)/(4.0*ratell*ratell);
    ratscl = max(0.25,min(10.0,ratscl));
    for (i = 0; i < NSAMPLE; i++) {
        xmag = 5.0 + 0.1*(float)(i+1);
        xflux = pow(10.0,0.4*xmag);
        ratio = 2.5*log10(1.0+xnoise/xflux);
        lower[i] = fit_final - 5.0*sqrt(sigma_final*sigma_final + ratio*ratio);
        upper[i] = fit_final + sqrt(9.0*sigma_final*sigma_final + 0.0*ratio*ratio);
	uppere[i] = fitell + 2.0*sqrt(sigell*sigell + ratscl*ratio*ratio);
	uppere[i] = min(0.5,uppere[i]);
    }
    elllim = min(0.5,max(0.25,fitell+2.0*sigell));
    fluxlim = 2.5*log10((double)(2.5*sqrt(PI*rcore*rcore)*skynoise));

    /* Ok, final classification loop now... */

    nstar = 0;
    ngal = 0;
    njunk = 0;
    ncmp = 0;
    for (i = 0; i < nrows; i++) {
	ell = ellipticity[i];
        pk = peak_height[i] + skylevel;
	pkht = peak_mag[i];
	core = core_flux[i];
	iarg = vircam_nint(10.0*(core - 5.0));
	iarg = max(1,min(NSAMPLE,iarg)) - 1;
	if (! poor) {
	    sig1 = max(0.01,(fit1 - lower1[iarg])/3.0);
	    sig2 = max(0.01,(fit2 - lower2[iarg])/3.0);
	} else {
	    sig1 = max(0.01,(fit4 - lower1[iarg])/3.0);
	    sig2 = max(0.01,(fit5 - lower2[iarg])/3.0);
	}
	sig3 = max(0.01,(fit3 - lower3[iarg])/3.0);
	denom = (wt1/sig1 + wt2/sig2 + wt3/sig3);
	w1 = (wt1/sig1)/denom;
	w2 = (wt2/sig2)/denom;
	w3 = (wt3/sig3)/denom;
	if (! poor) {
	    core_small = core1_flux[i];
	    core_large = core3_flux[i];
	    statistic = (core - core_small - fit1)*w1 + 
		(max(-3.0*sig2,core_large - core - fit2))*w2 + 
                (core - pkht - fit3)*w3;
	} else {
	    core_midd = core2_flux[i];
	    core_large = core4_flux[i];
	    statistic = (core_midd - core - fit4)*w1 +
		(max(-3.0*sig2,core_large - core - fit5))*w2 + 
                (core - pkht - fit3)*w3;
	}
        cls[i] = -1.0;
        statcut = upper[iarg] + 3.0*sigma_final*(exp(max(0.0,core-corlim+1.0)) - 1.0);
        if (statistic  >= statcut) 
 	    cls[i] = 1.0;
        else if (statistic <= lower[iarg])
  	    cls[i] = 0.0;

        /* Save distance from the stellar locus */

        sigtot = (fit_final - lower[iarg])/5.0;
        sig[i] = (statistic - fit_final)/sigtot;

        /* Right, now here are lots of overrides for special circumstances */
        /* Too spikey? -> junk */

        if (core - pkht - fit3 < -4.0*sig3) 
            cls[i] = 0.0;

        /* Elliptical star? -> compact */

	ellbound = max(elllim,uppere[iarg]);
        if (ell > ellbound && cls[i] == -1.0 && core < flim && sig[i] > -2.0)
	    cls[i] = -2.0;

        /* Saturated? -> star */

	if (core > corlim && statistic >= lower[iarg])
  	    cls[i] = -1.0;

	/* Too elliptical? -> junk */

        if (ell > 0.9 && core < corlim)
	    cls[i] = 0.0;

        /* Too faint? -> junk */

        if (core < fluxlim)
	    cls[i] = 0.0;

        /* Now count how many you have of each */

        if (cls[i] == -1.0)
	    nstar++;
        else if (cls[i] == 1.0)
	    ngal++;
	else if (cls[i] == -2.0)
	    ncmp++;
        else
	    njunk++;
    }

    /* Do stats to get the aperture corrections */

    classstats_ap(&fit0,&sigma0);
    fit0 = max(fit5,fit0);
    apcpkht = fit0 + fit3; /* pkht */
    switch (ncols) {
    case 32:
        apcor1 = fit0 + fit1;  /* 0.5*core */
        apcor = fit0;          /* core */
        apcor2 = fit0 - fit4;  /* sqrt(2)*core */
        apcor3 = fit0 - fit2;  /* 2*core */
        apcor4 = fit0 - fit5;  /* 2*sqrt(2)*core */
        apcor5 = 0.0;          /* 4*core */
	break;
    case 80:
        apcor1 = fit0 + fit1;      /* 0.5*core */
        apcor2 = fit0 + 0.5*fit1;  /* 1/sqrt(2) * core */
        apcor3 = fit0;             /* core */
        apcor4 = fit0 - fit4;      /* core * sqrt(2) */
        apcor5 = fit0 - fit2;      /* 2*core */
        apcor6 = fit0 - fit5;      /* 2*sqrt(2)*core */
        apcor7 = 0.0;              /* 4*core */
	break;
    }

    /* Ok, now get rid of some workspace */
        
    freespace(lower1);
    freespace(lower2);
    freespace(lower3);
    freespace(upper1);
    freespace(upper2);
    freespace(upper3);
    freespace(lower);
    freespace(upper);
    freespace(uppere);

}

static void classstats(float *core1, float *core2, int small, float *medval, 
		       float *sigma) {

    int i,iloop,n;
    float *work,*dc,sigmaold,amult;

    /* Initialise the output values to something stupid */

    *medval = 0.0;
    *sigma = 1.0e6;
    amult = (small == 1 ? -1.0 : 1.0);

    /* Get some workspace */

    work = cpl_malloc(nrows*sizeof(float));
    dc = cpl_malloc(nrows*sizeof(float));

    /* Work out differences */

    for (i = 0; i < nrows; i++)
	dc[i] = amult*(core2[i] - core1[i]);

    /* Do an iteration loop */

    for (iloop = 0; iloop < MAXLOOP; iloop++) {
        sigmaold = *sigma;
        n = 0;

        /* Ok, gather up all the stats */

        for (i = 0; i < nrows; i++) {
            
            /* Clipping criteria */

            if (ellipticity[i] < elllim && core1[i]< blim && core1[i] > flim &&
		fabs(dc[i] - *medval) < 3.0*(*sigma)) {
	        work[n++] = dc[i];
            }
        }

        /* Sort the work array and find the median and sigma */

	if (n > 2) {
            sort1(work,n);
            if (iloop == 0 && n > 10) {
	        anhist(work,n,medval,sigma);
            } else {
  	        medstat(work,n,medval,sigma);
                *sigma = min(sigmaold,*sigma);
            }
	} else {
	    *medval = 0.0;
	    *sigma = 0.01;
	}

        /* Just in case... */

        *sigma = max(*sigma,0.01);
    }

    /* Tidy and exit */

    freespace(work);
    freespace(dc);
}

static void classstats_el(void) {
    int iloop,n,i;
    float *work;

    /* Initialise the mean and sigma to something stupid */

    sigell = 1.0e6;
    fitell = 0.0;

    /* Get some workspace */

    work = cpl_malloc(nrows*sizeof(float));

    /* Do iteration loop */

    for (iloop = 0; iloop < MAXLOOP; iloop++) {
        n = 0;
        for (i = 0; i < nrows; i++) {
            if (ellipticity[i] < 0.5 && core_flux[i] < blim && core_flux[i] > flim &&
		fabs(ellipticity[i] - fitell) < 2.0*sigell)
 	        work[n++] = ellipticity[i];
        }
	if (n > 2)
            medstat(work,n,&fitell,&sigell);
	else {
	    fitell = 0.25;
	    sigell = 0.05;
        }
    }
    elllim = min(0.5,max(0.2,fitell+2.0*sigell));

    /* Get out of here */

    freespace(work);
}

static void classstats_ellf(float fluxlim) {
    int iloop,n,i;
    float *work;

    /* Initialise the mean and sigma to something stupid */

    sigellf = 1.0e6;
    fitellf = 0.0;

    /* Get some workspace */

    work = cpl_malloc(nrows*sizeof(float));

    /* Do iteration loop */

    for (iloop = 0; iloop < MAXLOOP; iloop++) {
        n = 0;
        for (i = 0; i < nrows; i++) {
            if (ellipticity[i] < 0.75 && core_flux[i] > fluxlim+1.0 && 
		core_flux[i] < fluxlim+2.0 &&
		fabs(ellipticity[i] - fitellf) < 2.0*sigellf)
 	        work[n++] = ellipticity[i];
        }
	if (n > 2)
            medstat(work,n,&fitellf,&sigellf);
	else {
	    fitellf = 0.25;
	    sigellf = 0.05;
        }
    }

    /* Get out of here */

    freespace(work);
}

static void classstats_ap(float *medval, float *sigma) {

    int i,iloop,n;
    float *work,*dc,c2;

    /* Initialise the output values to something stupid */

    *medval = 0.0;
    *sigma = 1.0e6;
    elllim = min(0.5,max(0.2,fitell+2.0*sigell));

    /* Get some workspace */

    work = cpl_malloc(nrows*sizeof(float));
    dc = cpl_malloc(nrows*sizeof(float));

    /* Work out differences */

    for (i = 0; i < nrows; i++) {
        c2 = max(iso_flux[i],max(total_flux[i],core5_flux[i]));
        dc[i] = c2 - core_flux[i];
    }

    /* Do an iteration loop */

    for (iloop = 0; iloop < MAXLOOP; iloop++) {
        n = 0;

        /* Ok, gather up all the stats */

        for (i = 0; i < nrows; i++) {
            
            /* Clipping criteria */

            if (ellipticity[i] < elllim && core_flux[i] < blim + 1.0 && 
		core_flux[i] > flim + 1.0 && 
		fabs(dc[i] - *medval) < 3.0*(*sigma) && cls[i] == -1.0)
	        work[n++] = dc[i];
        }

        /* Sort the work array and find the median and sigma */

	if (n > 2) {
            sort1(work,n);
            if (iloop == 0 && n > 10) {
	        anhist(work,n,medval,sigma);
            } else {
  	        medstat(work,n,medval,sigma);
            }
	} else {
	    *medval = 0.0;
	    *sigma = 0.01;
	}

        /* Just in case... */

        *sigma = max(*sigma,0.01);
    }

    /* Tidy and exit */

    freespace(work);
    freespace(dc);
}

static void classstats_final(void) {
    int n,i,iloop,iarg,ii,iend,ncls,kk,k;
    float *work,ell,core,sig1,sig2,sig3,denom,w1,w2,w3,core_small;
    float core_large,*statistic,core_midd,pkht,xcor,cfit,csig;
    float *work1,junk,corlim1,corval1,corlim2,corval2,sigmaold;

    /* Initialise */

    sigma_final = 1.0e6;
    fit_final = 0.0;
    ncls = 0;

    /* Get some workspace */

    work = cpl_malloc(nrows*sizeof(float));
    work1 = cpl_malloc(nrows*sizeof(float));
    statistic = cpl_malloc(nrows*sizeof(float));

    /* Calculate the statistic now */

    for (i = 0; i < nrows; i++) {
	ell = ellipticity[i];
	pkht = peak_mag[i];
	core = core_flux[i];
	iarg = vircam_nint(10.0*(core - 5.0));
	iarg = max(1,min(NSAMPLE,iarg)) - 1;
	if (! poor) {
	    sig1 = max(0.01,(fit1 - lower1[iarg])/3.0);
	    sig2 = max(0.01,(fit2 - lower2[iarg])/3.0);
	} else {
	    sig1 = max(0.01,(fit4 - lower1[iarg])/3.0);
	    sig2 = max(0.01,(fit5 - lower2[iarg])/3.0);
	}
	sig3 = max(0.01,(fit3 - lower3[iarg])/3.0);
	denom = (wt1/sig1 + wt2/sig2 + wt3/sig3);
	w1 = (wt1/sig1)/denom;
	w2 = (wt2/sig2)/denom;
	w3 = (wt3/sig3)/denom;
	if (! poor) {
	    core_small = core1_flux[i];
	    core_large = core3_flux[i];
	    statistic[i] = (core - core_small - fit1)*w1 + 
		(core_large - core - fit2)*w2 + (core - pkht - fit3)*w3;
	} else {
	    core_midd = core2_flux[i];
	    core_large = core4_flux[i];
	    statistic[i] = (core_midd - core - fit4)*w1 +
		(core_large - core - fit5)*w2 + (core - pkht - fit3)*w3;
	}
    }

    /* Iteration loop.  Use only lower ellipticity images and relevant
       peak height range */

    for (iloop = 0; iloop < MAXLOOP; iloop++) {
        sigmaold = sigma_final;
        n = 0;
        for (i = 0; i < nrows ; i++) {

  	    ell = ellipticity[i];
 	    core = core_flux[i];
   	    if (ell < elllim && core < blim && core > flim && 
	        fabs((double)(statistic[i] - fit_final)) < 3.0*sigma_final) 
	        work[n++] = statistic[i];

	    /* This information is to be used later to find the curvature of 
               saturated region */

            if (core > corlim && iloop == MAXLOOP-2) {
 	        cls[ncls] = statistic[i];
                sig[ncls++] = core;
            }
        }

        /* Median defines general fit */

	if (n > 2) {
            sort1(work,n);
            if (iloop == 0 && n > 10) {
                anhist(work,n,&fit_final,&sigma_final);
            } else {
	        medstat(work,n,&fit_final,&sigma_final);
            }
            sigma_final = max(0.01,min(sigmaold,sigma_final));
	} else {
	    fit_final = 0.0;
	    sigma_final = 0.01;
        }
    }

    /* Now work out the curvature in the saturated region */

    sort2(sig,cls,ncls);
    ii = 0;
    xcor = 12.5;
    iend = 0;
    i = -1;
    corlim1 = 0.0;
    corlim2 = 0.0;
    corval1 = 0.0;
    corval2 = 0.0;
    while (iend == 0 && i < ncls-1) {
        i++;
        if (sig[i] > xcor+0.25 && ii >= 3) {
	    medstat(work,ii,&cfit,&csig);
            for (iloop = 0; iloop < 3; iloop++) {
	        kk = 0;
                for (k = 0; k < ii; k++) {
		    if (work[k] <= cfit + 3.0*csig)
		        work1[kk++] = work[k];
                }
                medstat(work1,kk,&cfit,&junk);
	    }
            if (cfit <= fit_final + 3.0*sigma_final) {
	        corlim1 = xcor;
                corval1 = cfit;
	    } else {
	        corlim2 = xcor;
                corval2 = cfit;
                iend = 1;
            }
	} else {
 	    work[ii++] = cls[i];
        }
    }

    /* Estimate where core measure and statistic become unreliable */

    if (iend == 1) 
        corlim = corlim2 - 0.5*(corval2 - fit_final - 3.0*sigma_final)/(corval2 - corval1);
    else 
        corlim = corlim1;
    corlim = max(cormin,corlim);
    kk = 0;
    for (i = 0; i < nrows; i++) {
        core = core_flux[i];
        if (core >= corlim)
	    work[kk++] = peak_height[i] + skylevel;
    }
    medstat(work,kk,&avsat,&junk);
    avsat = max(10000.0,avsat);

    /* Tidy and exit */
  
    freespace(work);
    freespace(work1);
    freespace(statistic);
}


static void medstat(float *array, int n, float *medval, float *sigval) {
    int lev1,lev2,lev3;

    /* Sort the array first, then choose the median.  The sigma is defined
       as half the distance between the two quartile points multiplied by
       the appropriate scaling factor (1.48) */

    if (n == 0) {
	*medval = 0.0;
	*sigval = 0.0;
	return;
    }
    sort1(array,n);
    lev1 = (int)(0.5*(float)(n + 1));
    lev2 = (int)(0.25*(float)(3*n + 3));
    lev3 = (int)(0.25*(float)(n + 3));
    *medval = array[lev1-1];
    *sigval = 1.48*0.5*(array[lev2-1] - array[lev3-1]);
}


static void sort1(float *a, int n) {
    int iii,ii,i,ifin,j;
    float b;

    iii = 2;
    while (iii < n)
        iii *= 2;
    iii = min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a[i] > a[j]) {
                b = a[j];
                while (1) {
                    a[j] = a[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a[i] <= b) 
                        break;
                }
                a[j] = b;
            }
        }
    }
}

static void sort2(float *a1, float *a2, int n) {
    int iii,ii,i,ifin,j;
    float b1,b2;

    iii = 4;
    while (iii < n)
        iii *= 2;
    iii = min(n,(3*iii)/4 - 1);

    while (iii > 1) {
        iii /= 2;
        ifin = n - iii;
        for (ii = 0; ii < ifin; ii++) {
            i = ii;
            j = i + iii;
            if (a1[i] > a1[j]) {
                b1 = a1[j];
                b2 = a2[j];
                while (1) {
                    a1[j] = a1[i];
                    a2[j] = a2[i];
                    j = i;
                    i = i - iii;
                    if (i < 0 || a1[i] <= b1) 
                        break;
                }
                a1[j] = b1;
                a2[j] = b2;
            }
        }
    }
}


/*

$Log: classify.c,v $
Revision 1.9  2007/05/02 09:11:34  jim
Modified to allow for inclusion of table WCS keywords into FITS header

Revision 1.8  2007/03/01 12:38:26  jim
Small modifications after a bit of code checking

Revision 1.7  2006/06/13 14:06:57  jim
The classification and statistic data rows must come from the original table
rather than the copy as they are being written to

Revision 1.6  2006/05/26 15:00:36  jim
Now makes a copy of the input table so that it doesn't muck up the column
values

Revision 1.5  2006/05/18 12:34:20  jim
Fixed header keywords for input information

Revision 1.4  2006/03/15 10:43:42  jim
Fixed a few things

Revision 1.3  2005/11/28 13:50:06  jim
Touched up a few things to make splint happy

Revision 1.2  2005/11/03 13:28:51  jim
All sorts of changes to tighten up error handling

Revision 1.1  2005/09/22 08:41:03  jim
first entry


*/
