/* $Id: vircam_platesol.c,v 1.13 2007/10/25 19:38:22 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 19:38:22 $
 * $Revision: 1.13 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"
#include "vircam_paf.h"

/* Function prototypes */

static int vircam_platesol_create(cpl_plugin *) ;
static int vircam_platesol_exec(cpl_plugin *) ;
static int vircam_platesol_destroy(cpl_plugin *) ;
static int vircam_platesol_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_platesol_save(void);
static void vircam_platesol_init(void);
static void vircam_platesol_tidy(void);

static struct {

    /* Input */

    int         nconst;
    int         shiftan;
    int         extenum;

} vircam_platesol_config;

static struct {
    int         *labels;
    cpl_frame   *img;
    cpl_frame   *mstds;
    vir_fits    *imgf;
    vir_tfits   *mstdsf;
    FILE        *paf;
} ps;


static char vircam_platesol_description[] =
"vircam_platesol -- VIRCAM plate solution fitting test recipe.\n\n"
"Fit a plate solution to a matched standards table and write the resulting\n"
"WCS to an input image header.\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A input uncorrected image\n"
"    %-21s A matched standards table\n"
"The WCS values from the solution are written to a paf file. This is the\n"
"only product of this recipe\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_platesol vircam_platesol
    \brief Test recipe to drive the vircam_platesol library function.

    \par Name:
        vircam_platesol
    \par Purpose:
        Test recipe to drive the vircam_platesol library function.
    \par Description:
        Test the library function vircam_platesol fitting a plate solution
	to a matched standards table. The results are written to the extension
	header of an input image, but the image header is not updated in
	this recipe. Rather the WCS values and QC information are written
	to a paf file.
    \par Language:
        C
    \par Parameters:
        - \b nconst (float): The number of plate constants in the fit.
        - \b shiftan (bool): If set, then an attempt will be made to relocate
	  the tangent point by looking at the predicted vs true equatorial
	  coordinates in the matched standards catalogue.
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b SCIENCE_IMAGE (required): A single raw image. This is needed for 
	  its input WCS.
        - \b MATCHED_STANDARDS_TABLE (required): A matched standards table.
    \par Output File Types:
        None. Results written to PAF
    \par Output QC Parameters:
        None.
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input image or matched standards table.
	- Missing input WCS information in the input image
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_platesol.c
*/


/* Function code */


/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_platesol_description,
		   VIRCAM_TEST_SCIENCE_RAW,VIRCAM_CAL_MSTDTAB);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_platesol",
                    "VIRCAM plate solution test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_platesol_create,
                    vircam_platesol_exec,
                    vircam_platesol_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_platesol_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Fill in the parameters. First the number of constants */

    p = cpl_parameter_new_enum("vircam.vircam_platesol.nconst",
			       CPL_TYPE_INT,
			       "Number of plate constants",
			       "vircam.vircam_platesol",6,2,4,6);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"nconst");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Now whether or not to shift the tangent point */

    p = cpl_parameter_new_value("vircam.vircam_platesol.shiftan",
				CPL_TYPE_BOOL,
				"Shift position of tangent point",
				"vircam.vircam_platesol",FALSE);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"shiftan");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_platesol.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_platesol",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_platesol_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_platesol_test(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_platesol_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_platesol_test(cpl_parameterlist *parlist, 
				cpl_frameset *framelist) {
    const char *fctid="vircam_platesol";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_platesol_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_platesol.nconst");
    vircam_platesol_config.nconst = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_platesol.shiftan");
    vircam_platesol_config.shiftan = cpl_parameter_get_bool(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_platesol.extenum");
    vircam_platesol_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_platesol_tidy();
        return(-1);
    }

    /* Get the frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_platesol_tidy();
        return(-1);
    }
    if ((ps.mstds = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					       VIRCAM_CAL_MSTDTAB)) == NULL) {
        cpl_msg_info(fctid,"No matched standards table found -- cannot continue");
	vircam_platesol_tidy();
	return(-1);
    }
    if ((ps.img = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
                                             VIRCAM_TEST_SCIENCE_RAW)) == NULL) {
        cpl_msg_info(fctid,"No raw image found -- cannot continue");
	vircam_platesol_tidy();
	return(-1);
    }

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_platesol_config.extenum,(const cpl_frame *)ps.img,
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_platesol_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {

        /* Load up the images */

        ps.imgf = vircam_fits_load(ps.img,CPL_TYPE_FLOAT,j);
        ps.mstdsf = vircam_tfits_load(ps.mstds,j);
        if (ps.img == NULL || ps.mstdsf == NULL) {
  	    freefits(ps.imgf);
	    freetfits(ps.mstdsf);
	    cpl_msg_warning(fctid,"Unable to load one of the inputs");
	    continue;
	}

	/* Now do the correction */

	cpl_msg_info(fctid,"Doing the plate solution for extension %d",j);
        (void)vircam_platesol(vircam_fits_get_ehu(ps.imgf),NULL,
			      vircam_tfits_get_table(ps.mstdsf),
			      vircam_platesol_config.nconst,
			      vircam_platesol_config.shiftan,&status);
	if (status != VIR_OK) {
	    cpl_msg_warning(fctid,"Plate solution failed");
	    status = VIR_OK;
	}

        /* Now save the result */

	cpl_msg_info(fctid,"Saving results for extension %d",j);
	if (vircam_platesol_save() != 0)
	    cpl_msg_warning(fctid,"Save routine failed");

	/* Tidy a few things before the next image */

	freefits(ps.imgf);
	freetfits(ps.mstdsf);
    }
    vircam_platesol_tidy();
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_platesol_save(void) {
    const char *fctid = "vircam_platesol_save";
    const char *outfile = "platesol";
    const char *keys[] = {"CRVAL1","CRVAL2","CRPIX1","CRPIX2","CD1_1","CD1_2",
			  "CD2_1","CD2_2","PV2_3","ESO DRS NUMBRMS",
			  "ESO DRS STDCRMS","ESO DRS WCSRAOFF",
			  "ESO DRS WCSDECOFF"};
    int i,nkeys=13;
    cpl_propertylist *plist,*p2,*p3;

    /* Get the propertylist we need */

    plist = vircam_fits_get_ehu(ps.imgf);

    /* Extract the standard things */

    if ((p2 = vircam_paf_req_items(plist)) == NULL) {
	cpl_msg_error(fctid,"Unable to find required items in header");
	return(-1);
    }
    p3 = vircam_paf_phu_items(vircam_fits_get_phu(ps.imgf));
    vircam_merge_propertylists(p2,p3);
    freepropertylist(p3);

    /* Add in the extra stuff */

    for (i = 0; i < nkeys; i++) {
	cpl_propertylist_copy_property(p2,plist,keys[i]);
        if (cpl_error_get_code() != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"A header parameter %s is missing",keys[i]);
  	    cpl_propertylist_delete(p2);
 	    return(-1);
	}
    }

    /* Now write it all out */
    
    if (vircam_paf_print((char *)outfile,"VIRCAM/vircam_platesol",
			 "Test QC file",p2) != VIR_OK) {
	cpl_msg_error(fctid,"Error writing PAF\n");
	cpl_propertylist_delete(p2);
	return(-1);
    }

    /* Tidy and exit */

    cpl_propertylist_delete(p2);
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_platesol_init(void) {
    ps.labels = NULL;
    ps.img = NULL;
    ps.imgf = NULL;
    ps.mstds = NULL;
    ps.mstdsf = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_platesol_tidy(void) {
    freespace(ps.labels);
    freefits(ps.imgf);
    freetfits(ps.mstdsf);
    freeframe(ps.mstds);
    freeframe(ps.img);
}


/*

$Log: vircam_platesol.c,v $
Revision 1.13  2007/10/25 19:38:22  jim
modified to keep lint happy

Revision 1.12  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.11  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.10  2007/05/02 09:17:04  jim
uses new vircam_platesol api

Revision 1.9  2007/04/23 12:49:43  jim
Modified error condition behaviour

Revision 1.8  2007/04/13 12:27:39  jim
Added some extra docs

Revision 1.7  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.6  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.5  2007/02/15 12:17:45  jim
Modified to use new version of PAF files

Revision 1.4  2006/06/15 09:59:00  jim
Minor changes to docs

Revision 1.3  2006/05/04 11:53:45  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.2  2006/04/27 14:22:06  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:45  jim
New routine


*/




