/* $Id: vircam_mkconf.c,v 1.11 2007/10/15 12:53:55 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/15 12:53:55 $
 * $Revision: 1.11 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"

/* Function prototypes */

static int vircam_mkconf_create(cpl_plugin *) ;
static int vircam_mkconf_exec(cpl_plugin *) ;
static int vircam_mkconf_destroy(cpl_plugin *) ;
static int vircam_mkconf_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_mkconf_save(cpl_frameset *framelist,
			      cpl_parameterlist *parlist);
static void vircam_mkconf_init(void);
static void vircam_mkconf_tidy(void);

static struct {

    /* Input */

    int         extenum;

} vircam_mkconf_config;

static struct {
    int              *labels;
    cpl_frame        *flat;
    vir_mask         *bpm;
    vir_fits         *flatf;
    cpl_image        *outimg;
    cpl_propertylist *drs;
} ps;

static int isfirst;
static cpl_frame *product_frame = NULL;

static char vircam_mkconf_description[] =
"vircam_mkconf -- VIRCAM confidence map generation test recipe.\n\n"
"Create a confidence map using an input flat field and an input mask\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A input master flat field\n"
"    %-21s A master bad pixel map or\n"
"    %-21s A master confidence map\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_mkconf vircam_mkconf
    \brief Test recipe to drive the vircam_mkconf library function.

    \par Name:
        vircam_mkconf
    \par Purpose:
        Test recipe to drive the vircam_mkconf library function.
    \par Description:
        Test the library function vircam_mkconf creating a new confidence map
	from a given input master flat field and an input mask. The latter
	can be either a pre-existing confidence map or a bad pixel mask.
    \par Language:
        C
    \par Parameters:
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b MASTER_TWILIGHT_FLAT (required): A master twilight flat field.
        - \b MASTER_BPM or \b MASTER_CONF (required): Which ever is given
             will be used to mask out bad pixels.
    \par Output File Types:
        The following lists the output data frames that are generated by 
        this recipe. The word in bold gives the PRO CATG keyword value for 
        each.
        - \b CONFIDENCE_MAP_TEST: An output confidence map
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input flat or mask frames
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_mkconf.c
*/


/* Function code */


/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_mkconf_description,
		   VIRCAM_CAL_TWILIGHT_FLAT,VIRCAM_CAL_BPM,VIRCAM_CAL_CONF);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_mkconf",
                    "VIRCAM confidence map test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_mkconf_create,
                    vircam_mkconf_exec,
                    vircam_mkconf_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_mkconf_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_mkconf.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_mkconf",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mkconf_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_mkconf_test(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mkconf_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mkconf_test(cpl_parameterlist *parlist, 
			      cpl_frameset *framelist) {
    const char *fctid="vircam_mkconf";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j,nx,ny;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_mkconf_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_mkconf.extenum");
    vircam_mkconf_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_mkconf_tidy();
        return(-1);
    }

    /* Get the flat */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_mkconf_tidy();
        return(-1);
    }
    if ((ps.flat = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					      VIRCAM_CAL_TWILIGHT_FLAT)) == NULL) {
        cpl_msg_info(fctid,"No master flat found -- cannot continue");
	vircam_mkconf_tidy();
	return(-1);
    }

    /* Get the master mask */

    ps.bpm = vircam_mask_define(framelist,ps.labels,nlab);

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_mkconf_config.extenum,(const cpl_frame *)ps.flat,
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_mkconf_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
        isfirst = (j == jst);

        /* Load up the images */

        ps.flatf = vircam_fits_load(ps.flat,CPL_TYPE_FLOAT,j);
        if (ps.flatf == NULL) {
	    vircam_mkconf_tidy();
	    return(-1);
	}

        /* Get the size of the flat image */
        
	nx = cpl_image_get_size_x(vircam_fits_get_image(ps.flatf));
	ny = cpl_image_get_size_y(vircam_fits_get_image(ps.flatf));

        /* Load the data for the bpm */

        (void)vircam_mask_load(ps.bpm,j,nx,ny);

	/* Now do the correction */

	cpl_msg_info(fctid,"Making confidence map for extension %d",j);
        (void)vircam_mkconf(vircam_fits_get_image(ps.flatf),
			    vircam_fits_get_fullname(ps.flatf),ps.bpm,
			    &(ps.outimg),&(ps.drs),&status);
	if (status != VIR_OK) {
	    vircam_mkconf_tidy();
	    return(-1);
	}

        /* Now save the result */

	cpl_msg_info(fctid,"Saving results for extension %d",j);
	if (vircam_mkconf_save(framelist,parlist) != 0) {
	    vircam_mkconf_tidy();
	    return(-1);
	}

	/* Tidy a few things before the next image */

	freefits(ps.flatf);
	vircam_mask_clear(ps.bpm);
	freeimage(ps.outimg);
	freepropertylist(ps.drs);
    }
    vircam_mkconf_tidy();
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mkconf_save(cpl_frameset *framelist, 
			      cpl_parameterlist *parlist) {
    const char *recipeid = "vircam_mkconf";
    const char *fctid = "vircam_mkconf_save";
    const char *outfile = "mkconf.fits";
    cpl_propertylist *plist;

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_TEST);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

	/* Create product frame phu header */

        plist = vircam_fits_get_phu(ps.flatf);
        vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
                                              parlist,(char *)recipeid,
                                              "?Dictionary?");

        /* 'Save' the PHU image */

        if (cpl_image_save(NULL,outfile,CPL_BPP_8_UNSIGNED,plist,
                           CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product PHU");
            cpl_frame_delete(product_frame);
            return(-1);
        }
        cpl_frameset_insert(framelist,product_frame);
    }

    /* Get the extension property list */

    plist = vircam_fits_get_ehu(ps.flatf);

    /* Fiddle with the header now */

    vircam_dfs_set_product_exten_header(plist,product_frame,framelist,parlist,
					(char *)recipeid,"?Dictionary?");

    /* Save the image */

    if (cpl_image_save(ps.outimg,outfile,CPL_BPP_IEEE_FLOAT,plist,
		       CPL_IO_EXTEND) != CPL_ERROR_NONE) {
        cpl_msg_error(fctid,"Cannot save product image extension");
        return(-1);
    }

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_mkconf_init(void) {
    ps.labels = NULL;
    ps.flat = NULL;
    ps.bpm = NULL;
    ps.flatf = NULL;
    ps.outimg = NULL;
    ps.drs = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_mkconf_tidy(void) {
    freespace(ps.labels);
    freeframe(ps.flat);
    freemask(ps.bpm);
    freefits(ps.flatf);
    freeimage(ps.outimg);
    freepropertylist(ps.drs);
}

/**@}*/

/*

$Log: vircam_mkconf.c,v $
Revision 1.11  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.10  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.9  2007/04/23 12:49:34  jim
Fixed bug where the wrong image was being saved

Revision 1.8  2007/04/13 12:27:39  jim
Added some extra docs

Revision 1.7  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.6  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.5  2006/06/15 09:59:00  jim
Minor changes to docs

Revision 1.4  2006/05/04 11:53:44  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.3  2006/05/02 11:29:16  jim
Fixed problem where propertylist was being deleted incorrectly

Revision 1.2  2006/04/27 14:22:06  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:45  jim
New routine


*/




