/* $Id: vircam_imcore.c,v 1.15 2007/10/25 19:38:22 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 19:38:22 $
 * $Revision: 1.15 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <math.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_mods.h"
#include "vircam_fits.h"
#include "../vircam/catalogue/imcore.h"

/* Function prototypes */

static int vircam_imcore_create(cpl_plugin *) ;
static int vircam_imcore_exec(cpl_plugin *) ;
static int vircam_imcore_destroy(cpl_plugin *) ;
static int vircam_imcore_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_imcore_save(cpl_frameset *framelist,
			      cpl_parameterlist *parlist);
static void vircam_imcore_init(void);
static void vircam_imcore_tidy(void);

/* Static global variables */

static struct {

    /* Input */

    int         ipix;
    float       threshold;
    int         icrowd;
    float       rcore;
    float       filtfwhm;
    int         nbsize;
    int         cattype;
    int         extenum;

    /* Output */

    float       skylevel;
    float       skynoise;

} vircam_imcore_config ;

static struct {
    int              *labels;
    cpl_frame        *img;
    cpl_frame        *conf;
    vir_fits         *imgf;
    vir_fits         *conff;
    vir_tfits        *outcat;
} ps;

static int isfirst;
static cpl_frame *product_frame = NULL;

#define BUZZ_OFF {vircam_imcore_tidy(); return(-1);}

static char vircam_imcore_description[] =
"vircam_imcore -- VIRCAM catalogue generation recipe.\n\n"
"Extract objects from an image and write the catalogue to a FITS table.\n\n"
"If more than one file of each type is specified in the SOF only the \n"
"first will be used and the rest will be ingored\n"
"The program requires the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s An input image\n"
"    %-21s A master confidence maps\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_imcore vircam_imcore
    \brief Test recipe to drive the vircam_imcore library function.

    \par Name:
        vircam_imcore
    \par Purpose:
        Test recipe to drive the vircam_imcore library function.
    \par Description:
        Test the library function vircam_imcore by extracting objects from
        an input frame.
    \par Language:
        C
    \par Parameters:
        - \b ipix (int) The minimum allowable size of an object
        - \b thr (float) The detection threshold in sigma above sky
        - \b icrowd If set then the deblending software will be used
        - \b rcore The core radius in pixels
        - \b nb The smoothing box size for background map estimation
        - \b cattype The type of catalogue to be produced
	- \b filtfwhm The FWHM of the smoothing kernel used in the detection
	  algorithm
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b SCIENCE_IMAGE (required): A single raw image.
        - \b MASTER_CONF (optional): A master confidence map.
    \par Output File Types
        The following lists the output data frames that are generated by 
        this recipe. The word in bold gives the PRO CATG keyword value for 
        each.
        - \b OBJECT_CATALOGUE_TEST: A catalogue of objects extracted from 
          the input image
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input image 
    \par Non-Fatal Error Conditions:
        - Missing confidence map
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_imcore.c
*/

/* Function code */


/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_imcore_description,
                   VIRCAM_TEST_SCIENCE_RAW,VIRCAM_CAL_CONF);


    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_imcore",
                    "VIRCAM recipe to extract objects from a frame [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_imcore_create,
                    vircam_imcore_exec,
                    vircam_imcore_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcore_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Fill in the minimum object size */

    p = cpl_parameter_new_value("vircam.vircam_imcore.ipix",
                                CPL_TYPE_INT,
                                "Minimum pixel area for each detected object",
                                "vircam.vircam_imcore",5);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ipix");
    cpl_parameterlist_append(recipe->parameters,p);    

    /* Fill in the detection threshold parameter */

    p = cpl_parameter_new_value("vircam.vircam_imcore.thresh",
                                CPL_TYPE_DOUBLE,
                                "Detection threshold in sigma above sky",
                                "vircam.vircam_imcore",2.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"thr");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in flag to use deblending software or not */

    p = cpl_parameter_new_value("vircam.vircam_imcore.icrowd",
				CPL_TYPE_BOOL,"Use deblending?",
				"vircam.vircam_imcore",0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"icrowd");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in core radius */

    p = cpl_parameter_new_value("vircam.vircam_imcore.rcore",
				CPL_TYPE_DOUBLE,"Value of Rcore in pixels",
				"vircam.vircam_imcore",4.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"rcore");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in background smoothing box size */

    p = cpl_parameter_new_value("vircam.vircam_imcore.nbsize",
				CPL_TYPE_INT,"Background smoothing box size",
				"vircam.vircam_imcore",64);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"nb");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in detection algorithm smoothing kernel FWHM */

    p = cpl_parameter_new_value("vircam.vircam_imcore.filtfwhm",
				CPL_TYPE_DOUBLE,
				"FWHM of smoothing kernel in pixels",
				"vircam.vircam_imcore",2.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"filtfwhm");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in catalogue type */

    p = cpl_parameter_new_enum("vircam.vircam_imcore.cattype",
			       CPL_TYPE_INT,"Catalogue type",
			       "vircam.vircam_imcore",2,4,1,2,3,4);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"cattype");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_imcore.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_imcore",
                                1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcore_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcore_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_imcore_test(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcore_test(cpl_parameterlist *parlist, 
			      cpl_frameset *framelist) {
    const char *fctid="vircam_imcore";
    cpl_parameter *p;
    int nlab,j,jst,jfn,retval,ipix,icrowd,nbsize,mcattype,status;
    float thresh,rcore,filtfwhm;

    /* Initialise a few things */

    vircam_imcore_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.ipix");
    vircam_imcore_config.ipix = cpl_parameter_get_int(p);
    ipix = vircam_imcore_config.ipix;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.thresh");
    vircam_imcore_config.threshold = (float)cpl_parameter_get_double(p);
    thresh = vircam_imcore_config.threshold;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.icrowd");
    vircam_imcore_config.icrowd = cpl_parameter_get_bool(p);
    icrowd = vircam_imcore_config.icrowd;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.rcore");
    vircam_imcore_config.rcore = (float)cpl_parameter_get_double(p);
    rcore = vircam_imcore_config.rcore;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.filtfwhm");
    vircam_imcore_config.filtfwhm = (float)cpl_parameter_get_double(p);
    filtfwhm = vircam_imcore_config.filtfwhm;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.nbsize");
    vircam_imcore_config.nbsize = cpl_parameter_get_int(p);
    nbsize = vircam_imcore_config.nbsize;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.cattype");
    vircam_imcore_config.cattype = cpl_parameter_get_int(p);
    mcattype = vircam_imcore_config.cattype;
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcore.extenum");
    vircam_imcore_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_imcore_tidy();
        return(-1);
    }

    /* Get the images and confidence maps */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
					   &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frameset");
	BUZZ_OFF
    }
    if ((ps.img = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					   VIRCAM_TEST_SCIENCE_RAW)) == NULL) {
	cpl_msg_error(fctid,"Cannot find any image frames in input frameset");
	BUZZ_OFF
    } 
    if ((ps.conf = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					    VIRCAM_CAL_CONF)) == NULL) {
	cpl_msg_info(fctid,"No confidence map specified. Proceding without one");
	ps.conf = NULL;
    }

    /* Loop for each of the image extensions */

    vircam_exten_range(vircam_imcore_config.extenum,(const cpl_frame *)ps.img,
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_imcore_tidy();
	return(-1);
    }
    for (j = jst; j <= jfn; j++) {
	cpl_msg_info(fctid,"Processing extension %d\n",j);
        status = VIR_OK;
	isfirst = (j == jst);

        /* Load up the images */

        ps.imgf = vircam_fits_load(ps.img,CPL_TYPE_FLOAT,j);
        if (ps.imgf == NULL) {
            vircam_imcore_tidy();
            return(-1);
        }
	if (ps.conf != NULL) 
            ps.conff = vircam_fits_load(ps.conf,CPL_TYPE_INT,j);

        /* Call the imcore routine */

        (void)vircam_imcore(ps.imgf,ps.conff,ipix,thresh,icrowd,rcore,
			    nbsize,mcattype,filtfwhm,&(ps.outcat),&status);
        if (status != VIR_OK) {
	    cpl_msg_error(fctid,"Error extracting objects in extension %d",j);
	    cpl_error_reset();
	    ps.outcat = vircam_tfits_wrap(vircam_dummy_catalogue(mcattype),
					  NULL,
					  vircam_fits_get_phu(ps.imgf),
					  vircam_fits_get_ehu(ps.imgf));
	}
  
        /* Save the products */

        retval = vircam_imcore_save(framelist,parlist);
	if (retval != 0) {
	    cpl_msg_error(fctid,"Error saving products in extension %d",j);
	    BUZZ_OFF
	}

	/* Clean up a bit */

	freetfits(ps.outcat);
	freefits(ps.imgf);
	freefits(ps.conff);
    }
    vircam_imcore_tidy();
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here 
  @param    framelist    The input framelist
  @param    parlist      The input parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcore_save(cpl_frameset *framelist,
			      cpl_parameterlist *parlist) {
    const char *recipeid = "vircam_imcore";
    const char *fctid = "vircam_imcore_save";
    const char *outfile = "imcoretab.fits";
    cpl_propertylist *elist,*plist;

    /* Create the output table. First see if you need a primary */

    if (isfirst) {
 
        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_OBJCAT_TEST);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_TABLE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

        /* Fiddle with the primary header now */

        plist = vircam_tfits_get_phu(ps.outcat);
        vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
					      parlist,(char *)recipeid,
					      "?Dictionary?");

        /* Get the extension header and tack the extra header items onto it. */

        elist = vircam_tfits_get_ehu(ps.outcat);
        vircam_dfs_set_product_exten_header(elist,product_frame,framelist,
					    parlist,(char *)recipeid,
					    "?Dictionary?");

        /* 'Save' the PHU and create a table extension */

        if (cpl_table_save(vircam_tfits_get_table(ps.outcat),plist,elist,
			   outfile,CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product table");
	    cpl_frame_delete(product_frame);
	    return(-1);
	}
        cpl_frameset_insert(framelist,product_frame);

    /* Otherwise save the next extension */

    } else {

        /* Get the extension header and tack the extra header items onto it. */

        elist = vircam_tfits_get_ehu(ps.outcat);
        vircam_dfs_set_product_exten_header(elist,product_frame,framelist,
					    parlist,(char *)recipeid,
					    "?Dictionary?");

	/* Save the table */

        if (cpl_table_save(vircam_tfits_get_table(ps.outcat),NULL,elist,
			   outfile,CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product table");
	    return(-1);
	}
    }

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_imcore_init(void) {
    ps.labels = NULL;
    ps.img = NULL;
    ps.conf = NULL;
    ps.imgf = NULL;
    ps.conff = NULL;
    ps.outcat = NULL;
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_imcore_tidy(void) {
    freespace(ps.labels);
    freeframe(ps.img);
    freeframe(ps.conf);
    freefits(ps.imgf);
    freefits(ps.conff);
    freetfits(ps.outcat);
}

/**@}*/


/*

$Log: vircam_imcore.c,v $
Revision 1.15  2007/10/25 19:38:22  jim
modified to keep lint happy

Revision 1.14  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.13  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.12  2007/05/02 09:16:54  jim
uses new vircam_imcore api

Revision 1.11  2007/04/13 12:27:38  jim
Added some extra docs

Revision 1.10  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.9  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.8  2006/08/01 11:34:17  jim
uses newer version of the imcore software

Revision 1.7  2006/06/15 09:58:59  jim
Minor changes to docs

Revision 1.6  2006/05/18 12:31:50  jim
Fixed bug where raw and calib frames were not being classified

Revision 1.5  2006/05/18 09:49:13  jim
Fixed error in online doc

Revision 1.4  2006/05/04 11:53:41  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.3  2006/04/28 08:55:26  jim
Modified to allow both types of confidence maps

Revision 1.2  2006/04/27 14:22:04  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:44  jim
New routine


*/
