/* $Id: vircam_imcombine.c,v 1.13 2007/10/25 19:38:22 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 19:38:22 $
 * $Revision: 1.13 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>
#include <math.h>

#include "vircam_utils.h"
#include "vircam_mask.h"
#include "vircam_dfs.h"
#include "vircam_mods.h"
#include "vircam_fits.h"

/* Function prototypes */

static int vircam_imcombine_create(cpl_plugin *) ;
static int vircam_imcombine_exec(cpl_plugin *) ;
static int vircam_imcombine_destroy(cpl_plugin *) ;
static int vircam_imcombine_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_imcombine_save(cpl_frameset *framelist, 
				 cpl_parameterlist *parlist);
static void vircam_imcombine_init(void);
static void vircam_imcombine_tidy(void);

/* Static global variables */

static struct {

    /* Input */

    int         combtype;
    int         scaletype;
    int         xrej;
    float       thresh;
    int         extenum;

} vircam_imcombine_config;


static struct {
    int              *labels;
    cpl_frameset     *imagelist;
    vir_fits         **images;
    int              nimages;
    cpl_image        *outimage;
} ps;

static int isfirst;
static cpl_frame *product_frame = NULL;

static char vircam_imcombine_description[] =
"vircam_imcombine -- VIRCAM test imcombine recipe.\n\n"
"Combine a list of frames into a mean frame.\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A list of images\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_imcombine vircam_imcombine
    \brief Combine a series of exposures into a single mean frame

    \par Name: 
        vircam_imcombine
    \par Purpose: 
        Combine a series of exposures into a single mean frame
    \par Description: 
        A load of images are combined into a mean frame.
    \par Language:
        C
    \par Parameters:
        - \b comb (int): Determines the type of combination that is done to
          form the output map. Can take the following values:
            - (1): The output pixels are medians of the input pixels
            - (2): The output pixels are means of the input pixels
        - \b scale (int): Determines how the input data are scaled or 
	  offset before they are combined. Can take the following values:
            - (0): No scaling of offsetting
            - (1): All input frames are biassed additively to bring their
	      backgrounds to a common mean.
            - (2): All input frames are scaled multiplicatively to bring their
              backgrounds to a common mean.
            - (3): All input frames are scaled to a uniform exposure time and
              then additively corrected to bring their backgrounds to a common 
	      mean.
        - \b xrej (int): If set, then an extra rejection cycle will be run. 
        - \b thr (float): The rejection threshold in numbers of sigmas.
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then 
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b SCIENCE_IMAGE (required): A list of images for combining
    \par Output File Types:
        The following lists the output data frames that are generated by 
        this recipe. The word in bold gives the PRO CATG keyword value for 
        each.
        - \b SIMPLE_IMAGE_TEST: An output combined image
    \par Notes
        None.
    \par Fatal Error Conditions:
        - No input images
    \par Non-Fatal Error Conditions:
        None
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_imcombine.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_imcombine_description,
		   VIRCAM_TEST_SCIENCE_RAW);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_imcombine",
                    "VIRCAM test image combination recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_imcombine_create,
                    vircam_imcombine_exec,
                    vircam_imcombine_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcombine_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Fill in the parameters. First the combination type */

    p = cpl_parameter_new_range("vircam.vircam_imcombine.combtype",
			        CPL_TYPE_INT,
			        "1 == Median,\n 2 == Mean",
			        "vircam.vircam_imcombine",
			        2,1,2);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"comb");
    cpl_parameterlist_append(recipe->parameters,p);

    /* The requested scaling */

    p = cpl_parameter_new_range("vircam.vircam_imcombine.scaletype",
			        CPL_TYPE_INT,
			        "0 == none,\n 1 == additive offset,\n 2 == multiplicative offset,\n 3 == exposure time scaling + additive offset",
			        "vircam.vircam_imcombine",
			        1,0,3);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"scale");
    cpl_parameterlist_append(recipe->parameters,p);
    
    /* Extra rejection cycle */

    p = cpl_parameter_new_value("vircam.vircam_imcombine.xrej",
				CPL_TYPE_BOOL,
				"True if using extra rejection cycle",
				"vircam.vircam_imcombine",
				TRUE);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"xrej");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Rejection threshold */

    p = cpl_parameter_new_value("vircam.vircam_imcombine.thresh",
				CPL_TYPE_DOUBLE,
				"Rejection threshold in sigma above background",
				"vircam.vircam_imcombine",5.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"thr");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_imcombine.extenum",
			        CPL_TYPE_INT,
			        "Extension number to be done, 0 == all",
			        "vircam.vircam_imcombine",
			        1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);
        
    /* Get out of here */

    return(0);
}
    
    
/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcombine_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    return(vircam_imcombine_test(recipe->parameters,recipe->frames));
}
				
/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcombine_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here 
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcombine_test(cpl_parameterlist *parlist, 
			         cpl_frameset *framelist) {
    const char *fctid="vircam_imcombine";
    int nlab,j,jst,jfn,retval,status;
    cpl_parameter *p;
    unsigned char *rejmask,*rejplus;
    cpl_propertylist *drs;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
	cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
	return(-1);
    }

    /* Initialise some things */

    vircam_imcombine_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcombine.combtype");
    vircam_imcombine_config.combtype = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcombine.scaletype");
    vircam_imcombine_config.scaletype = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcombine.xrej");
    vircam_imcombine_config.xrej = cpl_parameter_get_bool(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcombine.thresh");
    vircam_imcombine_config.thresh = (float)cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_imcombine.extenum");
    vircam_imcombine_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
	cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
	vircam_imcombine_tidy();
	return(-1);
    }

    /* Get the frames frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
					   &nlab)) == NULL) {
	cpl_msg_error(fctid,"Cannot labelise the input frames");
	vircam_imcombine_tidy();
	return(-1);
    }
    if ((ps.imagelist = vircam_frameset_subgroup(framelist,ps.labels,nlab,
						VIRCAM_TEST_SCIENCE_RAW)) == NULL) {
	cpl_msg_error(fctid,"Cannot find images in input frameset");
	vircam_imcombine_tidy();
        return(-1);
    }
    ps.nimages = cpl_frameset_get_size(ps.imagelist);

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_imcombine_config.extenum,
		       (const cpl_frame *)cpl_frameset_get_frame(ps.imagelist,0),
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_imcombine_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
	isfirst = (j == jst);

	/* Load the images */

        ps.images = vircam_fits_load_list(ps.imagelist,CPL_TYPE_FLOAT,j);

	/* Call the combine module */

	cpl_msg_info(fctid,"Doing combination for extension %d\n",j);
	(void)vircam_imcombine(ps.images,ps.nimages,
			       vircam_imcombine_config.combtype,
			       vircam_imcombine_config.scaletype,
			       vircam_imcombine_config.xrej,
			       vircam_imcombine_config.thresh,
			       &(ps.outimage),&rejmask,&rejplus,&drs,&status);
	freespace(rejmask);
	freespace(rejplus);
	freepropertylist(drs);
	if (status != VIR_OK) {
	    vircam_imcombine_tidy();
	    return(-1);
	}

	/* Save everything */

	cpl_msg_info(fctid,"Saving combined image extension %d\n",j);
	retval = vircam_imcombine_save(framelist,parlist);
	if (retval != 0) {
	    vircam_imcombine_tidy();
	    return(-1);
	}
	freefitslist(ps.images,ps.nimages);
	freeimage(ps.outimage);
    }
    vircam_imcombine_tidy();
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_imcombine_save(cpl_frameset *framelist, 
				 cpl_parameterlist *parlist) {
    cpl_propertylist *plist;
    const char *recipeid = "vircam_imcombine";
    const char *fctid = "vircam_imcombine_save";
    const char *outfile = "comb.fits";

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

	/* Create a new product frame object and define some tags */

	product_frame = cpl_frame_new();
	cpl_frame_set_filename(product_frame,outfile);
	cpl_frame_set_tag(product_frame,VIRCAM_PRO_SIMPLE_TEST);
	cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

	/* Set up the phu header */

	plist = vircam_fits_get_phu(ps.images[0]);
	vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
					      parlist,(char *)recipeid,
					      "?Dictionary?");
        /* 'Save' the PHU image */			 

        if (cpl_image_save(NULL,outfile,CPL_BPP_8_UNSIGNED,plist,
			   CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product PHU");
	    cpl_frame_delete(product_frame);
	    return(-1);
	}
        cpl_frameset_insert(framelist,product_frame);
    }

    /* Get the extension property list */

    plist = vircam_fits_get_ehu(ps.images[0]);

    /* Fiddle with the header now */

    vircam_dfs_set_product_exten_header(plist,product_frame,framelist,
					parlist,(char *)recipeid,
					"?Dictionary?");
		
    /* Now save the mean dome flat image extension */

    if (cpl_image_save(ps.outimage,outfile,CPL_BPP_IEEE_FLOAT,plist,
		       CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(fctid,"Cannot save product image extension");
	return(-1);
    }

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_imcombine_init(void) {
    ps.labels = NULL;
    ps.imagelist = NULL;
    ps.images = NULL;
    ps.outimage = NULL;
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_imcombine_tidy(void) {
    freespace(ps.labels);
    freeframeset(ps.imagelist);
    freefitslist(ps.images,ps.nimages);
    freeimage(ps.outimage);
}

/**@}*/

/*

$Log: vircam_imcombine.c,v $
Revision 1.13  2007/10/25 19:38:22  jim
modified to keep lint happy

Revision 1.12  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.11  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.10  2007/05/02 12:53:11  jim
typo fixes in docs

Revision 1.9  2007/04/13 12:27:38  jim
Added some extra docs

Revision 1.8  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.7  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.6  2006/06/15 09:58:59  jim
Minor changes to docs

Revision 1.5  2006/05/04 11:53:40  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.4  2006/05/02 11:29:14  jim
Fixed problem where propertylist was being deleted incorrectly

Revision 1.3  2006/04/28 08:51:00  jim
Removed redundant parameter ncells

Revision 1.2  2006/04/27 14:22:04  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:44  jim
New routine


*/
