/* $Id: vircam_getstds.c,v 1.14 2007/10/25 18:39:22 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 18:39:22 $
 * $Revision: 1.14 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"

/* Function prototypes */

static int vircam_getstds_create(cpl_plugin *) ;
static int vircam_getstds_exec(cpl_plugin *) ;
static int vircam_getstds_destroy(cpl_plugin *) ;
static int vircam_getstds_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_getstds_save(cpl_frameset *framelist, 
			       cpl_parameterlist *parlist);
static void vircam_getstds_init(void);
static void vircam_getstds_tidy(void);

static struct {

    /* Input */
    
    int         extenum;

} vircam_getstds_config;

static struct {
    int              *labels;
    cpl_frame        *img;
    vir_fits         *imgf;
    cpl_propertylist *plist;
    cpl_table        *stds;
    char             *catpath;
    char             *catname;
} ps;

static int isfirst;
static cpl_frame *product_frame = NULL;

static char vircam_getstds_description[] =
"vircam_getstds -- VIRCAM test recipe to get standard stars for a frame\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A input image\n"
"    %-21s Standard catalogue index file\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_getstds vircam_getstds
    \brief Test recipe to drive the vircam_getstds library function.

    \par Name:
        vircam_getstds
    \par Purpose:
        Test recipe to drive the vircam_getstds library function.
    \par Description:
        Test the library function vircam_getstds by examining the WCS of
	an input image. Extract all the stars from the 2MASS point source
	catalogue that should appear on the input image.
    \par Language:
        C
    \par Parameters:
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b SCIENCE_IMAGE (required): An input image.
	- \b MASTER_2MASS_CATALOGUE (required): The index table for the 2mass 
	     catalogue
    \par Output File Types:
        The following lists the output data frames that are generated by 
        this recipe. The word in bold gives the PRO CATG keyword value for 
        each.
        - \b STANDARDS_TABLE: A FITS table with standard objects
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input image
	- Unable to locate the 2MASS catalogue
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_getstds.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_getstds_description,
                   VIRCAM_TEST_SCIENCE_RAW,VIRCAM_CAL_2MASS);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_getstds",
                    "VIRCAM standard star extraction test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_getstds_create,
                    vircam_getstds_exec,
                    vircam_getstds_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_getstds_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Get the extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_getstds.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_getstds",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_getstds_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_getstds_test(recipe->parameters,recipe->frames));
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_getstds_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_getstds_test(cpl_parameterlist *parlist, 
			       cpl_frameset *framelist) {
    const char *fctid="vircam_getstds";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j;
    cpl_frame *catindex;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_getstds_init();

    /* Get parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_getstds.extenum");
    vircam_getstds_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_getstds_tidy();
        return(-1);
    }

    /* Get the frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_getstds_tidy();
        return(-1);
    }
    if ((ps.img = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
                                             VIRCAM_TEST_SCIENCE_RAW)) == NULL) {
        cpl_msg_info(fctid,"No raw image found -- cannot continue");
	vircam_getstds_tidy();
	return(-1);
    }
    if ((catindex = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					       VIRCAM_CAL_2MASS)) == NULL) {
        cpl_msg_info(fctid,"No 2MASS index found -- cannot continue");
	vircam_getstds_tidy();
	return(-1);
    }
    
    /* Get catalogue parameters */

    vircam_catpars(catindex,&(ps.catpath),&(ps.catname));
    cpl_frame_delete(catindex);

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_getstds_config.extenum,(const cpl_frame *)ps.img,
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_getstds_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
        isfirst = (j == jst);

        /* Load up the images */

        ps.imgf = vircam_fits_load(ps.img,CPL_TYPE_FLOAT,j);
        if (ps.img == NULL) {
	    vircam_getstds_tidy();
	    return(-1);
	}

	/* Now do the correction */

	cpl_msg_info(fctid,"Extracting the standards");
        (void)vircam_getstds(vircam_fits_get_ehu(ps.imgf),1,ps.catpath,
			     ps.catname,&(ps.stds),&status);
	if (status != VIR_OK) {
	    vircam_getstds_tidy();
	    return(-1);
	}

        /* Now save the result */

	cpl_msg_info(fctid,"Saving results for extension %d",j);
	if (vircam_getstds_save(framelist,parlist) != 0) {
	    vircam_getstds_tidy();
	    return(-1);
	}

	/* Tidy a few things before the next image */

	freetable(ps.stds);
	freefits(ps.imgf);
    }
    vircam_getstds_tidy();
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_getstds_save(cpl_frameset *framelist,
			       cpl_parameterlist *parlist) {
    const char *fctid = "vircam_getstds_save";
    const char *outfile = "getstds.fits";
    const char *recipeid = "vircam_getstds";
    cpl_propertylist *plist,*elist;

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_STDTAB);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_TABLE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

	/* Set up product frame phu */

        plist = vircam_fits_get_phu(ps.imgf);
        vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
                                              parlist,(char *)recipeid,
                                              "?Dictionary?");

        /* Now fiddle with the extension header */

	elist = vircam_fits_get_ehu(ps.imgf);
        vircam_dfs_set_product_exten_header(elist,product_frame,framelist,
					    parlist,(char *)recipeid,
					    "?Dictionary?");

        /* 'Save' the table */

        if (cpl_table_save(ps.stds,plist,elist,outfile,CPL_IO_DEFAULT) 
	    != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product");
            cpl_frame_delete(product_frame);
            return(-1);
        }
        cpl_frameset_insert(framelist,product_frame);

    /* Subsequent extensions...*/

    } else {

        /* Now fiddle with the extension header */

	elist = vircam_fits_get_ehu(ps.imgf);
        vircam_dfs_set_product_exten_header(elist,product_frame,framelist,
					    parlist,(char *)recipeid,
					    "?Dictionary?");

        if (cpl_table_save(ps.stds,NULL,elist,outfile,CPL_IO_EXTEND)
			   != CPL_ERROR_NONE) {
  	    cpl_msg_error(fctid,"Cannot save product table extension");
 	    return(-1);
        }	
    }

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_getstds_init(void) {
    ps.img = NULL;
    ps.imgf = NULL;
    ps.labels = NULL;
    ps.plist = NULL;
    ps.stds = NULL;
    ps.catname = NULL;
    ps.catpath = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_getstds_tidy(void) {
    freeframe(ps.img);
    freefits(ps.imgf);
    freespace(ps.labels);
    freepropertylist(ps.plist);
    freetable(ps.stds);
    freespace(ps.catname);
    freespace(ps.catpath);
}

/**@}*/

/*

$Log: vircam_getstds.c,v $
Revision 1.14  2007/10/25 18:39:22  jim
Altered to remove some lint messages

Revision 1.13  2007/10/19 06:55:06  jim
Modifications made to use new method for directing the recipes to the
standard catalogues using the sof

Revision 1.12  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.11  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.10  2007/04/23 12:47:54  jim
Changed default location for 2mass catalogue

Revision 1.9  2007/04/13 12:27:38  jim
Added some extra docs

Revision 1.8  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.7  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.6  2006/11/27 12:11:10  jim
Modified to add the catname parameter

Revision 1.5  2006/06/15 09:58:59  jim
Minor changes to docs

Revision 1.4  2006/05/04 11:53:40  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.3  2006/05/02 11:29:13  jim
Fixed problem where propertylist was being deleted incorrectly

Revision 1.2  2006/04/27 14:22:04  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:44  jim
New routine


*/





