/* $Id: vircam_mesostep_analyse.c,v 1.18 2008/05/06 12:15:20 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2008/05/06 12:15:20 $
 * $Revision: 1.18 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_mask.h"
#include "vircam_pfits.h"
#include "vircam_dfs.h"
#include "vircam_mods.h"
#include "vircam_fits.h"
#include "vircam_tfits.h"
#include "vircam_jmp_utils.h"

/* Function prototypes */

static int vircam_mesostep_analyse_create(cpl_plugin *);
static int vircam_mesostep_analyse_exec(cpl_plugin *);
static int vircam_mesostep_analyse_destroy(cpl_plugin *);
static int vircam_mesostep_analyse(cpl_parameterlist *, cpl_frameset *);
static cpl_propertylist *vircam_mesostep_analyse_dummyqc(int type);


static char vircam_mesostep_analyse_description[] =
"vircam_mesostep_analyse -- VIRCAM mesostep processing recipe.\n\n"
"Process a complete mesostep sequence of vircam data. Remove instrumental\n"
"signature and sky subtract if desired. Work out the illumination correction\n"
"for all of the input frames and then smooth the result by fitting a 2d\n"
"polynomial. Evaluate the polynomial at the grid points to form the final\n"
"illumination correction data product\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A list of raw science images\n"
"    %-21s A master dark frame\n"
"    %-21s A master twilight flat frame\n"
"    %-21s A channel table\n"
"    %-21s A photometric calibration table\n"
"    %-21s A master confidence map or\n"
"    %-21s A master bad pixel mask\n"
"    %-21s A master standard star index\n"
"All of the above are required\n"
"\n";

/**@{*/

/**
    \ingroup recipelist
    \defgroup vircam_mesostep_analyse vircam_mesostep_analyse
    \brief Reduce a complete mesostep sequence 

    \par Name: 
        vircam_mesostep_analyse
    \par Purpose: 
        Reduce a complete mesostep sequence
    \par Description: 
        A mesostep sequence in a single filter is corrected for instrumental 
	signature and sky background if requested. The field is divided into
	cells and an illumination correction is calculated for each one from
	all the stars on all images in that cell. The result is smoothed by
	fitting a 2d polynomial to the result. The evaluation of that 
	polynomial at the grid points forms the output illumination correction
	data product.
    \par Language:
        C
    \par Parameters:
        - \b ipix (int): The minimum allowable size of an object
        - \b thr (float): The detection threshold in sigma above sky
        - \b icrowd (int): If set then the deblending software will be used
        - \b rcore (float): The core radius in pixels
        - \b nb (int): The smoothing box size for background map estimation
        - \b destripe (int): If set, then the single images will be destriped
        - \b skycor (int): If set, a simple sky background correction will be 
	     applied.
        - \b nord (int): The order of the polynomial to fit to the data
        - \b ext (int): The image extension of the input files to be done
             on this run. If all of the extensions are to be processed, then 
             this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b MASTER_DARK (required): A master dark frame.
        - \b MASTER_TWILIGHT_FLAT (required): A master twilight flat frame.
        - \b MASTER_CONF or \b MASTER_BPM (required): A master confidence map 
             for the filter used in the science images. If such is not 
             available, then there is the option to use a master bad pixel 
             mask.
        - \b MASTER_CHANNEL_TABLE (required): The channel table.
        - \b STD_ILLUMINATION (required): The list of input science images.
        - \b PHOTCAL_TAB (required): A photometric calibration table
	- \b MASTER_2MASS_CATALOGUE: The index file for the 2mass catalogue
    \par Output Products:
        The following list gives the output data products that are generated
	by this recipe. The word in bold gives the DPR CATG keyword value for
	each product:
        - An illumination correction table defined from the corrected simple
	  images (\b ILLCOR_TAB_MES)
    \par Output QC Parameters:
        None
    \par Notes
        None.
    \par Fatal Error Conditions:
        - NULL input frameset
        - Input frameset headers incorrect meaning that RAW and CALIB frame
          can't be distinguished
        - No science frames in the input frameset
        - Required master calibration images and tables are missing
        - Unable to save data products
    \par Non-Fatal Error Conditions:
        - None
    \par Conditions Leading To Dummy Products:
        - Master calibration images either won't load or are flagged as dummy
        - The detector for the current image extension is flagged dead in
	  all science frames
        - Various processing routines fail.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference: 
        vircam_mesostep_analyse.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_mesostep_analyse_description,
		   VIRCAM_ILLUM_RAW,VIRCAM_CAL_DARK,VIRCAM_CAL_TWILIGHT_FLAT,
		   VIRCAM_CAL_CHANTAB,VIRCAM_CAL_PHOTTAB,VIRCAM_CAL_CONF,
		   VIRCAM_CAL_BPM,VIRCAM_CAL_2MASS);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_mesostep_analyse",
                    "VIRCAM mesostep analysis recipe",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_mesostep_analyse_create,
                    vircam_mesostep_analyse_exec,
                    vircam_mesostep_analyse_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_mesostep_analyse_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Fill in the minimum object size */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.ipix",
                                CPL_TYPE_INT,
                                "Minimum pixel area for each detected object",
                                "vircam.vircam_mesostep_analyse",5);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ipix");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in the detection threshold parameter */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.thresh",
                                CPL_TYPE_DOUBLE,
                                "Detection threshold in sigma above sky",
                                "vircam.vircam_mesostep_analyse",2.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"thr");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in flag to use deblending software or not */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.icrowd",
                                CPL_TYPE_BOOL,"Use deblending?",
                                "vircam.vircam_mesostep_analyse",0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"icrowd");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in core radius */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.rcore",
                                CPL_TYPE_DOUBLE,"Value of Rcore in pixels",
                                "vircam.vircam_mesostep_analyse",4.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"rcore");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in background smoothing box size */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.nbsize",
                                CPL_TYPE_INT,"Background smoothing box size",
                                "vircam.vircam_mesostep_analyse",64);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"nb");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in flag to destripe the images */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.destripe",
                                CPL_TYPE_BOOL,"Destripe images?",
                                "vircam.vircam_mesostep_analyse",1);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"destripe");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Fill in flag to correct sky background */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.skycor",
                                CPL_TYPE_BOOL,"Sky correct images?",
                                "vircam.vircam_mesostep_analyse",1);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"skycor");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Order for surface fit */

    p = cpl_parameter_new_value("vircam.vircam_mesostep_analyse.nord",
                                CPL_TYPE_INT,
				"Polynomial order for surface fit",
                                "vircam.vircam_mesostep_analyse",3);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"nord");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_mesostep_analyse.extenum",
			        CPL_TYPE_INT,
			        "Extension number to be done, 0 == all",
			        "vircam.vircam_mesostep_analyse",
			        1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);
        
    /* Get out of here */

    return(0);
}
        
/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mesostep_analyse_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    return(vircam_mesostep_analyse(recipe->parameters,recipe->frames));
}
				
/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mesostep_analyse_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here 
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_mesostep_analyse(cpl_parameterlist *parlist, 
				   cpl_frameset *framelist) {
    const char *fctid="vircam_mesostep_analyse";
    cpl_parameter *p;
    cpl_polynomial *poly;
    int nlab,jst,jfn,status,isconf,j,i,live,nrows,n,ndit;
    float *x1,*x2,*y1,*y2,*ill,gaincor_fac;
    double *bv_x,*bv_y,*vdata,val;
    cpl_bivector *bv;
    cpl_vector *v;
    vir_fits *ff;
    cpl_table *ic;
    cpl_frame *catindex;
    cpl_propertylist *pp;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
	cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
	return(-1);
    }

    /* Initialise some things */

    vircam_jmp_init();
    (void)strncpy(vircam_recipename,fctid,VIRCAM_PATHSZ);
    (void)snprintf(vircam_recipepaf,VIRCAM_PATHSZ,"VIRCAM/%s",fctid);
    recflag = RECMES;

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.ipix");
    vircam_jmp_config.ipix = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.thresh");
    vircam_jmp_config.threshold = (float)cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.icrowd");
    vircam_jmp_config.icrowd = cpl_parameter_get_bool(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.rcore");
    vircam_jmp_config.rcore = (float)cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.nbsize");
    vircam_jmp_config.nbsize = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.destripe");
    vircam_jmp_config.destripe = cpl_parameter_get_bool(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.skycor");
    vircam_jmp_config.skycor = cpl_parameter_get_bool(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.nord");
    vircam_jmp_config.nord = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,
			       "vircam.vircam_mesostep_analyse.extenum");
    vircam_jmp_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
	cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
	vircam_jmp_tidy(0);
	return(-1);
    }

    /* Label the input frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
					   &nlab)) == NULL) {
	cpl_msg_error(fctid,"Cannot labelise the input frames");
	vircam_jmp_tidy(0);
	return(-1);
    }

    /* Get the input science frames */

    if ((ps.science_frames = 
	 vircam_frameset_subgroup(framelist,ps.labels,nlab,
				  VIRCAM_ILLUM_RAW)) == NULL) {
	cpl_msg_error(fctid,"No science images to process!");
	vircam_jmp_tidy(0);
	return(-1);
    }

    /* Check to see if there is a master dark frame */

    if ((ps.master_dark = 
	 vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
				    VIRCAM_CAL_DARK)) == NULL) {
        cpl_msg_error(fctid,"No master dark found");
	vircam_jmp_tidy(0);
	return(-1);
    }
	
    /* Check to see if there is a master twilight flat frame */

    if ((ps.master_twilight_flat = 
	 vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
				    VIRCAM_CAL_TWILIGHT_FLAT)) == NULL) {
        cpl_msg_error(fctid,"No master twilight flat found");
	vircam_jmp_tidy(0);
	return(-1);
    }
	
    /* Get the gain corrections */

    status = VIR_OK;
    if (vircam_gaincor_calc(ps.master_twilight_flat,&i,&(ps.gaincors),
			    &status) != VIR_OK) {
	cpl_msg_error(fctid,"Error calculating gain corrections");
	vircam_jmp_tidy(0);
	return(-1);
    }
	
    /* Check to see if there is a master confidence map. If there isn't
       then look for a bad pixel mask that can be converted into a 
       confidence map (in an emergency) */

    isconf = 1;
    if ((ps.master_conf = 
	 vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
				    VIRCAM_CAL_CONF)) == NULL) {
	isconf = 0;
        if ((ps.master_conf = 
	     vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					VIRCAM_CAL_BPM)) == NULL) {
            cpl_msg_error(fctid,"No master confidence map found");
	    vircam_jmp_tidy(0);
	    return(-1);
	}
    }
    ps.mask = vircam_mask_define(framelist,ps.labels,nlab);
	
    /* Check to see if there is a channel table */

    if ((ps.chantab = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
						 VIRCAM_CAL_CHANTAB)) == NULL) {
        cpl_msg_error(fctid,"No channel table found");
	vircam_jmp_tidy(0);
	return(-1);
    }

    /* Check to see if there is a photometric table */

    if ((ps.phottab = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
						 VIRCAM_CAL_PHOTTAB)) == NULL) {
        cpl_msg_error(fctid,"No photometric table found");
	vircam_jmp_tidy(0);
	return(-1);
    }
    if ((ps.tphottab = cpl_table_load(cpl_frame_get_filename(ps.phottab),1,0)) == NULL) {
	cpl_msg_error(fctid,"Unable to load photometric table");
	vircam_jmp_tidy(0);
	return(-1);
    }

    /* Is the 2mass index file specified? */

    if ((catindex = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
                                               VIRCAM_CAL_2MASS)) == NULL) {
        cpl_msg_info(fctid,"No 2MASS index found -- cannot continue");
        vircam_jmp_tidy(0);
        return(-1);
    }
    
    /* Get catalogue parameters */

    if (vircam_catpars(catindex,&(ps.catpath),&(ps.catname)) == VIR_FATAL) {
	vircam_jmp_tidy(0);
        cpl_frame_delete(catindex);
	return(-1);
    }
    cpl_frame_delete(catindex);

    /* Get the number of DITs */

    pp = cpl_propertylist_load(cpl_frame_get_filename(cpl_frameset_get_frame(ps.science_frames,0)),0);
    if (vircam_pfits_get_ndit(pp,&ndit) != VIR_OK) {
        cpl_msg_error(fctid,"No value for NDIT available");
        freepropertylist(pp);
        vircam_jmp_tidy(0);
        return(-1);
    }
    cpl_propertylist_delete(pp);

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_jmp_config.extenum,
		       (const cpl_frame *)cpl_frameset_get_frame(ps.science_frames,0),
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_jmp_tidy(0);
	return(-1);
    }

    /* Now loop for all the extensions... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
	isfirst = (j == jst);
	gaincor_fac = (ps.gaincors)[j-1];

        /* Load up the calibration frames into vir_fits/vir_tfits structures 
	   It is a fatal error if any one of them can't load properly */
	
	ps.fdark = vircam_fits_load(ps.master_dark,CPL_TYPE_FLOAT,j);
	if (ps.fdark == NULL) {
	    cpl_msg_error(fctid,"Error loading master dark %s[%d]\n%s",
			  cpl_frame_get_filename(ps.master_dark),j,
			  cpl_error_get_message());
	    vircam_jmp_tidy(0);
	    return(-1);
	}
	ps.fflat = vircam_fits_load(ps.master_twilight_flat,CPL_TYPE_FLOAT,j);
	if (ps.fflat == NULL) {
	    cpl_msg_error(fctid,"Error loading master flat %s[%d]\n%s",
			  cpl_frame_get_filename(ps.master_twilight_flat),j,
			  cpl_error_get_message());
	    vircam_jmp_tidy(0);
	    return(-1);
	}
	ps.fconf = vircam_fits_load(ps.master_conf,CPL_TYPE_INT,j);
	if (ps.fconf == NULL) {
	    cpl_msg_error(fctid,"Error loading master conf %s[%d]\n%s",
			  cpl_frame_get_filename(ps.master_conf),j,
			  cpl_error_get_message());
	    vircam_jmp_tidy(0);
	    return(-1);
	}
	if (! isconf) 
	    vircam_jmp_bpm2conf();
	if (vircam_mask_load(ps.mask,j,
			     cpl_image_get_size_x(vircam_fits_get_image(ps.fconf)),
			     cpl_image_get_size_y(vircam_fits_get_image(ps.fconf))) != VIR_OK) {
	    cpl_msg_error(fctid,"Error loading mask from master conf %s[%d]\n%s",
			  cpl_frame_get_filename(ps.master_conf),j,
			  cpl_error_get_message());
	    vircam_jmp_tidy(0);
	    return(-1);
	}
	ps.fchantab = vircam_tfits_load(ps.chantab,j);
	if (ps.fchantab == NULL) {
	    cpl_msg_error(fctid,"Error loading channel table %s[%d]\n%s",
			  cpl_frame_get_filename(ps.chantab),j,
			  cpl_error_get_message());
	    vircam_jmp_tidy(0);
	    return(-1);
	}

        /* Load up the vir_fits structures for the science images */

	ps.nscience = cpl_frameset_get_size(ps.science_frames);
        ps.sci_fits = vircam_fits_load_list(ps.science_frames,CPL_TYPE_FLOAT,j);
	if (ps.sci_fits == NULL) {
	    cpl_msg_error(fctid,"Error loading science frames extension %d: %s",
			  j,cpl_error_get_message());
	    vircam_jmp_tidy(0);
	    return(-1);
	}

	/* Loop through and mark the frames where the header says the detector
	   wasn't live */

        for (i = 0; i < ps.nscience; i++) {
	    ff = ps.sci_fits[i];
	    vircam_pfits_get_detlive(vircam_fits_get_ehu(ff),&live);
	    if (! live) 
		vircam_fits_set_error(ff,VIR_FATAL);
	}

	/* Loop for all the science frames and do the 2d corrections */

	cpl_msg_info(fctid,"Doing stage1 corrections on %s",
		     vircam_fits_get_extname(ps.sci_fits[0]));
	for (i = 0; i < ps.nscience; i++) {
	    ff = ps.sci_fits[i];
	    if (vircam_fits_get_status(ff) == VIR_FATAL) {
		cpl_msg_info(fctid,"Detector is flagged dead in %s",
			     vircam_fits_get_fullname(ff));
		continue;
	    }
	    status = VIR_OK;
	    (void)vircam_darkcor(ff,ps.fdark,1.0,&status);
	    (void)vircam_lincor(ff,ps.fchantab,1,ndit,&status);
	    (void)vircam_flatcor(ff,ps.fflat,&status);
	    (void)vircam_gaincor(ff,gaincor_fac,&status);
            if (vircam_jmp_config.destripe) 
		(void)vircam_destripe(ff,ps.mask,&status);
	    vircam_fits_set_error(ff,status);
	}

	/* Do a simple sky correction if requested */

	if (vircam_jmp_config.skycor) {
            cpl_msg_info(fctid,"Doing sky correction");
   	    vircam_jmp_skycor();
	}

	/* Calculate the illumination correction */

        cpl_msg_info(fctid,"Doing illumination correction");
        (void)strcpy(current_cat,ps.catname);
        (void)strcpy(current_catpath,ps.catpath);
	vircam_jmp_illum();	

	/* Get the data from the illumination correction table */

	if (ps.illcor != NULL) {
    	    ic = vircam_tfits_get_table(ps.illcor);
  	    nrows = cpl_table_get_nrow(ic);
	    x1 = cpl_table_get_data_float(ic,"xmin");
	    x2 = cpl_table_get_data_float(ic,"xmax");
	    y1 = cpl_table_get_data_float(ic,"ymin");
	    y2 = cpl_table_get_data_float(ic,"ymax");
	    ill = cpl_table_get_data_float(ic,"illcor");
	    n = 0;
	    for (i = 0; i < nrows; i++) 
		if (ill[i] != -99.0)
		    n++;

	    /* If there is nothing to fit, then don't bother with the rest
	       of this */

	    if (n == 0) {
		cpl_msg_warning(fctid,"Illum correction table is all NULLs");

	    } else {
		
		/* Create bivector and vector objects with the xy positions 
		   and the illumination correction */

		bv = cpl_bivector_new(n);
		bv_x = cpl_bivector_get_x_data(bv);
		bv_y = cpl_bivector_get_y_data(bv);
		v = cpl_vector_new(n);
		vdata = cpl_vector_get_data(v);
		n = 0;
		for (i = 0; i < nrows; i++) {
		    if (ill[i] == -99.0)
			continue;
		    bv_x[n] = 0.5*(double)(x2[i] + x1[i]);
		    bv_y[n] = 0.5*(double)(y2[i] + y1[i]);
		    vdata[n++] = (double)ill[i];
		}

		/* Now fit a surface to these results */

		poly = cpl_polynomial_fit_2d_create(bv,v,vircam_jmp_config.nord,
						    NULL);
		cpl_vector_delete(v);

		/* Evaluate the polynomial at the input points and replace 
		   the values in the illumination correction with the new 
		   value. */

		v = cpl_vector_new(2);
		vdata = cpl_vector_get_data(v);
		for (i = 0; i < nrows; i++) {
		    vdata[0] = 0.5*(double)(x2[i] + x1[i]);
		    vdata[1] = 0.5*(double)(y2[i] + y1[i]);
		    val = cpl_polynomial_eval(poly,v);
		    ill[i] = val;
		}
		cpl_vector_delete(v);
		cpl_bivector_delete(bv);
		cpl_polynomial_delete(poly);
	    }
	}

	/* Save the table */

        cpl_msg_info(fctid,"Saving illumination correction table");
	dummyqc = vircam_mesostep_analyse_dummyqc(3);
	vircam_jmp_save_illum(framelist,parlist);
	freepropertylist(dummyqc);
	
	/* Clean up on aisle 12! */

	vircam_jmp_tidy(1);
    }

    /* Final cleanup */

    vircam_jmp_tidy(0);
    return(0);
}

static cpl_propertylist *vircam_mesostep_analyse_dummyqc(int type) {
    cpl_propertylist *p;

    /* Get an empty property list */

    p = cpl_propertylist_new();

    /* Now switch for the various products */

    switch (type) {

    /* Illumination tables */

    case 3:
        cpl_propertylist_update_float(p,"ESO QC ILLUMCOR_RMS",0.0);
        cpl_propertylist_set_comment(p,"ESO QC ILLUMCOR_RMS",
                                     "RMS of illumination correction map");
        break;
    default:
        break;
    }

    /* Get out of here */

    return(p);
} 

/**@}*/

/*

$Log: vircam_mesostep_analyse.c,v $
Revision 1.18  2008/05/06 12:15:20  jim
Changed to use new version of vircam_catpars

Revision 1.17  2007/11/26 09:59:06  jim
Recipe now takes ndit into account when doing linearity correction

Revision 1.16  2007/10/25 18:39:22  jim
Altered to remove some lint messages

Revision 1.15  2007/10/19 06:55:06  jim
Modifications made to use new method for directing the recipes to the
standard catalogues using the sof

Revision 1.14  2007/07/09 13:21:56  jim
Modified to use new version of vircam_exten_range

Revision 1.13  2007/06/13 08:11:27  jim
Modified docs to reflect changes in DFS tags

Revision 1.12  2007/05/08 21:31:16  jim
fixed typo

Revision 1.11  2007/05/08 10:42:44  jim
Added gain correction

Revision 1.10  2007/05/02 12:53:11  jim
typo fixes in docs

Revision 1.9  2007/04/04 16:05:59  jim
Modified to make paf information a bit more correct

Revision 1.8  2007/04/04 10:36:18  jim
Modified to use new dfs tags

Revision 1.7  2007/03/29 12:19:38  jim
Little changes to improve documentation

Revision 1.6  2007/03/14 14:49:13  jim
Fixed problem with missing paf files in jmp recipes if detlive = F. Also
fixed problem where extra dummy products were being created

Revision 1.5  2007/02/07 10:12:40  jim
Removed calls to vircam_ndit_correct as this is now no longer necessary

Revision 1.4  2006/12/19 13:33:02  jim
Blocked off bivariate fit so that in the event that the original illumnation
correction fails, it doesn't then try to fit the surface.

Revision 1.3  2006/12/18 16:43:15  jim
fixed bug with nord parameter doc

Revision 1.2  2006/12/18 16:42:27  jim
Blocked off the fitting bit so that it doesn't get done if the initial
illumination correction fails.

Revision 1.1  2006/12/04 21:10:14  jim
Initial entry


*/
