/* $Id: vircam_paf.c,v 1.7 2007/10/25 17:34:01 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 17:34:01 $
 * $Revision: 1.7 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <cpl.h>
#include "vircam_paf.h"
#include "vircam_utils.h"


static char *vircam_paf_keyname(char *name);
static void vircam_paf_write_char(char *keyname, char val, char *comment,
				  FILE *paf);
static void vircam_paf_write_int(char *keyname, int val, char *comment,
				 FILE *paf);
static void vircam_paf_write_long(char *keyname, long val, char *comment,
				  FILE *paf);
static void vircam_paf_write_float(char *keyname, float val, char *comment,
				   FILE *paf);
static void vircam_paf_write_double(char *keyname, double val, char *comment,
				    FILE *paf);
static void vircam_paf_write_string(char *keyname, char *val, char *comment,
				    FILE *paf);


/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_paf_print
    \par Purpose:
        Open a new PAF file and write it.
    \par Description:
        A new PAF file is opened and a header written to it. A propertylist
	is scanned and those values will be dumped to the PAF.
    \param ftemp
        A filename template. A number indicating the detector number and the
	filename extension '.paf' will be added.
    \param paf_id 
        A PAF identifier
    \param paf_desc
        A PAF description.
    \param incl
        The propertylist with the properties to be dumped to the PAF
    \retval VIR_OK
        If everything went OK
    \retval VIR_FATAL
        If there was some sort of error.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_paf_print(char *ftemp, const char *paf_id, 
			    const char *paf_desc, cpl_propertylist *incl) {
    FILE *paf;
    char filename[BUFSIZ],*comment,*name,*keyname;
    const char *fctid="vircam_paf_print";
    int i,ichip;
    cpl_property *p;

    /* Check the propertylist even exists... */

    if (incl == NULL) {
	cpl_msg_warning(fctid,"NULL propertylist. No PAF file written");
	return(VIR_FATAL);
    }

    /* Find the number of the detector */

    ichip = cpl_propertylist_get_int(incl,"ESO DET CHIP NO");
    if (ichip == 0) {
	cpl_error_reset();
	cpl_msg_warning(fctid,"No entry DET CHIP NO in property list");
	return(VIR_FATAL);
    }

    /* Create a file name from the template and try to open it. If it won't
       open, then get out of here */

    (void)snprintf(filename,BUFSIZ,"%s_%02d.paf",ftemp,ichip);
    if ((paf = fopen(filename,"w")) == NULL) {
	cpl_msg_warning(fctid,"Unable to open %s.",filename);
	return(VIR_FATAL);
    }

    /* Write the header */

    fprintf(paf,"PAF.HDR.START         ;# start of header\n");
    fprintf(paf,"PAF.TYPE              \"pipeline product\" ;\n");
    fprintf(paf,"PAF.ID                \"%s\"\n", paf_id);
    fprintf(paf,"PAF.NAME              \"%s\"\n", filename);
    fprintf(paf,"PAF.DESC              \"%s\"\n", paf_desc);
    fprintf(paf,"PAF.CHCK.CHECKSUM     \"\"\n");
    fprintf(paf,"PAF.HDR.END           ;# end of header\n");
    fprintf(paf,"\n");

    /* Now loop through the propertylist and create a keyname that is standard
       format for a paf file */

    for (i = 0; i < cpl_propertylist_get_size(incl); i++) {
	p = cpl_propertylist_get(incl,i);
	name = (char *)cpl_property_get_name(p);
	keyname = vircam_paf_keyname(name);
	comment = (char *)cpl_property_get_comment(p);
	
	/* Now switch for each acceptable data type */

	switch (cpl_property_get_type(p)) {
	case CPL_TYPE_CHAR:
	    vircam_paf_write_char(keyname,cpl_property_get_char(p),
				  comment,paf);
	    break;
	case CPL_TYPE_INT:
	    vircam_paf_write_int(keyname,cpl_property_get_int(p),
				 comment,paf);
	    break;
	case CPL_TYPE_LONG:
	    vircam_paf_write_long(keyname,cpl_property_get_long(p),
				  comment,paf);
	    break;
	case CPL_TYPE_FLOAT:
	    vircam_paf_write_float(keyname,cpl_property_get_float(p),
				   comment,paf);
	    break;
	case CPL_TYPE_DOUBLE:
	    vircam_paf_write_double(keyname,cpl_property_get_double(p),
				    comment,paf);
	    break;
	case CPL_TYPE_STRING:
	    vircam_paf_write_string(keyname,(char *)cpl_property_get_string(p),
				    comment,paf);
	    break;
	default:
	    break;
	}
	cpl_free(keyname);
    }

    /* Close things up */

    fclose(paf);
    return(VIR_OK);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_paf_req_items
    \par Purpose:
        Create a propertylist with the required items for a PAF.
    \par Description:
        A propertylist is scanned for the items that are required to 
	be in a QC PAF. All those that fit the bill are extracted and
	put into an returned propertylist. If an error occurs a warning
	message is given and the return is NULL. More properties can
	be added later by other routines. The caller is responsible for
	freeing the returned propertylist
    \param src
        A propertylist with the properties that we want to extract.
    \return 
        A propertylist with the extracted properties or NULL in case
	of an error
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_propertylist *vircam_paf_req_items(cpl_propertylist *src) {
    cpl_propertylist *dest;
    int nreq=5,i;
    const char *req_items[] = {"ESO DET CHIP NO","EXTNAME","DATE-OBS",
			       "MJD-OBS","ESO DET DIT"};
    const char *req_reg = "^ESO QC";
    const char *fctid = "vircam_paf_req_items";

    /* Get a new propertylist */

    dest = cpl_propertylist_new();

    /* Copy the items over */

    for (i = 0; i < nreq; i++) {
	if (cpl_propertylist_copy_property(dest,src,req_items[i]) != CPL_ERROR_NONE) {
	    cpl_msg_warning(fctid,"Can't find property %s in source header",
			    req_items[i]);
	    cpl_propertylist_delete(dest);
	    return(NULL);
	}
    }
	
    /* Now the QC stuff */

    cpl_propertylist_copy_property_regexp(dest,src,req_reg,0);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
	cpl_msg_warning(fctid,"Can't find regexp %s in source header",
			req_reg);
	cpl_propertylist_delete(dest);
	return(NULL);
    }

    /* Otherwise get out of here */

    return(dest);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_paf_phu_items
    \par Purpose:
        Create a propertylist with the PHU items for a PAF.
    \par Description:
        A propertylist is scanned for the items that are required from the
	PHU to be in a QC PAF. All those that fit the bill are extracted and
	put into an returned propertylist. If an error occurs a warning
	message is given and the property is assigned a blank value. The 
	caller is responsible for freeing the returned propertylist
    \param src
        A propertylist with the properties that we want to extract.
    \return 
        A propertylist with the extracted properties.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_propertylist *vircam_paf_phu_items(cpl_propertylist *src) {
    cpl_propertylist *dest;
    int nreq=2,i;
    const char *req_items[] = {"ARCFILE","ESO TPL ID"};
    const char *fctid = "vircam_paf_req_items";

    /* Get a new propertylist */

    dest = cpl_propertylist_new();

    /* Copy the items over */

    for (i = 0; i < nreq; i++) {
	if (cpl_propertylist_copy_property(dest,src,req_items[i]) != CPL_ERROR_NONE) {
	    cpl_error_reset();
	    cpl_msg_warning(fctid,"Can't find property %s in source header",
			    req_items[i]);
            cpl_propertylist_update_string(dest,req_items[i],"");
	    cpl_propertylist_set_comment(dest,req_items[i],"Not available");
	}
    }
	
    /* Now get out of here */

    return(dest);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_paf_append
    \par Purpose:
        Append a property to the PAF propertylist
    \par Description:
        A property from a source propertylist is extracted and appended to
	a destination propertylist which will ultimately be used for a PAF.
	If the property list missing from the source a null string property
	is used.
    \param dest
        A destination propertylist 
    \param src
        A source propertylist with the property that we want to extract.
    \param prop
        The name of the property to be copied
    \return 
        A propertylist with the extracted properties.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_paf_append(cpl_propertylist *dest, cpl_propertylist *src,
			      char *prop) {

    /* Check to see if the property is even available */

    if (cpl_propertylist_has(src,prop)) {
        cpl_propertylist_copy_property(dest,src,prop);
    } else {
	cpl_propertylist_update_string(dest,prop,"");
	cpl_propertylist_set_comment(dest,prop,"Not available");
    }
}

static void vircam_paf_write_char(char *keyname, char val, char *comment,
				  FILE *paf) {

    /* Write a the value out */

    if (comment != NULL) 
	fprintf(paf,KEYFMT "\"%c\" ; # %s\n",keyname,val,comment);
    else
	fprintf(paf,KEYFMT "\"%c\"\n",keyname,val);
}

static void vircam_paf_write_int(char *keyname, int val, char *comment,
				 FILE *paf) {

    /* Write a the value out */

    if (comment != NULL) 
	fprintf(paf,KEYFMT "%d ; # %s\n",keyname,val,comment);
    else
	fprintf(paf,KEYFMT "%d\n",keyname,val);
}

static void vircam_paf_write_long(char *keyname, long val, char *comment,
				  FILE *paf) {

    /* Write a the value out */

    if (comment != NULL) 
	fprintf(paf,KEYFMT "%ld ; # %s\n",keyname,val,comment);
    else
	fprintf(paf,KEYFMT "%ld\n",keyname,val);
}

static void vircam_paf_write_float(char *keyname, float val, char *comment,
				   FILE *paf) {

    /* Write a the value out */

    if (comment != NULL) 
	fprintf(paf,KEYFMT "%.10g ; # %s\n",keyname,val,comment);
    else
	fprintf(paf,KEYFMT "%.10g\n",keyname,val);
}

static void vircam_paf_write_double(char *keyname, double val, char *comment,
				    FILE *paf) {

    /* Write a the value out */

    if (comment != NULL) 
	fprintf(paf,KEYFMT "%.10g ; # %s\n",keyname,val,comment);
    else
	fprintf(paf,KEYFMT "%.10g\n",keyname,val);
}

static void vircam_paf_write_string(char *keyname, char *val, char *comment,
				    FILE *paf) {

    /* Write a the value out */

    if (comment != NULL) 
	fprintf(paf,KEYFMT "\"%s\" ; # %s\n",keyname,val,comment);
    else
	fprintf(paf,KEYFMT "\"%s\"\n",keyname,val);
}


static char *vircam_paf_keyname(char *name) {
    char *keyname,*t;

    /* Locate the ESO part of the keyword and ditch it */

    keyname = cpl_malloc(SZKEY+1);
    t = strstr(name,"ESO");
    if (t == NULL) 
	(void)strncpy(keyname,name,SZKEY);
    else
	strncpy(keyname,t+4,SZKEY);
    keyname[SZKEY] = '\0';

    /* Now replace spaces with dots */

    for (t = keyname; *t != '\0'; t++) {
	if (*t == ' ')
	    *t = '.';
    }
    return(keyname);
}

	
/*

$Log: vircam_paf.c,v $
Revision 1.7  2007/10/25 17:34:01  jim
Modified to remove lint warnings

Revision 1.6  2007/10/15 12:50:28  jim
Modified for compatibility with cpl_4.0

Revision 1.5  2007/04/30 09:40:01  jim
Added vircam_paf_append

Revision 1.4  2007/04/04 10:33:05  jim
Modified to add vircam_paf_phu_items

Revision 1.3  2007/03/01 12:42:42  jim
Modified slightly after code checking

Revision 1.2  2007/02/14 14:00:40  jim
Added vircam_paf_req_items

Revision 1.1  2007/02/14 12:53:51  jim
New entry


*/
