/* $Id: vircam_jmp_utils.c,v 1.33 2008/07/10 13:05:53 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2008/07/10 13:05:53 $
 * $Revision: 1.33 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <math.h>
#include <string.h>

#include "vircam_utils.h"
#include "vircam_pfits.h"
#include "vircam_dfs.h"
#include "vircam_mods.h"
#include "vircam_stats.h"
#include "vircam_fits.h"
#include "vircam_tfits.h"
#include "vircam_wcsutils.h"
#include "vircam_paf.h"

#include "vircam_jmp_utils.h"

static char *vircam_jmp_outfile(const char *bname, int ind, int isfits);

/**
    \defgroup vircam_jmp_utils vircam_jmp_utils
    \ingroup supportroutines

    \brief
    These are utility routines to support the jitter/microstep recipes

    \author
    Jim Lewis, CASU
*/

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_save_simple
    \par Purpose:
        Save the simple image products
    \par Description:
        The simple image products are saved here.
    \par Language:
        C
    \param framelist
        The input recipe frameset
    \param parlist
        The input recipe parameter list
    \return 
        0 if everything is ok
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_jmp_save_simple(cpl_frameset *framelist,
				  cpl_parameterlist *parlist) {
    cpl_propertylist *plist;
    int i,isdummy;
    cpl_frame *product_frame;
    char *fname;
    const char *base[] = {"","simple_jmp","simple_std","simple_mes"};
    const char *fctid = "vircam_jmp_save_simple";

    /* Initialise the product frameset */

    if (ps.product_frames_simple == NULL)
        ps.product_frames_simple = cpl_malloc(ps.nscience*sizeof(cpl_frame *));

    /* Loop for each of the frames in the ustep sequence */

    for (i = 0; i < ps.nscience; i++) {
	fname = vircam_jmp_outfile(base[recflag],i,1);
	isdummy = (vircam_fits_get_status(ps.sci_fits[i]) != VIR_OK);

        /* If we need to make a PHU then do that now based on the first frame
           in the input frame list */

        if (isfirst) {

	    /* Create a new product frame object and define some tags */

  	    product_frame = cpl_frame_new();
  	    cpl_frame_set_filename(product_frame,fname);
	    switch (recflag) {
	    case RECSCI:
  	        cpl_frame_set_tag(product_frame,VIRCAM_PRO_SIMPLE_SCI);
		break;
	    case RECSTD:
  	        cpl_frame_set_tag(product_frame,VIRCAM_PRO_SIMPLE_STD);
		break;
	    default:
  	        cpl_frame_set_tag(product_frame,VIRCAM_PRO_SIMPLE_SCI);
		break;
	    }
 	    cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
 	    cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	    cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

  	    /* Set up the PHU header */

 	    plist = vircam_fits_get_phu(ps.sci_fits[i]);
	    if (i == 0)
		ps.phupaf = vircam_paf_phu_items(plist);;
	    vircam_dfs_set_product_primary_header(plist,product_frame,
						  framelist,parlist,
						  vircam_recipename,
						  "PRO-1.15");

	    /* 'Save' the PHU image */			 

	    if (cpl_image_save(NULL,fname,CPL_BPP_8_UNSIGNED,plist,
			       CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
		cpl_msg_error(fctid,"Cannot save product PHU");
		cpl_frame_delete(product_frame);
 	        freespace(fname);
		return(-1);
	    }
	    cpl_frameset_insert(framelist,product_frame);
	    ps.product_frames_simple[i] = product_frame;
	}

        /* Get the extension property list */

        plist = vircam_fits_get_ehu(ps.sci_fits[i]);
	if (isdummy)
	    vircam_dummy_property(plist);

        /* Fiddle with the header now */

	product_frame = ps.product_frames_simple[i];
        vircam_dfs_set_product_exten_header(plist,product_frame,framelist,
					    parlist,vircam_recipename,
					    "PRO-1.15");
        if (cpl_image_save(vircam_fits_get_image(ps.sci_fits[i]),fname,
			   CPL_BPP_IEEE_FLOAT,plist,
	  	           CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product image extension");
	    freespace(fname);
	    return(-1);
        }
	
	/* Clear the allocated workspace */

	freespace(fname);
    }

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_save_super
    \par Purpose:
        Save the super frame image products
    \par Description:
        The super frame image products are saved here.
    \par Language:
        C
    \param framelist
        The input recipe frameset
    \param parlist
        The input recipe parameter list
    \return 
        0 if everything is ok
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_jmp_save_super(cpl_frameset *framelist,
				 cpl_parameterlist *parlist) {
    cpl_propertylist *plist,*p;
    int i,isdummy,isdummyc;
    cpl_frame *product_frame;
    char *fname,*fnamec;
    const char *base[] = {"","super_jmp","super_std","super_mes"};
    const char *basec[] = {"","superc_jmp","superc_std","superc_mes"};
    vir_fits *ff,*ffc;
    cpl_image *fim,*fimc;
    const char *fctid = "vircam_jmp_save_super";

    /* Initialise the product frameset */

    if (ps.product_frames_super == NULL)
        ps.product_frames_super = cpl_malloc(ps.nustep_sets*sizeof(cpl_frame *));
    if (ps.product_frames_superc == NULL)
        ps.product_frames_superc = cpl_malloc(ps.nustep_sets*sizeof(cpl_frame *));

    /* Loop for each of the ustep sequences */

    for (i = 0; i < ps.nustep_sets; i++) {

	/* Work out which frame to base the output on. If this particular
	   microstep sequence failed for whatever reason, there will be
	   a dummy super frame. */
	
        ff = ps.ustep_sets[i].super;
        fim = vircam_fits_get_image(ff);
        ffc = ps.ustep_sets[i].superc;
        fimc = vircam_fits_get_image(ffc);
        isdummy = (vircam_fits_get_status(ff) != VIR_OK);
        isdummyc = (vircam_fits_get_status(ffc) != VIR_OK);

	/* Create some names */

	fname = vircam_jmp_outfile(base[recflag],i,1);
	fnamec = vircam_jmp_outfile(basec[recflag],i,1);

        /* If we need to make a PHU then do that now based on the first frame
           in the input frame list */

        if (isfirst) {

	    /* Create a new product frame object and define some tags */

  	    product_frame = cpl_frame_new();
  	    cpl_frame_set_filename(product_frame,fname);
	    switch (recflag) {
	    case RECSCI:
  	        cpl_frame_set_tag(product_frame,VIRCAM_PRO_INTER_SCI);
		break;
	    case RECSTD:
  	        cpl_frame_set_tag(product_frame,VIRCAM_PRO_INTER_STD);
		break;
	    default:
  	        cpl_frame_set_tag(product_frame,VIRCAM_PRO_INTER_SCI);
		break;
	    }
 	    cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
 	    cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	    cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

  	    /* Set up the PHU header. */

	    plist = vircam_fits_get_phu(ff);
	    vircam_dfs_set_product_primary_header(plist,product_frame,
						  framelist,parlist,
						  vircam_recipename,
						  "PRO-1.15");

	    /* 'Save' the PHU image */			 

	    if (cpl_image_save(NULL,fname,CPL_BPP_8_UNSIGNED,plist,
			       CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
		cpl_msg_error(fctid,"Cannot save product PHU");
		cpl_frame_delete(product_frame);
		freespace(fname);
		freespace(fnamec);
		return(-1);
	    }
	    cpl_frameset_insert(framelist,product_frame);
	    ps.product_frames_super[i] = product_frame;

	    /* Now do exactly the same thing for the confidence map */

  	    product_frame = cpl_frame_new();
  	    cpl_frame_set_filename(product_frame,fnamec);
	    switch (recflag) {
	    case RECSCI:
		cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_SCI);
		break;
	    case RECSTD:
		cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_STD);
		break;
	    default:
		cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_SCI);
		break;
	    }
 	    cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
 	    cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	    cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

  	    /* Set up the PHU header. */

	    plist = vircam_fits_get_phu(ffc);
	    vircam_dfs_set_product_primary_header(plist,product_frame,
						  framelist,parlist,
						  vircam_recipename,
						  "PRO-1.15");

	    /* 'Save' the PHU image */			 

	    if (cpl_image_save(NULL,fnamec,CPL_BPP_8_UNSIGNED,plist,
			       CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
		cpl_msg_error(fctid,"Cannot save product PHU");
		cpl_frame_delete(product_frame);
		freespace(fname);
		freespace(fnamec);
		return(-1);
	    }
	    cpl_frameset_insert(framelist,product_frame);
	    ps.product_frames_superc[i] = product_frame;
	}


        /* Get the extension property list */

        p = cpl_propertylist_duplicate(vircam_fits_get_ehu(ff));
	if (isdummy)
	    vircam_dummy_property(p);

        /* Fiddle with the header now */

	product_frame = ps.product_frames_super[i];
        vircam_dfs_set_product_exten_header(p,product_frame,framelist,parlist,
					    vircam_recipename,"PRO-1.15");
        if (cpl_image_save(fim,fname,CPL_BPP_IEEE_FLOAT,p,CPL_IO_EXTEND) != 
	    CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product image extension");
	    freespace(fname);
	    freespace(fnamec);
	    return(-1);
        }
	cpl_propertylist_delete(p);
	
        /* And now for the confidence map */

        p = cpl_propertylist_duplicate(vircam_fits_get_ehu(ffc));
	if (isdummyc)
	    vircam_dummy_property(p);

        /* Fiddle with the header now */

	product_frame = ps.product_frames_superc[i];
        vircam_dfs_set_product_exten_header(p,product_frame,framelist,parlist,
					    vircam_recipename,"PRO-1.15");
        if (cpl_image_save(fimc,fnamec,CPL_BPP_16_SIGNED,p,CPL_IO_EXTEND) != 
	    CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save confidence map image extension");
	    freespace(fname);
	    freespace(fnamec);
	    return(-1);
        }
        cpl_propertylist_delete(p);
	
	/* Clear the allocated workspace */

	freespace(fname);
	freespace(fnamec);
    }

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_save_stack
    \par Purpose:
        Save the stack frame image products
    \par Description:
        The stack frame image products are saved here.
    \par Language:
        C
    \param framelist
        The input recipe frameset
    \param parlist
        The input recipe parameter list
    \return 
        0 if everything is ok
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_jmp_save_stack(cpl_frameset *framelist,
				 cpl_parameterlist *parlist) {
    cpl_propertylist *plist,*p,*pafprop;
    int isdummy,isdummyc;
    vir_fits *ff,*ffc;
    cpl_image *fim,*fimc;
    cpl_frame *product_frame;
    char *fname,*fnamec,*fnamepaf;
    const char *base[] = {"","stack_jmp","stack_std","stack_mes"};
    const char *basec[] = {"","stackc_jmp","stackc_std","stackc_mes"};
    const char *fctid = "vircam_jmp_save_stack";

    /* Work out which frame to base the output on. If this particular
       jitter sequence failed for whatever reason, there will be
       a dummy stack frame. */

    ff = ps.stack_frame;
    fim = vircam_fits_get_image(ff);
    ffc = ps.stackc_frame;
    fimc = vircam_fits_get_image(ffc);
    isdummy = (vircam_fits_get_status(ff) != VIR_OK);
    isdummyc = (vircam_fits_get_status(ffc) != VIR_OK);

    /* Create some names */

    fname = vircam_jmp_outfile(base[recflag],0,1);
    fnamec = vircam_jmp_outfile(basec[recflag],0,1);
    fnamepaf = vircam_jmp_outfile(base[recflag],0,0);

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

	/* Create a new product frame object and define some tags */

	product_frame = cpl_frame_new();
	cpl_frame_set_filename(product_frame,fname);
	switch (recflag) {
	case RECSCI:
  	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_JITTERED_SCI);
	    break;
	case RECSTD:
  	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_JITTERED_STD);
	    break;
	default:
  	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_JITTERED_SCI);
	    break;
	}
	cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);
	ps.product_frame_stack = product_frame;

	/* Set up the PHU header. */

	plist = vircam_fits_get_phu(ff);
	vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
					      parlist,vircam_recipename,
					      "PRO-1.15");

	/* 'Save' the PHU image */			 

	if (cpl_image_save(NULL,fname,CPL_BPP_8_UNSIGNED,plist,
			   CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product PHU");
	    cpl_frame_delete(product_frame);
	    freespace(fname);
	    freespace(fnamec);
	    return(-1);
	}
	cpl_frameset_insert(framelist,product_frame);

	/* Now do exactly the same thing for the confidence map */

	product_frame = cpl_frame_new();
	cpl_frame_set_filename(product_frame,fnamec);
	switch (recflag) {
	case RECSCI:
	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_SCI);
	    break;
	case RECSTD:
	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_STD);
	    break;
	default:
	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_CONF_SCI);
	    break;
	}
	cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);
	ps.product_frame_stackc = product_frame;

	/* Set up the PHU header. */

	plist = vircam_fits_get_phu(ffc);
	vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
					      parlist,vircam_recipename,
					      "PRO-1.15");

	/* 'Save' the PHU image */			 

	if (cpl_image_save(NULL,fnamec,CPL_BPP_8_UNSIGNED,plist,
			   CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product PHU");
	    cpl_frame_delete(product_frame);
	    freespace(fname);
	    freespace(fnamec);
	    return(-1);
	}
	cpl_frameset_insert(framelist,product_frame);
    }

    /* Get the extension property list */

    p = cpl_propertylist_duplicate(vircam_fits_get_ehu(ff));
    if (isdummy) {
	vircam_dummy_property(p);
	vircam_merge_propertylists(p,dummyqc);
    }

    /* Fiddle with the header now */

    product_frame = ps.product_frame_stack;
    vircam_dfs_set_product_exten_header(p,product_frame,framelist,parlist,
					vircam_recipename,"PRO-1.15");
    if (cpl_image_save(fim,fname,CPL_BPP_IEEE_FLOAT,p,CPL_IO_EXTEND) != 
	CPL_ERROR_NONE) {
	cpl_msg_error(fctid,"Cannot save product image extension");
	freespace(fname);
	freespace(fnamec);
	return(-1);
    }

    /* Write PAF */

    pafprop = vircam_paf_req_items(p);
    vircam_merge_propertylists(pafprop,ps.phupaf);
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"ESO INS FILT1 NAME");
    vircam_paf_append(pafprop,p,"ESO DET NDIT");
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"RA");
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"DEC");
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"ESO TEL AIRM START");
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"ESO TEL GUID FWHM");
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"ESO TEL AMBI RHUM");
    vircam_paf_append(pafprop,vircam_fits_get_phu(ff),"ESO OBS TARG NAME");
    if (vircam_paf_print(fnamepaf,vircam_recipepaf,"QC file",
                         pafprop) != VIR_OK)
        cpl_msg_warning(fctid,"Unable to save PAF for stack");
    cpl_propertylist_delete(pafprop);

    /* Quick tidy */

    cpl_propertylist_delete(p);

    /* And now for the confidence map */

    p = cpl_propertylist_duplicate(vircam_fits_get_ehu(ffc));
    if (isdummyc) 
	vircam_dummy_property(p);

    /* Fiddle with the header now */

    product_frame = ps.product_frame_stackc;
    vircam_dfs_set_product_exten_header(p,product_frame,framelist,parlist,
					vircam_recipename,"PRO-1.15");
    if (cpl_image_save(fimc,fnamec,CPL_BPP_16_SIGNED,p,CPL_IO_EXTEND) != 
	CPL_ERROR_NONE) {
	cpl_msg_error(fctid,"Cannot save product image extension");
	freespace(fname);
	freespace(fnamec);
	freepropertylist(p);
	return(-1);
    }
    cpl_propertylist_delete(p);

    /* Clear the allocated workspace */

    freespace(fname);
    freespace(fnamec);
    freespace(fnamepaf);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_save_catalogue
    \par Purpose:
        Save the cataloge products
    \par Description:
        The catalogue products are saved here.
    \par Language:
        C
    \param framelist
        The input recipe frameset
    \param parlist
        The input recipe parameter list
    \return 
        0 if everything is ok
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_jmp_save_catalogue(cpl_frameset *framelist,
				     cpl_parameterlist *parlist) {
    cpl_frame *product_frame;
    int isdummy,i,status;
    cpl_table *ftab;
    cpl_propertylist *ehu,*phu,*ehu2,*pafprop;
    char *fname,*fnamepaf;
    const char *base[] = {"","catalogue_jmp","catalogue_std","catalogue_mes"};
    const char *fctid = "vircam_jmp_save_catalogue";

    /* Work out whether you have a table to save or not. If you don't
       then create a dummy table and save it */

    isdummy = 0;
    if (ps.outcat != NULL) {
	ftab = vircam_tfits_get_table(ps.outcat);
    } else {
	ftab = vircam_dummy_catalogue(2);
	isdummy = 1;
    }

    /* Make an attempt to get some header information */

    if (ps.outcat != NULL) {
	ehu = vircam_tfits_get_ehu(ps.outcat);
    } else if (ps.stack_frame != NULL) {
	ehu = vircam_fits_get_ehu(ps.stack_frame);
    } else {
	for (i = 0; i < ps.nscience; i++) {
	    if ((ehu = vircam_fits_get_ehu(ps.sci_fits[i])) != NULL)
		break;
	}
    }
    if (ps.outcat != NULL) {
	phu = vircam_tfits_get_phu(ps.outcat);
    } else if (ps.stack_frame != NULL) {
	phu = vircam_fits_get_phu(ps.stack_frame);
    } else {
	for (i = 0; i < ps.nscience; i++) {
	    if ((phu = vircam_fits_get_phu(ps.sci_fits[i])) != NULL)
		break;
	}
    }

    /* Create a name */

    fname = vircam_jmp_outfile(base[recflag],0,1);
    fnamepaf = vircam_jmp_outfile(base[recflag],0,0);

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

	/* Create a new product frame object and define some tags */

	product_frame = cpl_frame_new();
	cpl_frame_set_filename(product_frame,fname);
	switch (recflag) {
	case RECSCI:
	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_OBJCAT_SCI);
	    break;
	case RECSTD:
	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_OBJCAT_STD);
	    break;
	default:
	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_OBJCAT_SCI);
	    break;
	}
	cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_TABLE);
	cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);
	ps.product_frame_cat = product_frame;

	/* Set up the PHU header. */

	vircam_dfs_set_product_primary_header(phu,product_frame,framelist,
					      parlist,vircam_recipename,
					      "PRO-1.15");

	/* Copy the header and delete any WCS info */

	ehu2 = cpl_propertylist_duplicate(ehu);
	status = VIR_OK;
	if (isdummy) {
	    vircam_dummy_property(ehu2);
	    vircam_merge_propertylists(ehu2,dummyqc);
  	    (void)vircam_removewcs(ehu2,&status);
	}

	/* Setup the extension header */

        vircam_dfs_set_product_exten_header(ehu2,product_frame,framelist,
					    parlist,vircam_recipename,
					    "PRO-1.15");

	/* 'Save' the PHU image */			 

	if (cpl_table_save(ftab,phu,ehu2,fname,CPL_IO_DEFAULT) != 
	    CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product PHU");
	    cpl_frame_delete(product_frame);
	    freepropertylist(ehu2);
	    freespace(fname);
	    return(-1);
	}

	/* Write PAF */

	pafprop = vircam_paf_req_items(ehu2);
        vircam_merge_propertylists(pafprop,ps.phupaf);
	vircam_paf_append(pafprop,phu,"ESO INS FILT1 NAME");
	vircam_paf_append(pafprop,ehu,"ESO DET NDIT");
	vircam_paf_append(pafprop,phu,"RA");
	vircam_paf_append(pafprop,phu,"DEC");
	vircam_paf_append(pafprop,phu,"ESO TEL AIRM START");
	vircam_paf_append(pafprop,phu,"ESO TEL GUID FWHM");
	vircam_paf_append(pafprop,phu,"ESO TEL AMBI RHUM");
	vircam_paf_append(pafprop,phu,"ESO OBS TARG NAME");
	if (vircam_paf_print(fnamepaf,vircam_recipepaf,"QC file",
			     pafprop) != VIR_OK)
	    cpl_msg_warning(fctid,"Unable to save PAF for catalogue");
	cpl_propertylist_delete(pafprop);

	/* Quick tidy */

	freepropertylist(ehu2);
	if (isdummy)
	    cpl_table_delete(ftab);
	cpl_frameset_insert(framelist,product_frame);

    } else {


        /* Fiddle with the header now */

        product_frame = ps.product_frame_cat;
        vircam_dfs_set_product_exten_header(ehu,product_frame,framelist,
					    parlist,vircam_recipename,
					    "PRO-1.15");
	
	/* Copy the header and delete any WCS info */

	ehu2 = cpl_propertylist_duplicate(ehu);
	status = VIR_OK;
	if (isdummy) {
	    vircam_dummy_property(ehu2);
	    vircam_merge_propertylists(ehu2,dummyqc);
  	    (void)vircam_removewcs(ehu2,&status);
	}
	
	/* Save the table */

        if (cpl_table_save(ftab,NULL,ehu2,fname,CPL_IO_EXTEND) != 
	    CPL_ERROR_NONE) {
  	    cpl_msg_error(fctid,"Cannot save product image extension");
	    freepropertylist(ehu2);
	    freespace(fname);
	    return(-1);
        }

	/* Write PAF */

	pafprop = vircam_paf_req_items(ehu2);
        vircam_merge_propertylists(pafprop,ps.phupaf);
	vircam_paf_append(pafprop,phu,"ESO INS FILT1 NAME");
	vircam_paf_append(pafprop,ehu,"ESO DET NDIT");
	vircam_paf_append(pafprop,phu,"RA");
	vircam_paf_append(pafprop,phu,"DEC");
	vircam_paf_append(pafprop,phu,"ESO TEL AIRM START");
	vircam_paf_append(pafprop,phu,"ESO TEL GUID FWHM");
	vircam_paf_append(pafprop,phu,"ESO TEL AMBI RHUM");
	vircam_paf_append(pafprop,phu,"ESO OBS TARG NAME");
	if (vircam_paf_print(fnamepaf,vircam_recipepaf,"QC file",
			     pafprop) != VIR_OK)
	    cpl_msg_warning(fctid,"Unable to save PAF for catalogue");
	cpl_propertylist_delete(pafprop);

	/* Quick tidy */

	freepropertylist(ehu2);
        if (isdummy)
	    cpl_table_delete(ftab);
    }
    freespace(fname);
    freespace(fnamepaf);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_save_illum
    \par Purpose:
        Save the illumination correction table product
    \par Description:
        The illumination correction table is saved here
    \par Language:
        C
    \param framelist
        The input recipe frameset
    \param parlist
        The input recipe parameter list
    \return 
        0 if everything is ok
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_jmp_save_illum(cpl_frameset *framelist,
				 cpl_parameterlist *parlist) {
    cpl_frame *product_frame;
    int isdummy,i,status;
    cpl_table *ftab;
    cpl_propertylist *ehu,*phu,*ehu2,*pafprop;
    const char *fname = "illum.fits";
    const char *fnamepaf = "illum";
    const char *fctid = "vircam_jmp_save_illum";

    /* Work out whether you have a table to save or not. If you don't
       then create a dummy table and save it */

    isdummy = 0;
    if (ps.illcor != NULL) {
	ftab = vircam_tfits_get_table(ps.illcor);
    } else {
	ftab = vircam_illcor_newtab(1);
	isdummy = 1;
    }

    /* Make an attempt to get some header information */

    if (ps.illcor != NULL) {
	ehu = vircam_tfits_get_ehu(ps.illcor);
    } else {
	for (i = 0; i < ps.nscience; i++) {
	    if ((ehu = vircam_fits_get_ehu(ps.sci_fits[i])) != NULL)
		break;
	}
    }
    if (ps.illcor != NULL) {
	phu = vircam_tfits_get_phu(ps.illcor);
    } else {
	for (i = 0; i < ps.nscience; i++) {
	    if ((phu = vircam_fits_get_phu(ps.sci_fits[i])) != NULL)
		break;
	}
    }

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

	/* Create a new product frame object and define some tags */

	product_frame = cpl_frame_new();
	cpl_frame_set_filename(product_frame,fname);
	switch (recflag) {
	case RECSTD:
    	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_ILLCOR_STD);
	    break;
	case RECMES:
    	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_ILLCOR_MES);
	    break;
	default:
    	    cpl_frame_set_tag(product_frame,VIRCAM_PRO_ILLCOR_STD);
	    break;
	}
	cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_TABLE);
	cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
	cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);
	ps.product_frame_illcor = product_frame;

	/* Set up the PHU header. */

	if (ps.phupaf != NULL) 
	    ps.phupaf = vircam_paf_phu_items(phu);
	vircam_dfs_set_product_primary_header(phu,product_frame,framelist,
					      parlist,vircam_recipename,
					      "PRO-1.15");

	/* Copy the header and delete any WCS info */

	ehu2 = cpl_propertylist_duplicate(ehu);
	status = VIR_OK;
	(void)vircam_removewcs(ehu2,&status);
	if (isdummy) {
	    vircam_dummy_property(ehu2);
	    vircam_merge_propertylists(ehu2,dummyqc);
	}
	    

	/* Setup the extension header */

        vircam_dfs_set_product_exten_header(ehu2,product_frame,framelist,
					    parlist,vircam_recipename,
					    "PRO-1.15");

	/* 'Save' the PHU image */			 

	if (cpl_table_save(ftab,phu,ehu2,fname,CPL_IO_DEFAULT) != 
	    CPL_ERROR_NONE) {
	    cpl_msg_error(fctid,"Cannot save product PHU");
	    cpl_frame_delete(product_frame);
	    freepropertylist(ehu2);
	    return(-1);
	}

	/* Write the PAF */

        pafprop = vircam_paf_req_items(ehu2);
        vircam_merge_propertylists(pafprop,ps.phupaf);
	vircam_paf_append(pafprop,phu,"ESO INS FILT1 NAME");
	vircam_paf_append(pafprop,ehu,"ESO DET NDIT");
	vircam_paf_append(pafprop,phu,"RA");
	vircam_paf_append(pafprop,phu,"DEC");
	vircam_paf_append(pafprop,phu,"ESO TEL AIRM START");
	vircam_paf_append(pafprop,phu,"ESO TEL GUID FWHM");
	vircam_paf_append(pafprop,phu,"ESO TEL AMBI RHUM");
	vircam_paf_append(pafprop,phu,"ESO OBS TARG NAME");
        if (vircam_paf_print((char *)fnamepaf,vircam_recipepaf,"QC file",
                             pafprop) != VIR_OK)
            cpl_msg_warning(fctid,"Unable to save PAF for illcor table");
        cpl_propertylist_delete(pafprop);
	
	/* Quick tidy */

        if (isdummy)
	    cpl_table_delete(ftab);
	freepropertylist(ehu2);
	cpl_frameset_insert(framelist,product_frame);

    } else {


        /* Fiddle with the header now */

        product_frame = ps.product_frame_illcor;
        vircam_dfs_set_product_exten_header(ehu,product_frame,framelist,
					    parlist,vircam_recipename,
					    "PRO-1.15");
	
	/* Copy the header and delete any WCS info */

	ehu2 = cpl_propertylist_duplicate(ehu);
	status = VIR_OK;
	(void)vircam_removewcs(ehu2,&status);
	if (isdummy) {
	    vircam_dummy_property(ehu2);
	    vircam_merge_propertylists(ehu2,dummyqc);
	}
	
	/* Save the table */

        if (cpl_table_save(ftab,NULL,ehu2,fname,CPL_IO_EXTEND) != 
	    CPL_ERROR_NONE) {
  	    cpl_msg_error(fctid,"Cannot save product image extension");
	    freepropertylist(ehu2);
	    return(-1);
        }

	/* Write the PAF */

        pafprop = vircam_paf_req_items(ehu2);
        vircam_merge_propertylists(pafprop,ps.phupaf);
	vircam_paf_append(pafprop,phu,"ESO INS FILT1 NAME");
	vircam_paf_append(pafprop,ehu,"ESO DET NDIT");
	vircam_paf_append(pafprop,phu,"RA");
	vircam_paf_append(pafprop,phu,"DEC");
	vircam_paf_append(pafprop,phu,"ESO TEL AIRM START");
	vircam_paf_append(pafprop,phu,"ESO TEL GUID FWHM");
	vircam_paf_append(pafprop,phu,"ESO TEL AMBI RHUM");
	vircam_paf_append(pafprop,phu,"ESO OBS TARG NAME");
        if (vircam_paf_print((char *)fnamepaf,vircam_recipepaf,"QC file",
                             pafprop) != VIR_OK)
            cpl_msg_warning(fctid,"Unable to save PAF for illcor table");
        cpl_propertylist_delete(pafprop);
	
	/* Quick tidy */

	freepropertylist(ehu2);
        if (isdummy)
	    cpl_table_delete(ftab);
    }

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_ustep_seq
    \par Purpose:
        Group input files into microstep sequences
    \par Description:
        An input list of files are grouped by the microstep number in
	the header.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_ustep_seq(void) {
    int nalloc,i,match,j,ustepnum,nustep;
    vir_fits *ff;
    cpl_propertylist *plist;
    const char *fctid = "vircam_jmp_ustep_seq";

    /* Allocate an initial amount of workspace for the microstep sequence
       image sets */

    nalloc = INITALLOC;
    ps.ustep_sets = cpl_malloc(nalloc*sizeof(ustep_set));
    ps.nustep_sets = 0;

    /* Loop for each frame and get the microstep sequence number from the
       primary header */

    for (i = 0; i < ps.nscience; i++) {
	ff = ps.sci_fits[i];
	plist = vircam_fits_get_phu(ff);
	if (vircam_pfits_get_ustepnum(plist,&ustepnum) != VIR_OK) {
	    cpl_msg_error(fctid,"No microstep number in %s",
			  vircam_fits_get_filename(ff));
	    vircam_fits_set_error(ff,VIR_FATAL);
	    continue;
	}

	/* See if this sequence number matches any of the others we've
	   already defined. If it does, then simply add this frame into
	   that sequences frameset */
	
	match = 0;
	for (j = 0; j < ps.nustep_sets; j++) {
	    if (ustepnum == ps.ustep_sets[j].ustep_number) {
		match = 1;
	  	ps.ustep_sets[j].f[ps.ustep_sets[j].nframes] = ff;
 		ps.ustep_sets[j].nframes += 1;
		if (vircam_fits_get_status(ff) != VIR_FATAL) 
		    ps.ustep_sets[j].ngood += 1;
		break;
	    }
	}

	/* If it doesn't match increment the number of sets and check to 
	   make sure that we haven't overrun our allocation for ustep sets. */

	if (! match) {
	    if (ps.nustep_sets+1 == nalloc) {
		nalloc += INITALLOC;
		ps.ustep_sets = cpl_realloc(ps.ustep_sets,nalloc*sizeof(ustep_set));
	    }

	    /* Now define this ustep set */
	    
	    (void)vircam_pfits_get_nusteps(plist,&nustep);
	    ps.ustep_sets[ps.nustep_sets].f = cpl_malloc(nustep*sizeof(vir_fits *));
	    ps.ustep_sets[ps.nustep_sets].ustep_number = ustepnum;
	    ps.ustep_sets[ps.nustep_sets].nustep = nustep;
            ps.ustep_sets[ps.nustep_sets].status = VIR_OK;
	    ps.ustep_sets[ps.nustep_sets].super = NULL;
	    ps.ustep_sets[ps.nustep_sets].superc = NULL;
	    ps.ustep_sets[ps.nustep_sets].nframes = 0;
	    ps.ustep_sets[ps.nustep_sets].f[0] = ff;
	    ps.ustep_sets[ps.nustep_sets].nframes = 1;
	    ps.ustep_sets[ps.nustep_sets].ngood = 0;
	    if (vircam_fits_get_status(ff) != VIR_FATAL)
		ps.ustep_sets[ps.nustep_sets].ngood += 1;
	    ps.nustep_sets++;
        }    
    }

    /* Fix the allocation to what we need and throw the rest away */

    ps.ustep_sets = cpl_realloc(ps.ustep_sets,
				ps.nustep_sets*sizeof(ustep_set));

    /* Loop through each of the defined sets and see if each is complete */

    for (i = 0; i < ps.nustep_sets; i++) {
	if (ps.ustep_sets[i].ngood == 0) {
	    cpl_msg_warning(fctid,"Microstep sequence %d has no input",
			    ps.ustep_sets[i].ustep_number);
	    ps.ustep_sets[i].status = VIR_FATAL;
	} else if (ps.ustep_sets[i].ngood != ps.ustep_sets[i].nustep) {
	    cpl_msg_warning(fctid,"Microstep sequence %d incomplete",
			    ps.ustep_sets[i].ustep_number);
	    ps.ustep_sets[i].status = VIR_WARN;
	}
    }
}


/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_interleave
    \par Purpose:
        Group input files by microstep sequence and interleave them
    \par Description:
        Group input files by microstep sequence and interleave them into
	superframes.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_interleave(void) {
    int i,refset,k,nstep,status,nk,*d;
    long npts;
    float val;
    double refx,refy,refra,refdec,x,y;
    cpl_image *fi,*outimage,*outconf;
    cpl_propertylist *plist;
    vir_fits *ff,**tmp;
    cpl_wcs *wcs;
    cpl_array *dims;
    const char *fctid = "vircam_jmp_interleave";

    /* Work out the microstep sequences */

    vircam_jmp_ustep_seq(); 
   
    /* Get workspace to contain the output superframes */

    ps.dith_input = cpl_malloc(ps.nustep_sets*sizeof(vir_fits*));
    ps.dithc_input = cpl_malloc(ps.nustep_sets*sizeof(vir_fits*));
	    
    /* Loop for each of the ustep sets */

    ps.ndith = 0;
    for (i = 0; i < ps.nustep_sets; i++) {
	if (ps.ustep_sets[i].status == VIR_FATAL) {
	    outimage = vircam_dummy_image(ps.ustep_sets[i].f[0]);
	    outconf = vircam_dummy_image(ps.fconf);
	    ff = vircam_fits_wrap(outimage,ps.ustep_sets[i].f[0],NULL,NULL);
            ps.dith_input[ps.ndith] = ff;
	    vircam_fits_set_error(ff,VIR_FATAL);
  	    ps.ustep_sets[i].super = ff;
	    ff = vircam_fits_wrap(outconf,ps.ustep_sets[i].f[0],NULL,NULL);
            ps.dithc_input[ps.ndith++] = ff;
	    vircam_fits_set_error(ff,VIR_FATAL);
  	    ps.ustep_sets[i].superc = ff;
	    continue;
	}

	/* Work out the offsets in the sequence from the WCS in the header.
	   Fail the ustep sequence if any of it's components have an
	   unreadable WCS */

	refset = 0;
	for (k = 0; k < ps.ustep_sets[i].nframes; k++) {
	    ff = ps.ustep_sets[i].f[k];
	    if (vircam_fits_get_status(ff) == VIR_FATAL)
		continue;
	    wcs = cpl_wcs_new_from_propertylist(vircam_fits_get_ehu(ff));
	    if (wcs == NULL) {
		cpl_msg_error(fctid,"Unable to open WCS structure %s",
			      vircam_fits_get_fullname(ff));
		vircam_fits_set_error(ff,VIR_FATAL);
		continue;
	    }
    
	    /* Get the background value for this image */

	    fi = vircam_fits_get_image(ff);
	    npts = vircam_getnpts(fi);
	    val = vircam_med(cpl_image_get_data_float(fi),NULL,npts);
	    cpl_propertylist_update_float(vircam_fits_get_ehu(ff),
					  "ESO DRS BACKMED",val);

	    /* If this is the first frame, then set up the reference coords. */
    
	    if (refset == 0) {
		refset = 1;
		dims = cpl_wcs_get_image_dims(wcs);
		d = cpl_array_get_data_int(dims);
		refx = 0.5*(double)d[0];
		refy = 0.5*(double)d[1];
		vircam_xytoradec(wcs,refx,refy,&refra,&refdec);
		cpl_propertylist_update_double(vircam_fits_get_ehu(ff),
					       "ESO DRS XOFFMICRO",0.0);
		cpl_propertylist_update_double(vircam_fits_get_ehu(ff),
					       "ESO DRS YOFFMICRO",0.0);
		cpl_wcs_delete(wcs);
		continue;
	    }
    
	    /* Take the reference ra and dec and see where that occurs on
	       the program image in x,y space */
    
	    vircam_radectoxy(wcs,refra,refdec,&x,&y);
	    x = refx - x;
	    y = refy - y;
	    cpl_propertylist_update_double(vircam_fits_get_ehu(ff),
					   "ESO DRS XOFFMICRO",x);
	    cpl_propertylist_update_double(vircam_fits_get_ehu(ff),
					   "ESO DRS YOFFMICRO",y);
	    cpl_wcs_delete(wcs);
	}

	/* Get a temporary workspace to hold all the good files in the list */

	tmp = cpl_malloc(ps.ustep_sets[i].nframes*sizeof(vir_fits *));
	nk = 0;
	for (k = 0; k < ps.ustep_sets[i].nframes; k++) {
	    ff = ps.ustep_sets[i].f[k];
	    if (vircam_fits_get_status(ff) != VIR_FATAL)
		tmp[nk++] = ff;
	}
	if (nk < ps.ustep_sets[i].nframes) {
	    cpl_msg_error(fctid,"A frame in this ustep sequence failed");
	    ps.ustep_sets[i].status = VIR_WARN;
	}

        /* Otherwise interleave them */

	status = VIR_OK;
	nstep = (int)sqrt((double)(ps.ustep_sets[i].nustep));
        (void)vircam_interleave(tmp,nk,&(ps.fconf),1,nstep,&plist,&outimage,
				&outconf,&status);
	freespace(tmp);
        if (status != VIR_OK) {
	    cpl_msg_error(fctid,"Interleaving failed for ugroup %d extn %d",
			  ps.ustep_sets[i].ustep_number,
			  vircam_fits_get_nexten(ps.ustep_sets[i].f[0]));
	    freepropertylist(plist);
	    freeimage(outimage);
	    freeimage(outconf);
	    outimage = vircam_dummy_image(ps.ustep_sets[i].f[0]);
	    outconf = vircam_dummy_image(ps.fconf);
	    ff = vircam_fits_wrap(outimage,ps.ustep_sets[i].f[0],NULL,NULL);
	    vircam_fits_set_error(ff,VIR_FATAL);
  	    ps.ustep_sets[i].super = ff;
	    ff = vircam_fits_wrap(outconf,ps.ustep_sets[i].f[0],NULL,NULL);
	    vircam_fits_set_error(ff,VIR_FATAL);
  	    ps.ustep_sets[i].superc = ff;
	}

        /* Wrap the output results and store them away into the list of
           frames that will be dithered later on. */

        ps.dith_input[ps.ndith] = vircam_fits_wrap(outimage,
						   ps.ustep_sets[i].f[0],NULL,
						   plist);
        ps.dithc_input[ps.ndith++] = vircam_fits_wrap(outconf,
						      ps.ustep_sets[i].f[0],
						      NULL,plist);
	ps.ndithc = ps.ndith;
	ps.ustep_sets[i].super = ps.dith_input[ps.ndith - 1];
	ps.ustep_sets[i].superc = ps.dithc_input[ps.ndithc - 1];
	freepropertylist(plist);
    }
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_dither_offsets
    \par Purpose:
        Work out the dither offsets for a group of files that will eventually
	be jittered into a stack frame.
    \par Description:
        The dither offsets will be calculated for a group of files that will
	eventually be jittered into a stack frame. The WCS is used to get
	an initial estimate. Then objects are located on the frames and the
	offset estimates are refined by looking at the positions of the 
	objects.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_dither_offsets(void) {
    int status,i,ustepnum,nmatch,refset;
    float *xoffs,*yoffs,xoff,yoff,filtfwhm;
    cpl_wcs *wcsref,*wcs;
    vir_fits *ff,*ffc;
    vir_tfits *catref,*outcat;
    cpl_table *cr,*oc;
    const char *fctid = "vircam_jmp_dither_offsets";

    /* Is there anything to dither? If not, then get out of here. NB: We
       don't have to check the status of the input dither files as the
       calling routine won't have included any files with bad status in
       the input list for this routine. */

    if (ps.ndith == 0)
	return;

    /* If there is only 1 then set the offsets to zero and get out of here */

    if (ps.ndith == 1) {
	cpl_propertylist_update_double(vircam_fits_get_ehu(ps.dith_input[0]),
				       "ESO DRS XOFFDITHER",(double)0.0);
	cpl_propertylist_update_double(vircam_fits_get_ehu(ps.dith_input[0]),
				       "ESO DRS YOFFDITHER",(double)0.0);
	return;
    }

    /* Initialise the status variable and get some workspace for the offsets */

    status = VIR_OK;
    xoffs = cpl_malloc(ps.ndith*sizeof(float));
    yoffs = cpl_malloc(ps.ndith*sizeof(float));

    /* Loop for all the input files and get the FITS WCS information. */

    refset = 0;
    wcsref = NULL;
    for (i = 0; i < ps.ndith; i++) {
	if (vircam_fits_get_status(ps.dith_input[i]) == VIR_FATAL)
	    continue;	
	wcs = cpl_wcs_new_from_propertylist(vircam_fits_get_ehu(ps.dith_input[i]));
        (void)vircam_pfits_get_ustepnum(vircam_fits_get_phu(ps.dith_input[i]),
					&ustepnum);

	/* If we can't get a WCS for this image, then signal that with
	   a warning */

	if (wcs == NULL) {
	    cpl_msg_warning(fctid,"Unable to get WCS for ustep %d",ustepnum);
	    xoffs[i] = 0.0;
	    yoffs[i] = 0.0;
	    vircam_fits_set_error(ps.dith_input[i],VIR_WARN);
	    continue;
	}

	/* If we don't have a reference WCS yet, then make the current frame
	   WCS the reference and move on */

	if (! refset) {
	    xoffs[i] = 0.0;
	    yoffs[i] = 0.0;
	    refset = 1;
	    wcsref = wcs;
	    continue;
	}

	/* Right, assuming we're here, then we need to work out the xy 
	   differences */

        (void)vircam_diffxywcs(wcs,wcsref,&xoff,&yoff,&status);

	/* Did it work? If not the set a warning status for this file */

	if (status != VIR_OK) {
	    xoffs[i] = 0.0;
	    yoffs[i] = 0.0;
	    cpl_msg_warning(fctid,"Unable to WCS difference for %d",ustepnum);
	} else {
	    xoffs[i] = xoff;
	    yoffs[i] = yoff;
	}
	cpl_wcs_delete(wcs);
    }
    if (wcsref != NULL)
	cpl_wcs_delete(wcsref);

    /* Now generate a catalogue for each of the input images */

    catref = NULL;
    filtfwhm = (interlv ? 3.5 : 2.0);
    for (i = 0; i < ps.ndith; i++) {
	if (vircam_fits_get_status(ps.dith_input[i]) == VIR_FATAL)
	    continue;
	status = VIR_OK;
	ff = ps.dith_input[i];
	if (ps.ndithc != 1)
 	    ffc = ps.dithc_input[i];
	else
	    ffc = ps.dithc_input[0];
        (void)vircam_pfits_get_ustepnum(vircam_fits_get_phu(ff),&ustepnum);
	outcat = NULL;
	(void)vircam_imcore(ff,ffc,25,5.0,0,3.5,64,1,filtfwhm,&outcat,
			    &status);

	/* If we get a bad status from imcore and this frame has already
	   failed in the WCS stage, then mark it with VIR_FATAL status
	   and move to the next one */

	if (status != VIR_OK && vircam_fits_get_status(ff) != VIR_OK) {
	    vircam_fits_set_error(ff,VIR_FATAL);
	    freetfits(outcat);
	    cpl_msg_error(fctid,"Unable to get offsets for %d",ustepnum);
	    continue;

	/* If we get bad status, but this file has a perfectly good offset
	   from the WCS then just go with it that and issue a warning */

	} else if (status != VIR_OK && vircam_fits_get_status(ff) == VIR_OK) {
	    vircam_fits_set_error(ff,VIR_WARN);
	    freetfits(outcat);
	    cpl_msg_error(fctid,"Unable to get object offset for %d. Going with WCS offset",
			  ustepnum);
	    continue;

	/*  OK, we got a good status from imcore. See if the reference 
	    catalogue has already been defined. If it hasn't then this 
	    catalogue becomes the reference */

	} else {
	    oc = vircam_tfits_get_table(outcat);
	    cpl_table_add_scalar(oc,"X_coordinate",(double)xoffs[i]);
	    cpl_table_add_scalar(oc,"Y_coordinate",(double)yoffs[i]);
	    if (catref == NULL) {
		catref = outcat;
		cr = oc;
		continue;
	    }

	    /* If this isn't the reference file, then do the cross match */


	    (void)vircam_matchxy(oc,cr,10.0,&xoff,&yoff,&nmatch,&status);
	    freetfits(outcat);
    
            /* If we got a bad result with the cross match and this current
	       file is already got a warning status, then flag it as fatal
	       status */
    
            if ((nmatch == 0 || status == VIR_FATAL) && 
		vircam_fits_get_status(ff) != VIR_OK) {
		xoff = 0.0;
		yoff = 0.0;
		vircam_fits_set_error(ff,VIR_FATAL);
		cpl_msg_error(fctid,"Unable to match stars for %d",ustepnum);
    
	    /* If we got a bad result with the cross match and this current
	       file has a good status, then just go with the WCS offset
	       we already have */
    
	    } else if ((nmatch == 0 || status == VIR_FATAL) && 
		       vircam_fits_get_status(ff) == VIR_OK) {
		vircam_fits_set_error(ff,VIR_WARN);
		xoff = 0.0;
		yoff = 0.0;
		cpl_msg_warning(fctid,"Unable to match stars for %d. Going with WCS offsets",
				ustepnum);
	    }
    
	    /* Add the current offsets to what we already have */
    
	    xoffs[i] += xoff;
	    yoffs[i] += yoff;
	}
    }

    /* Write the results to the headers */

    for (i = 0; i < ps.ndith; i++) {
	if (vircam_fits_get_status(ps.dith_input[i]) == VIR_FATAL)
	    continue;
	ff = ps.dith_input[i];
	cpl_propertylist_update_double(vircam_fits_get_ehu(ff),
				       "ESO DRS XOFFDITHER",(double)xoffs[i]);
	cpl_propertylist_update_double(vircam_fits_get_ehu(ff),
				       "ESO DRS YOFFDITHER",(double)yoffs[i]);
    }

    /* Get rid of some workspace */

    freespace(xoffs);
    freespace(yoffs);
    freetfits(catref);
}		

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_dither_images
    \par Purpose:
        Dither input images into a stack frame.
    \par Description:
        Dither input images into a stack frame. The offsets will have been
	worked out beforehand in vircam_jmp_dither_offsets.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_dither_images(void) {
    int status,ngood,i,n;
    vir_fits **d,**dc,*ff;
    cpl_propertylist *dither_ehu,*dither_phu;
    cpl_image *outdither,*outditherc;
    const char *fctid = "vircam_jmp_dither_images";

    /* Count how many good input images there are. */

    ngood = 0;
    for (i = 0; i < ps.ndith; i++)
	if (vircam_fits_get_status(ps.dith_input[i]) != VIR_FATAL)
	    ngood++;
    
    /* If there are none, then get out of here. The output jittered image
       and it's confidence map should still be set to NULL */

    if (ngood == 0) {
	cpl_msg_error(fctid,"No good input images for jittering");


	outdither = vircam_dummy_image(ps.sci_fits[0]);
	outditherc = vircam_dummy_image(ps.dithc_input[0]);
	ff = vircam_fits_wrap(outdither,ps.sci_fits[0],NULL,NULL);
	ps.stack_frame = ff;
	vircam_fits_set_error(ff,VIR_FATAL);
	ff = vircam_fits_wrap(outditherc,ps.dithc_input[0],NULL,NULL);
	vircam_fits_set_error(ff,VIR_FATAL);
	ps.stackc_frame = ff;
	return;
    }
     
    /* Now transfer the input images over to new arrays... */

    d = cpl_malloc(ngood*sizeof(vir_fits *));
    dc = cpl_malloc(ngood*sizeof(vir_fits *));
    n = 0;
    for (i = 0; i < ps.ndith; i++) {
	if (vircam_fits_get_status(ps.dith_input[i]) != VIR_FATAL) {
 	    d[n] = ps.dith_input[i];
	    if (ps.ndithc != 1) 
	        dc[n++] = ps.dithc_input[i];
	    else 
		dc[n++] = ps.dithc_input[0];
	}
    }

    /* Dither the remaining images */

    status = VIR_OK;
    (void)vircam_imdither(d,dc,ngood,ngood,5.0,5.0,&dither_ehu,&outdither,
			  &outditherc,&status);
    dither_phu = cpl_propertylist_duplicate(vircam_fits_get_phu(d[0]));
    if (status != VIR_OK) {
	freeimage(outdither);
	freeimage(outditherc);
	cpl_msg_error(fctid,"Error jittering to output");
    } else {
	ps.stack_frame = vircam_fits_wrap(outdither,d[0],dither_phu,
					  dither_ehu);
	ps.stackc_frame = vircam_fits_wrap(outditherc,dc[0],dither_phu,
					   dither_ehu);
    }

    /* Tidy and exit */

    freepropertylist(dither_phu);
    freepropertylist(dither_ehu);
    freespace(d);
    freespace(dc);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_catalogue
    \par Purpose:
        Extract a catalogue of objects on a the dithered stacked frame.
    \par Description:
        Extract a catalogue of objects on a the dithered stacked frame.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_catalogue(void) {
    const char *fctid = "vircam_jmp_catalogue";
    int status;
    vir_tfits *outtab;
    float filtfwhm;

    /* Check to see if there has been a jitter frame defined */

    if (ps.stack_frame == NULL || vircam_fits_get_status(ps.stack_frame) == VIR_FATAL) {
	cpl_msg_error(fctid,"No stack image available. No catalogue generated");
	return;
    }

    /* Generate the catalogue */

    status = VIR_OK;
    filtfwhm = (interlv ? 3.5 : 2);
    (void)vircam_imcore(ps.stack_frame,ps.stackc_frame,
			vircam_jmp_config.ipix,
			vircam_jmp_config.threshold,
			vircam_jmp_config.icrowd,
			vircam_jmp_config.rcore,
			vircam_jmp_config.nbsize,2,filtfwhm,
			&outtab,&status);

    /* If it failed, then get rid of workspaces and get out of here with
       an error message. If it went well, then wrap the result in a
       vir_tfits object */

    if (status != VIR_OK) {
	cpl_msg_error(fctid,"Error generating catalogue");
	freetfits(outtab);
	vircam_fits_set_error(ps.stack_frame,VIR_FATAL);
    } else {
	ps.outcat = outtab;
    }
			  
    return;
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_matched_stds
    \par Purpose:
        Match the extracted objects catalogue with a table of 2mass standards
    \par Description:
        The catalogue of extracted objects from the stacked image is matched
	to a table of 2mass standards to form a matched standards table.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/
    
extern void vircam_jmp_matched_stds(void) {
    int status;
    const char *fctid = "vircam_jmp_matched_stds";
    cpl_table *stdscat;

    /* Initialise status */

    status = VIR_OK;

    /* Check that we have a catalogue */
    
    if (ps.outcat == NULL) {
	cpl_msg_error(fctid,"No input catalogue found");
	return;
    }

    /* Get some standard stars */

    (void)vircam_getstds(vircam_fits_get_ehu(ps.stack_frame),1,
			 current_catpath,current_cat,&stdscat,&status);
    if (status != VIR_OK) {
	freetable(stdscat);
	cpl_msg_error(fctid,"Failed to find any standards");
	return;
    }

    /* Now match this against the catalogue */

    (void)vircam_matchstds(vircam_tfits_get_table(ps.outcat),stdscat,
			   10.0,&(ps.matchstds),&status);
    freetable(stdscat);
    if (status != VIR_OK) {
	freetable(ps.matchstds);
	cpl_msg_error(fctid,"Failed to match standards to catalogue");
	return;
    }
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_wcsfit
    \par Purpose:
        Fit a world-coordinate system
    \par Description:
        The matched standards table is used to fit a world coordinate system.
	The resulting WCS parameters are written to the header of the stacked
	image using the usual FITS conventions.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/
    
extern void vircam_jmp_wcsfit(void) {
    int status,n,i;
    const char *fctid = "vircam_jmp_wcsfit";
    float *ra,*dec,*x,*y;
    double r,d;
    cpl_table *cat;
    cpl_wcs *wcs;

    /* Initialise status */

    status = VIR_OK;

    /* Check that we have a catalogue */
    
    if (ps.matchstds == NULL) {
	cpl_msg_error(fctid,"No input matched standards catalogue found");
	return;
    }

    /* Fit the plate solution */

    (void)vircam_platesol(vircam_fits_get_ehu(ps.stack_frame),
			  vircam_tfits_get_ehu(ps.outcat),ps.matchstds,
			  6,1,&status);
    if (status != VIR_OK) {
	cpl_msg_error(fctid,"Failed to fit WCS");
	return;
    }

    /* Update the RA and DEC of the objects in the object catalogue */

    cat = vircam_tfits_get_table(ps.outcat);
    n = cpl_table_get_nrow(cat);
    wcs = cpl_wcs_new_from_propertylist(vircam_fits_get_ehu(ps.stack_frame));
    if (wcs == NULL) {
	cpl_msg_error(fctid,"Failed to fill RA and Dec in catalogue");
	return;
    }
    x = cpl_table_get_data_float(cat,"X_coordinate"); 
    y = cpl_table_get_data_float(cat,"Y_coordinate");
    ra = cpl_table_get_data_float(cat,"RA");
    dec = cpl_table_get_data_float(cat,"DEC");
    for (i = 0; i < n; i++) {
	vircam_xytoradec(wcs,x[i],y[i],&r,&d);
	ra[i] = (float)r;
	dec[i] = (float)d;
    }
    cpl_wcs_delete(wcs);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_photcal
    \par Purpose:
        Work out the photometric zeropoint
    \par Description:
        The matched standards table is used to calculate a photometric
	zeropoint.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_photcal(void) {
    int status;
    const char *fctid = "vircam_jmp_photcal";
    char filt[32];
    cpl_propertylist *pl;

    /* Initialise status */

    status = VIR_OK;

    /* Check that we have a catalogue */
    
    if (ps.matchstds == NULL || cpl_table_get_nrow(ps.matchstds) == 0) {
	cpl_msg_error(fctid,"No input matched standards catalogue found");
	return;
    }

    /* What filter is this? */

    if (vircam_pfits_get_filter(vircam_fits_get_phu(ps.stack_frame),filt) != VIR_OK) {
	cpl_msg_error(fctid,"No filter name in stack header");
	return;
    }

    /* Fit the photometric calibration */
    
    pl = vircam_tfits_get_ehu(ps.outcat);
    (void)vircam_photcal(&(ps.stack_frame),&(ps.matchstds),&pl,1,filt,
			 ps.tphottab,&status);
    if (status != VIR_OK) {
	cpl_msg_error(fctid,"Failed to fit photometric zeropoint");
	return;
    }
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_bpm2conf
    \par Purpose:
        Turn a bad pixel mask into a dummy confidence map
    \par Description:
        Turn a bad pixel mask into a dummy confidence map. Any pixel which
	is flagged in the bpm gets a confidence of 0. Otherwise it has a
	value of 100.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_bpm2conf(void) {
    cpl_image *im;
    int i,n,*data;

    /* Get the image */

    im = vircam_fits_get_image(ps.fconf);
    n = cpl_image_get_size_x(im)*cpl_image_get_size_y(im);
    data = cpl_image_get_data_int(im);

    /* Convert it now */ 

    for (i = 0; i < n; i++)
	data[i] = (data[i] == 1 ? 0 : 100);
	
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_skycor
    \par Purpose:
        Correct for sky background in the science images
    \par Description:
        All good images are combined with rejection to form a sky background
	image. The combined image is normalised to zero median and then
	subtracted off of the input images.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_skycor(void) {
    int i,ngood,status;
    long npts;
    vir_fits **ftmp,*ff;
    const char *fctid = "vircam_jmp_skycor";
    unsigned char *rejmask,*rejplus;
    cpl_propertylist *drs;
    cpl_image *skyimg,*fim;
    float *data,med,sig;

    /* Sort out all images with good status */

    ftmp = cpl_malloc(ps.nscience*sizeof(vir_fits *));
    ngood = 0;
    for (i = 0; i < ps.nscience; i++) {
	ff = ps.sci_fits[i];
	if (vircam_fits_get_status(ff) != VIR_FATAL)
	    ftmp[ngood++] = ff;
    }

    /* If there aren't any good images, then get out of here now */

    if (ngood == 0) {
	freespace(ftmp);
	cpl_msg_error(fctid,"Sky correction impossible. No good science frames available");
	return;
    }

    /* Combine all the good science images */

    status = VIR_OK;
    (void)vircam_imcombine(ftmp,ngood,2,1,1,3.0,&skyimg,&rejmask,
			   &rejplus,&drs,&status);
    freespace(rejmask);
    freespace(rejplus);
    freepropertylist(drs);
    freespace(ftmp);

    /* Normalise the sky frame to zero median */

    data = cpl_image_get_data_float(skyimg);
    npts = cpl_image_get_size_x(skyimg)*cpl_image_get_size_y(skyimg);
    vircam_qmedsig(data,NULL,npts,5.0,1,-1000.0,65535.0,&med,&sig);
    for (i = 0; i < npts; i++)
	data[i] -= med;

    /* Subtract the normalised sky frame from the science images */

    for (i = 0; i < ps.nscience; i++) {
	ff = ps.sci_fits[i];
	fim = vircam_fits_get_image(ff);
	if (vircam_fits_get_status(ff) != VIR_FATAL) {
	    drs = vircam_fits_get_ehu(ff);
	    cpl_image_subtract(fim,skyimg);	    
            cpl_propertylist_update_bool(drs,"ESO DRS SKYCOR",TRUE);
            cpl_propertylist_set_comment(drs,"ESO DRS SKYCOR",
					 "Image has been sky corrected");
	}
    }

    /* Clean up and get out of here */

    cpl_image_delete(skyimg);
    return;
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_get_readnoise_gain
    \par Purpose:
        Get a readnoise and gain estimate
    \par Description:
        Get a readnoise and gain estimate. If one doesn't exist, then fill
	one in.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_get_readnoise_gain(int jext, float *readnoise, 
					  float *gain) {
    cpl_propertylist *p_rg;
    const char *fctid = "vircam_jmp_get_readnoise_gain";

    /* Load the propertylist */

    p_rg = cpl_propertylist_load(cpl_frame_get_filename(ps.readgain_file),
				 jext);

    /* Check the readnoise property type and read it */

    switch (cpl_propertylist_get_type(p_rg,"ESO QC READNOISE")) {
    case CPL_TYPE_FLOAT:
	*readnoise = cpl_propertylist_get_float(p_rg,"ESO QC READNOISE");
	break;
    case CPL_TYPE_DOUBLE:
	*readnoise = (float)cpl_propertylist_get_double(p_rg,
							"ESO QC READNOISE");
	break;
    default:
	cpl_error_reset();
	*readnoise = 25.0;
	cpl_msg_error(fctid,"Unable to get READNOISE estimate, guessing %g\n",
		      *readnoise);
    }

    /* Now the gain */

    switch (cpl_propertylist_get_type(p_rg,"ESO QC GAIN")) {
    case CPL_TYPE_FLOAT:
	*gain = cpl_propertylist_get_float(p_rg,"ESO QC GAIN");
	break;
    case CPL_TYPE_DOUBLE:
	*gain = (float)cpl_propertylist_get_double(p_rg,"ESO QC GAIN");
	break;
    default:
	cpl_error_reset();
	*gain = 1.0;
	cpl_msg_error(fctid,"Unable to get GAIN estimate, guessing %g\n",
		      *gain);
    }
    cpl_propertylist_delete(p_rg);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_illum
    \par Purpose:
        Do the illumination correction calculation
    \par Description:
        Work out the illumination corrections for a series of images
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_illum(void) {
    int ngood,i,status,ii;
    float illcor_rms;
    vir_fits **ftmp,*ff,*ffc;
    char filt[32];
    cpl_table **mstds,*stdscat,*ms,*illcor,*ot;
    vir_tfits *outtab;
    cpl_propertylist **pl,*phu,*ehu;
    const char *fctid = "vircam_jmp_illum";

    /* Set some default values */

    ps.illcor = NULL;
    
    /* Sort out all images with good status */

    ftmp = cpl_malloc(ps.nscience*sizeof(vir_fits *));
    ngood = 0;
    for (i = 0; i < ps.nscience; i++) {
	ff = ps.sci_fits[i];
	if (vircam_fits_get_status(ff) != VIR_FATAL)
	    ftmp[ngood++] = ff;
    }

    /* If there aren't any good images, then get out of here now */

    if (ngood == 0) {
	freespace(ftmp);
	cpl_msg_error(fctid,"Illumination correction impossible. No good science frames available");
	return;
    }

    /* What filter is this? */

    if (vircam_pfits_get_filter(vircam_fits_get_phu(ftmp[0]),filt) != VIR_OK) {
	cpl_msg_error(fctid,"No filter name in stack header");
	freespace(ftmp);
	return;
    }

    /* Get some workspace for the various arrays you need for the 
       illumination correction routine */

    mstds = cpl_malloc(ngood*sizeof(cpl_table *));
    for (i = 0; i < ngood; i++)
	mstds[i] = NULL;
    pl = cpl_malloc(ngood*sizeof(cpl_propertylist *));
    for (i = 0; i < ngood; i++)
	pl[i] = NULL;

    /* For each of the good input frames, do a catalogue generation and
       get some matched standards */

    ffc = ps.fconf;
    for (i = 0; i < ngood; i++) {
	status = VIR_OK;
	ff = ftmp[i];
        (void)vircam_imcore(ff,ffc,vircam_jmp_config.ipix,
			    1.5*vircam_jmp_config.threshold,0,
			    vircam_jmp_config.rcore,vircam_jmp_config.nbsize,2,
			    3.5,&outtab,&status);
	pl[i] = cpl_propertylist_duplicate(vircam_tfits_get_ehu(outtab));

	/* Get some standard stars */

	(void)vircam_getstds(vircam_fits_get_ehu(ff),1,current_catpath,
			     current_cat,&stdscat,&status);
	if (status == VIR_FATAL) {
	    freetfits(outtab);
	    freespace(ftmp);
	    for (ii = 0; ii < ngood; ii++) {
		freetable(mstds[ii]);
		freepropertylist(pl[ii]);
	    }
	    freespace(mstds);
	    freespace(pl);
	    cpl_msg_error(fctid,"Illumination correction fails");
	    return;
	} else if (status == VIR_WARN) {
	    freetfits(outtab);
	    continue;
	}

	/* Now match this against the catalogue */

	ot = vircam_tfits_get_table(outtab);
	(void)vircam_matchstds(ot,stdscat,10.0,&ms,&status);
	if (status == VIR_FATAL) {
 	    freetable(stdscat);
	    freetfits(outtab);
	    freespace(ftmp);
	    for (ii = 0; ii < ngood; ii++) {
		freetable(mstds[ii]);
		freepropertylist(pl[ii]);
	    }
	    freespace(mstds);
	    freespace(pl);
	    cpl_msg_error(fctid,"%s",cpl_error_get_message());
	    return;
	}
	mstds[i] = ms;
	freetfits(outtab);
	freetable(stdscat);
    }
    
    /* Call the illumination routine */

    status = VIR_OK;
    (void)vircam_illum(ftmp,mstds,pl,ngood,filt,ps.tphottab,128,&illcor,
		       &illcor_rms,&status);
    
    /* Wrap the result */
    
    phu = cpl_propertylist_duplicate(vircam_fits_get_phu(ftmp[0]));
    ehu = cpl_propertylist_duplicate(vircam_fits_get_ehu(ftmp[0]));
    ps.illcor = vircam_tfits_wrap(illcor,NULL,phu,ehu);
    cpl_propertylist_update_float(ehu,"ESO QC ILLUMCOR_RMS",illcor_rms);
    cpl_propertylist_set_comment(ehu,"ESO QC ILLUMCOR_RMS",
				 "RMS of illumination correction map");
    
    /* Tidy up */

    for (i = 0; i < ngood; i++) {
	freetable(mstds[i]);
	freepropertylist(pl[i]);
    }
    freespace(mstds);
    freespace(pl);
    freespace(ftmp);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_outfile
    \par Purpose:
        Create an output product file name
    \par Description:
        Create an output product file name given a basename and index and
	a flag to indicate if it's a FITS file or not.
    \par Language:
        C
    \param bname
        The basename of the output file.
    \param ind
        An index number (>= 0)
    \param isfits
        If set, then a .fits extension is added to the filename
    \return
        The output file name.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

static char *vircam_jmp_outfile(const char *bname, int ind, int isfits) {
    int nf;
    char *fname;

    /* Count up how much space you need for the output string. The 'isfits'
       block accounts for ".fits", an underscore and an EOS*/

    nf = strlen(bname);
    if (ind == 0) 
	nf++;
    else 
	nf += ((int)log10((double)ind)+1);
    if (isfits) 
	nf += 7;
    else
	nf += 2;

    /* Get the space for the filename */

    fname = cpl_malloc(nf);

    /* Now write the name */

    if (isfits) 
        (void)snprintf(fname,nf,"%s_%d.fits",bname,ind);
    else
	(void)snprintf(fname,nf,"%s_%d",bname,ind);
    return(fname);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_init
    \par Purpose:
        Initialise pointers in the global memory structure.
    \par Description:
        Initialise pointers in the global memory structure.
    \par Language:
        C
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_init(void) {

    /* Level 0 stuff */

    ps.labels = NULL;
    ps.master_dark = NULL;
    ps.master_twilight_flat = NULL;
    ps.master_conf = NULL;
    ps.mask = NULL;
    ps.chantab = NULL;
    ps.phottab = NULL;
    ps.tphottab = NULL;
    ps.readgain_file = NULL;
    ps.science_frames = NULL;
    ps.product_frames_simple = NULL;
    ps.product_frames_super = NULL;
    ps.product_frames_superc = NULL;
    ps.product_frame_stack = NULL;
    ps.product_frame_stackc = NULL;
    ps.product_frame_cat = NULL;
    ps.product_frame_illcor = NULL;
    ps.phupaf = NULL;
    ps.gaincors = NULL;
    ps.catpath = NULL;
    ps.catname = NULL;
    ps.catpath2 = NULL;
    ps.catname2 = NULL;

    /* Level 1 stuff */

    ps.fdark = NULL;
    ps.fflat = NULL;
    ps.fconf = NULL;
    ps.fchantab = NULL;
    ps.nscience = 0;
    ps.sci_fits = NULL;
    ps.nustep_sets = 0;
    ps.ustep_sets = NULL;
    ps.ndith = 0;
    ps.ndithc = 0;
    ps.dith_input = NULL;
    ps.dithc_input = NULL;
    ps.stack_frame = NULL;
    ps.stackc_frame = NULL;
    ps.outcat = NULL;
    ps.illcor = NULL;
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_jmp_tidy
    \par Purpose:
        Free allocated workspace in the global memory structure
    \par Description:
        Free allocated workspace in the global memory structure. The tidy works
	on two levels. Level 1 is for items that are usually cleared up after
	each extension is processed. Level 2 is for cleaning up the whole
	recipe
    \par Language:
        C
    \param level
        The level of the tidy to be done. 1: Tidy up after finishing an 
	extension, 2: Tidy up after finishing the recipe.
    \return
        Nothing
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_jmp_tidy(int level) {
    int i;

    /* Level 1 stuff */

    freefits(ps.fdark); 
    freefits(ps.fflat); 
    freefits(ps.fconf);
    freetfits(ps.fchantab);
    freefitslist(ps.sci_fits,ps.nscience);
    ps.nscience = 0;
    for (i = 0; i < ps.nustep_sets; i++) {
	freespace(ps.ustep_sets[i].f);
	freefits(ps.ustep_sets[i].super);
	freefits(ps.ustep_sets[i].superc);
    }
    freespace(ps.ustep_sets);
    ps.nustep_sets = 0;
    freespace(ps.dith_input);
    ps.ndith = 0;
    freespace(ps.dithc_input);
    ps.ndithc = 0;

    freefits(ps.stack_frame);
    freefits(ps.stackc_frame);
    freetfits(ps.outcat);
    freetable(ps.matchstds);
    freetfits(ps.illcor);

    if (level == 1)
	return;
    
    /* Level 0 stuff */

    freespace(ps.labels);
    freeframe(ps.master_dark);
    freeframe(ps.master_twilight_flat);
    freeframe(ps.master_conf);
    freemask(ps.mask);
    freeframe(ps.chantab);
    freeframe(ps.phottab);
    freeframe(ps.readgain_file);
    freetable(ps.tphottab);
    freeframeset(ps.science_frames);
    freepropertylist(ps.phupaf);
    freespace(ps.product_frames_simple); /* NB: We only have to delete the */
    freespace(ps.product_frames_super);  /* arrays and not the frames */
    freespace(ps.product_frames_superc); /* as these get passed back to esorex */
    freespace(ps.gaincors);
    freespace(ps.catpath);
    freespace(ps.catname);
    freespace(ps.catpath2);
    freespace(ps.catname2);
}

/**@}*/

/*

$Log: vircam_jmp_utils.c,v $
Revision 1.33  2008/07/10 13:05:53  jim
Modified to use v4.2 version of cpl_wcs

Revision 1.32  2008/06/20 11:13:35  jim
Fixed dodgy call to cpl_wcs_get_image_dims

Revision 1.31  2008/05/06 08:40:10  jim
Modified to use cpl_wcs interface

Revision 1.30  2007/11/22 12:34:08  jim
Added line to vircam_jmp_save_illum to make sure that phupaf is defined
when _save_simple isn't called

Revision 1.29  2007/10/25 17:34:00  jim
Modified to remove lint warnings

Revision 1.28  2007/10/19 09:25:10  jim
Fixed problems with missing includes

Revision 1.27  2007/10/19 06:55:06  jim
Modifications made to use new method for directing the recipes to the
standard catalogues using the sof

Revision 1.26  2007/10/15 12:50:28  jim
Modified for compatibility with cpl_4.0

Revision 1.25  2007/06/13 08:10:49  jim
Modified to allow for different output file names depending upon the calling
recipe

Revision 1.24  2007/05/15 08:54:07  jim
Fixed small bug in stack_save and dither_offsets. Also modified dither_offsets
to just send back zeros if there is only one dither frame

Revision 1.23  2007/05/08 10:41:49  jim
Added gaincor variables

Revision 1.22  2007/05/02 09:15:37  jim
Modified to use new api for vircam_imcore and vircam_platesol

Revision 1.21  2007/04/30 09:40:01  jim
Added vircam_paf_append

Revision 1.20  2007/04/13 12:29:11  jim
Fixed bug in save_simple which mean that ps.pafphu was allocated for every
file rather than just the first one

Revision 1.19  2007/04/04 16:05:59  jim
Modified to make paf information a bit more correct

Revision 1.18  2007/04/04 10:34:55  jim
Modified to use new dfs tags

Revision 1.17  2007/03/14 22:08:54  jim
Fixed typo

Revision 1.16  2007/03/14 14:49:13  jim
Fixed problem with missing paf files in jmp recipes if detlive = F. Also
fixed problem where extra dummy products were being created

Revision 1.15  2007/03/13 09:50:53  jim
Fixed bug in vircam_jmp_save_illum where PAF wasn't being saved for the
first extension

Revision 1.14  2007/03/06 14:06:24  jim
Fixed missing paf write

Revision 1.13  2007/03/06 13:49:48  jim
Created static routine vircam_jmp_outfile to create a 'predictable'
output filename for each product

Revision 1.12  2007/03/01 12:42:42  jim
Modified slightly after code checking

Revision 1.11  2007/02/25 06:34:20  jim
Plugged memory leak

Revision 1.10  2007/02/15 06:59:38  jim
Added ability to write QC paf files

Revision 1.9  2007/02/06 13:11:12  jim
Fixed entry for PRO dictionary in cpl_dfs_set_product_header

Revision 1.8  2007/01/17 23:54:00  jim
Plugged some memory leaks

Revision 1.7  2006/12/19 13:30:01  jim
Fixed vircam_jmp_illum to initialise pointers

Revision 1.6  2006/12/18 12:51:20  jim
Tightened up some of the error reporting

Revision 1.5  2006/12/15 09:58:28  jim
read noise and gain keywords were wrong...

Revision 1.4  2006/11/29 12:28:45  jim
Modified so that the correct recipe names would appear in the headers of
data products

Revision 1.3  2006/11/28 22:10:22  jim
Added illcor_rms to calling sequence of vircam_illum

Revision 1.2  2006/11/28 20:56:31  jim
Added vircam_jmp_illum and vircam_jmp_save_illum

Revision 1.1  2006/11/27 11:54:37  jim
Initial entry


*/

