/* $Id: vircam_matchstds.c,v 1.9 2007/07/09 13:22:09 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/07/09 13:22:09 $
 * $Revision: 1.9 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"

/* Function prototypes */

static int vircam_matchstds_create(cpl_plugin *);
static int vircam_matchstds_exec(cpl_plugin *);
static int vircam_matchstds_destroy(cpl_plugin *);
static int vircam_matchstds_test(cpl_parameterlist *, cpl_frameset *);
static int vircam_matchstds_save(cpl_frameset *framelist,
				 cpl_parameterlist *parlist);
static void vircam_matchstds_init(void);
static void vircam_matchstds_tidy(void);

static struct {

    /* Input */

    int         extenum;

} vircam_matchstds_config;


static struct {
    int         *labels;
    cpl_frame   *cat;
    cpl_frame   *stds;
    vir_tfits   *catf;
    vir_tfits   *stdsf;
    cpl_table   *outtab;
} ps;

static int isfirst;
static cpl_frame *product_frame = NULL;

static char vircam_matchstds_description[] =
"vircam_matchstds -- VIRCAM recipe to test vircam_matchstds.\n\n"
"Match a catalogue with a table of standards\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s An input catalogue of objects extracted from an image\n"
"    %-21s An input catalogue of standard stars\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_matchstds vircam_matchstds
    \brief Test recipe to drive the vircam_matchstds library function.

    \par Name:
        vircam_matchstds
    \par Purpose:
        Test recipe to drive the vircam_matchstds library function.
    \par Description:
        Test the library function vircam_matchstds by matching an object
	catalogue with a standard star catalogue
    \par Language:
        C
    \par Parameters:
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b OBJECT_CATALOGUE (required): An object catalogue
        - \b STANDARDS_TABLE (required): A standards table
    \par Output File Types:
        The following lists the output data frames that are generated by 
        this recipe. The word in bold gives the PRO CATG keyword value for 
        each.
        - \b MATCHED_STANDARDS_TABLE: A matched standards table
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input tables
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_matchstds.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_matchstds_description,
		   VIRCAM_CAL_OBJCAT,VIRCAM_CAL_STDTAB);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_matchstds",
                    "VIRCAM catalogue and standards matching test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_matchstds_create,
                    vircam_matchstds_exec,
                    vircam_matchstds_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchstds_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_matchstds.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_matchstds",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchstds_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_matchstds_test(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchstds_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchstds_test(cpl_parameterlist *parlist, 
				 cpl_frameset *framelist) {
    const char *fctid="vircam_matchstds";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_matchstds_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_matchstds.extenum");
    vircam_matchstds_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_matchstds_tidy();
        return(-1);
    }

    /* Get the frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_matchstds_tidy();
        return(-1);
    }
    if ((ps.cat = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					     VIRCAM_CAL_OBJCAT)) == NULL) {
        cpl_msg_info(fctid,"No object catalogue found -- cannot continue");
	vircam_matchstds_tidy();
	return(-1);
    }
    if ((ps.stds = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
                                              VIRCAM_CAL_STDTAB)) == NULL) {
        cpl_msg_info(fctid,"No standards catalogue found -- cannot continue");
	vircam_matchstds_tidy();
	return(-1);
    }

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_matchstds_config.extenum,
		       (const cpl_frame *)ps.cat,&jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_matchstds_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
        isfirst = (j == jst);

        /* Load up the tables */

        ps.catf = vircam_tfits_load(ps.cat,j);
        ps.stdsf = vircam_tfits_load(ps.stds,j);
        if (ps.stdsf == NULL || ps.catf == NULL) {
	    freetfits(ps.catf);
	    freetfits(ps.stdsf);
	    cpl_msg_info(fctid,"No matching possible");
	    continue;
	}

	/* Now do the correction */

	cpl_msg_info(fctid,"Doing the matching for extension %d",j);
        (void)vircam_matchstds(vircam_tfits_get_table(ps.catf),
			       vircam_tfits_get_table(ps.stdsf),10.0,
			       &(ps.outtab),&status);
	if (status != VIR_OK) {
	    cpl_msg_info(fctid,"No matching done");
	    status = VIR_OK;
	}

        /* Now save the result */

	cpl_msg_info(fctid,"Saving results for extension %d",j);
	if (vircam_matchstds_save(framelist,parlist) != 0) 
	    cpl_msg_info(fctid,"No matching saved");


	/* Tidy a few things before the next image */

	freetfits(ps.catf);
	freetfits(ps.stdsf);
	freetable(ps.outtab);
    }
    vircam_matchstds_tidy();
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchstds_save(cpl_frameset *framelist, 
				 cpl_parameterlist *parlist) {
    const char *recipeid = "vircam_matchstds";
    const char *fctid = "vircam_matchstds_save";
    const char *outfile = "matchstds.fits";
    cpl_propertylist *plist,*elist;

    /* Create the output table. First see if you need a primary */

    if (isfirst) {

        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_MSTDTAB);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_TABLE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

        /* Fiddle with the header new */

        plist = vircam_tfits_get_phu(ps.catf);
        vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
					      parlist,(char *)recipeid,
                                              "?Dictionary?");

        /* Get the extension header and tack the extra header items onto it. */

        elist = vircam_tfits_get_ehu(ps.catf);
        vircam_dfs_set_product_exten_header(elist,product_frame,framelist,
					    parlist,(char *)recipeid,
					    "?Dictionary?");

        /* 'Save' the PHU and create a table extension */

        if (cpl_table_save(ps.outtab,plist,elist,outfile,CPL_IO_DEFAULT)
            != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product table");
            cpl_frame_delete(product_frame);
            return(-1);
        }
        cpl_frameset_insert(framelist,product_frame);

    /* Otherwise save the next extension */

    } else {

        /* Get the extension header and tack the extra header items onto it. */

        elist = vircam_tfits_get_ehu(ps.catf);
        vircam_dfs_set_product_exten_header(elist,product_frame,framelist,
					    parlist,(char *)recipeid,
					    "?Dictionary?");

	/* Sae the table */

        if (cpl_table_save(ps.outtab,NULL,elist,outfile,CPL_IO_EXTEND)
            != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product table");
            return(-1);
        }
    }

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_matchstds_init(void) {
    ps.labels = NULL;
    ps.cat = NULL;
    ps.catf = NULL;
    ps.stds = NULL;
    ps.stdsf = NULL;
    ps.outtab = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_matchstds_tidy(void) {
    freespace(ps.labels);
    freetfits(ps.catf);
    freetfits(ps.stdsf);
    freeframe(ps.stds);
    freeframe(ps.cat);
    freetable(ps.outtab);
}

/**@}*/


/*

$Log: vircam_matchstds.c,v $
Revision 1.9  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.8  2007/04/23 12:49:07  jim
Changed behaviour for error condition

Revision 1.7  2007/04/13 12:27:39  jim
Added some extra docs

Revision 1.6  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.5  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.4  2006/06/15 09:58:59  jim
Minor changes to docs

Revision 1.3  2006/05/04 11:53:43  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.2  2006/04/27 14:22:05  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:45  jim
New routine


*/




