/* $Id: vircam_gaincor.c,v 1.2 2007/10/25 17:34:00 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2007 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 17:34:00 $
 * $Revision: 1.2 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "vircam_mods.h"
#include "vircam_utils.h"
#include "vircam_fits.h"

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \ingroup reductionmodules
    \brief Gain correct input data frame

    \par Name:
        vircam_gaincor
    \par Purpose:
        Correct input data for inter-detector gain difference
    \par Description:
        An input image is scaled by a constant value so as to put all the 
	images in a single observation onto the same gain scale.
    \par Language:
        C
    \param infile 
        The input data image (overwritten by result).
    \param gainscl 
        The input scale factor
    \param status 
        An input/output status that is the same as the returned values below. 
    \retval VIR_OK 
        if everything is ok
    \retval VIR_WARN 
        if the scale factor is zero
    \retval VIR_FATAL 
        if image data fails to load
    \par QC headers:
        None
    \par DRS headers:
        The following DRS keywords are written to the infile extension header
        - \b GAINCOR
            The gain correction factor
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_gaincor(vir_fits *infile, float gainscl, int *status) {
    cpl_image *i;
    cpl_propertylist *oplist;

    /* Inherited status */

    if (*status != VIR_OK)
	return(*status);

    /* Get the images and check the dimensions of each */

    i = vircam_fits_get_image(infile);

    /* If the factor is zero, then send a warning out, but continue on. */

    if (gainscl <= 0.0) {
	WARN_CONTINUE
    } else {
	cpl_image_multiply_scalar(i,gainscl);
    }

    /* Now put some stuff in the DRS extension... */

    oplist = vircam_fits_get_ehu(infile);
    if (oplist != NULL) {
        cpl_propertylist_update_float(oplist,"ESO DRS GAINCOR",gainscl);
        cpl_propertylist_set_comment(oplist,"ESO DRS GAINCOR",
				     "Gain correction factor");
    } else 
	WARN_CONTINUE

    /* Get out of here */

    GOOD_STATUS
}

/**@}*/

/*

$Log: vircam_gaincor.c,v $
Revision 1.2  2007/10/25 17:34:00  jim
Modified to remove lint warnings

Revision 1.1  2007/05/08 10:40:15  jim
Initial entry


*/
