/* $Id: vircam_defringe.c,v 1.6 2007/10/15 12:53:55 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/15 12:53:55 $
 * $Revision: 1.6 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"

/* Function prototypes */

static int vircam_defringe_create(cpl_plugin *) ;
static int vircam_defringe_exec(cpl_plugin *) ;
static int vircam_defringe_destroy(cpl_plugin *) ;
static int vircam_defringe_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_defringe_save(cpl_frameset *framelist,
				cpl_parameterlist *parlist);
static int vircam_defringe_lastbit(int jext, cpl_frameset *framelist,
				   cpl_parameterlist *parlist);
static void vircam_defringe_init(void);
static void vircam_defringe_tidy(int level);

static struct {

    /* Input */

    int         nbsize;
    int         extenum;

} vircam_defringe_config;

static struct {
    int         *labels;
    cpl_frame   *fringe;
    cpl_frame   *img;
    vir_fits    *fringef;
    vir_fits    *imgf;
    vir_mask    *bpm;
} ps;

static int isfirst;
static int dummy;
static cpl_frame *product_frame = NULL;

static char vircam_defringe_description[] =
"vircam_defringe -- VIRCAM defringe test recipe.\n\n"
"Defringe an input frame using a master fringe frame\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A input uncorrected image\n"
"    %-21s A master fringe frame\n"
"    %-21s Optional master bad pixel map or\n"
"    %-21s Optional master confidence map\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_defringe vircam_defringe
    \brief Test recipe to drive the vircam_defringe library function.

    \par Name:
        vircam_defringe
    \par Purpose:
        Test recipe to drive the vircam_defringe library function.
    \par Description:
        Test the library function vircam_defringe by defringing a frame
	that has already been dark and flat corrected.
    \par Language:
        C
    \par Parameters:
        - \b nbsize (float): The size of a smoothing box used for modelling
	  out large scale background variations
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b SCIENCE_IMAGE (required): A single raw image.
        - \b MASTER_FRINGE (required): A master fringe frame.
        - \b MASTER_BPM or \b MASTER_CONF (optional): If either is given, then
             it will be used to mask out bad pixels during and statistical
             analysis that is done.
    \par Output File Types:
        The following lists the output data frames that are generated by 
	this recipe. The word in bold gives the PRO CATG keyword value for 
	each.
	- \b SIMPLE_IMAGE_TEST: A fringe corrected image.
    \par Fatal Error Conditions:
        - Missing input image or fringe frame.
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_defringe.c
*/


/* Function code */


/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_defringe_description,
		   VIRCAM_TEST_SCIENCE_RAW,VIRCAM_CAL_FRINGE,VIRCAM_CAL_BPM,
		   VIRCAM_CAL_CONF);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_defringe",
                    "VIRCAM fringe correction test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_defringe_create,
                    vircam_defringe_exec,
                    vircam_defringe_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_defringe_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Fill in the parameters. First the background cell size */

    p = cpl_parameter_new_value("vircam.vircam_defringe.nbsize",
                                CPL_TYPE_INT,
                                "Cell size in pixels",
                                "vircam.vircam_defringe",128);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"nbsize");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_defringe.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_defringe",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_defringe_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_defringe_test(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_defringe_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_defringe_test(cpl_parameterlist *parlist, 
				cpl_frameset *framelist) {
    const char *fctid="vircam_defringe";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j,retval,nx,ny;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_defringe_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_defringe.nbsize");
    vircam_defringe_config.nbsize = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_defringe.extenum");
    vircam_defringe_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_defringe_tidy(2);
        return(-1);
    }

    /* Get the frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_defringe_tidy(2);
        return(-1);
    }
    if ((ps.img = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
                                             VIRCAM_TEST_SCIENCE_RAW)) == NULL) {
        cpl_msg_info(fctid,"No raw image found -- cannot continue");
	vircam_defringe_tidy(2);
	return(-1);
    }
    if ((ps.fringe = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
						VIRCAM_CAL_FRINGE)) == NULL) {
        cpl_msg_info(fctid,"No fringe image found -- cannot continue");
	vircam_defringe_tidy(2);
	return(-1);
    }

    /* Get the master mask */

    ps.bpm = vircam_mask_define(framelist,ps.labels,nlab);

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_defringe_config.extenum,(const cpl_frame *)ps.img,
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_defringe_tidy(2);
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
        isfirst = (j == jst);

        /* Load up the images */

        ps.imgf = vircam_fits_load(ps.img,CPL_TYPE_FLOAT,j);
        ps.fringef = vircam_fits_load(ps.fringe,CPL_TYPE_FLOAT,j);
        if (ps.img == NULL || ps.fringef == NULL) {
	    cpl_msg_info(fctid,"Extension %d frame wouldn't load",j);
	    dummy = 1;
	    retval = vircam_defringe_lastbit(j,framelist,parlist);
	    if (retval != 0)
		return(-1);
	}

        /* Get the size of the flat image */
        
	nx = cpl_image_get_size_x(vircam_fits_get_image(ps.imgf));
	ny = cpl_image_get_size_y(vircam_fits_get_image(ps.imgf));

        /* Load the data for the bpm */

        (void)vircam_mask_load(ps.bpm,j,nx,ny);

	/* Now do the correction */

	cpl_msg_info(fctid,"Doing the fringe correction for extension %d",j);
        (void)vircam_defringe(&(ps.imgf),1,&(ps.fringef),1,ps.bpm,
			      vircam_defringe_config.nbsize,&status);
	if (status != VIR_OK) {
	    cpl_msg_info(fctid,"Extension %d defringe failed",j);
	    dummy = 1;
	    retval = vircam_defringe_lastbit(j,framelist,parlist);
	    if (retval != 0) 
 	        return(-1);
	}

        /* Now save the result */

	retval = vircam_defringe_lastbit(j,framelist,parlist);
	if (retval != 0)
	    return(-1);
    }
    vircam_defringe_tidy(2);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_defringe_save(cpl_frameset *framelist, 
				cpl_parameterlist *parlist) {
    const char *fctid = "vircam_defringe_save";
    const char *outfile = "defringe.fits";
    const char *recipeid = "vircam_defringe";
    cpl_propertylist *plist;

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_SIMPLE_TEST);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

	/* Set up the header */

        plist = vircam_fits_get_phu(ps.imgf);
        vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
                                              parlist,(char *)recipeid,
                                              "?Dictionary?");

        /* 'Save' the PHU image */

        if (cpl_image_save(NULL,outfile,CPL_BPP_8_UNSIGNED,plist,
                           CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product PHU");
            cpl_frame_delete(product_frame);
            return(-1);
        }
        cpl_frameset_insert(framelist,product_frame);
    }

    /* Get the extension property list */

    plist = vircam_fits_get_ehu(ps.imgf);

    /* Fiddle with the header now */

    vircam_dfs_set_product_exten_header(plist,product_frame,framelist,parlist,
					(char *)recipeid,"?Dictionary?");
    if (dummy)
	vircam_dummy_property(plist);

    /* Save the image */

    if (cpl_image_save(vircam_fits_get_image(ps.imgf),outfile,CPL_BPP_IEEE_FLOAT,
		       plist,CPL_IO_EXTEND) != CPL_ERROR_NONE) {
        cpl_msg_error(fctid,"Cannot save product image extension");
        return(-1);
    }

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Do make dummys and save
  @param    jext         the extension number
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_defringe_lastbit(int jext, cpl_frameset *framelist,
				   cpl_parameterlist *parlist) {
    int retval;
    const char *fctid="vircam_defringe_lastbit";

    /* Save everything */

    cpl_msg_info(fctid,"Saving products for extension %d",jext);
    retval = vircam_defringe_save(framelist,parlist);
    if (retval != 0) {
	vircam_defringe_tidy(2);
	return(-1);
    }

    /* Free some stuff up */

    vircam_defringe_tidy(1);
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_defringe_init(void) {
    ps.labels = NULL;
    ps.fringe = NULL;
    ps.fringef = NULL;
    ps.img = NULL;
    ps.imgf = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_defringe_tidy(int level) {
    freefits(ps.imgf);
    freefits(ps.fringef);
    vircam_mask_clear(ps.bpm);
    if (level == 1)
	return;
    freemask(ps.bpm);
    freeframe(ps.img);
    freeframe(ps.fringe);
    freespace(ps.labels);
}

/**@}*/

/*

$Log: vircam_defringe.c,v $
Revision 1.6  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.5  2007/07/09 13:22:08  jim
Modified to use new version of vircam_exten_range

Revision 1.4  2007/04/13 12:27:38  jim
Added some extra docs

Revision 1.3  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.2  2007/03/01 12:42:58  jim
Modified slightly after code checking

Revision 1.1  2006/12/06 13:00:04  jim
Initial entry


*/




