/* $Id: vircam_destripe.c,v 1.7 2007/10/25 17:34:00 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 17:34:00 $
 * $Revision: 1.7 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <cxtypes.h>
#include <math.h>

#include "vircam_mods.h"
#include "vircam_utils.h"
#include "vircam_stats.h"
#include "vircam_mask.h"

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \ingroup reductionmodules
    \brief Remove stripes from the background of an image

    \par Name:
        vircam_destripe
    \par Purpose:
        Remove stripes from the background of an image
    \par Description:
        Remove stripes from the background of an image by finding the median
	of each row and putting the value into a profile. Normalise the profile
	to zero median. Then subtract the normalised median for each row from
	the original data.
    \par Language:
        C
    \param in
        The input image. This is corrected in-situ.
    \param inbpm
        The input image bad pixel mask.
    \param status
        An input/output status that is the same as the returned values below.
    \retval VIR_OK
        if everything is ok
    \par QC headers:
        None
    \par DRS headers:
        The following DRS keywords are written to the plist propertylist
        - \b STRIPECOR
            A flag saying that the stripe correction has been done
        - \b STRIPERMS
            The RMS of the stripe profile
    \par Other headers:
        None
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_destripe(vir_fits *in, vir_mask *inbpm, int *status) {
    float *data,*d,medprofile,profilerms,*wptr,val;
    unsigned char *bpm,*b,*rb;
    long i,j,nx,ny;
    cpl_propertylist *plist;

    /* Inherited status */

    if (*status != VIR_OK)
        return(*status);

    /* Get the data array for the input image and the bad pixel mask */

    data = cpl_image_get_data(vircam_fits_get_image(in));
    if (inbpm != NULL) 
	bpm = vircam_mask_get_data(inbpm);
    else
	bpm = NULL;

    /* Get the data array size */

    nx = (long)cpl_image_get_size_x(vircam_fits_get_image(in));
    ny = (long)cpl_image_get_size_y(vircam_fits_get_image(in));

    /* Get some workspace to hold the 1d profile */

    wptr = cpl_malloc(nx*sizeof(*wptr));
    rb = cpl_calloc(nx,sizeof(*rb));

    /* Loop for each row */

    d = data;
    b = bpm;
    for (i = 0; i < ny; i++) {

	/* Get the median for that row, ignoring any bad pixels */

	val = vircam_med(d,b,nx);
	if (val == CX_MAXFLOAT) {
     	    wptr[i] = 0.0;
	    rb[i] = 1;
	} else {
     	    wptr[i] = val;
        }
	d += nx;
	if (b != NULL)
	    b += nx;
    }

    /* Get the median of the profile and normalise the profile 
       to zero median */

    vircam_medsig(wptr,rb,ny,&medprofile,&profilerms);
    for (i = 0; i < ny; i++) {
        if (rb[i]) {
	    wptr[i] = 0.0;
        } else {
	    wptr[i] -= medprofile;
        }
    }
    cpl_free(rb);

    /* Now do the correction */

    d = data;
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) 
	    d[j] -= wptr[i];
	d += nx;
    }

    /* Store the RMS of the profile away */

    plist = vircam_fits_get_ehu(in);
    cpl_propertylist_update_bool(plist,"ESO DRS STRIPECOR",TRUE);
    cpl_propertylist_set_comment(plist,"ESO DRS STRIPECOR",
				 "Stripe correction done");
    cpl_propertylist_update_float(plist,"ESO DRS STRIPERMS",profilerms);
    cpl_propertylist_set_comment(plist,"ESO DRS STRIPERMS",
				 "RMS of the removed stripe profile");

    /* Ditch the workspace and get out of here */

    freespace(wptr);
    GOOD_STATUS
}
	

/**@}*/


/*

$Log: vircam_destripe.c,v $
Revision 1.7  2007/10/25 17:34:00  jim
Modified to remove lint warnings

Revision 1.6  2007/09/07 10:45:10  jrl
Fixed bug which arises if an entire row is flagged bad

Revision 1.5  2007/03/29 12:19:39  jim
Little changes to improve documentation

Revision 1.4  2007/03/01 12:42:41  jim
Modified slightly after code checking

Revision 1.3  2006/11/27 12:09:35  jim
Tidied up some docs. Also modified definition of DRS STRIPECOR so that it's
now a boolean

Revision 1.2  2006/11/10 10:27:56  jim
Added STRIPECOR header parameter

Revision 1.1  2006/10/02 13:43:31  jim
new file


*/
