/* $Id: vircam_detector_noise.c,v 1.48 2007/10/25 18:39:22 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 18:39:22 $
 * $Revision: 1.48 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>
#include <math.h>
#include <string.h>

#include "vircam_utils.h"
#include "vircam_mask.h"
#include "vircam_dfs.h"
#include "vircam_stats.h"
#include "vircam_pfits.h"
#include "vircam_paf.h"

/* Function prototypes */

static int vircam_detector_noise_create(cpl_plugin *) ;
static int vircam_detector_noise_exec(cpl_plugin *) ;
static int vircam_detector_noise_destroy(cpl_plugin *) ;
static int vircam_detector_noise(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_detector_noise_save(cpl_frameset *framelist, 
				      cpl_parameterlist *parlist);
static void vircam_detector_noise_init(void);
static void vircam_detector_noise_tidy(void);

/* Static global variables */

static struct {

    /* Input */

    float       thresh;
    int         extenum;

    /* Output */

    float       gain;
    float       readnoise;
} vircam_detector_noise_config ;

static struct {
    int              *labels;
    cpl_frameset     *darklist;
    cpl_frameset     *domelist;
    cpl_frameset     *sorted;
    cpl_image        *darkim1;
    cpl_image        *darkim2;
    cpl_image        *domeim1;
    cpl_image        *domeim2;
    vir_mask         *master_mask;
    cpl_propertylist *ph;
    cpl_propertylist *eh;
    cpl_propertylist *phupaf;
} ps;

static cpl_frame *product_frame = NULL;
static int isfirst;
static int dummy;

#define BUZZ_OFF {vircam_detector_noise_tidy(); return(-1);}

static char vircam_detector_noise_description[] =
"vircam_detector_noise -- VIRCAM readnoise and gain recipe.\n\n"
"Measures the read noise and gain of a chip using two dome flat exposures\n"
"and two dark exposures. The flats should have the same illumination.\n"
"All four frames should have the same exposure time. If the SOF\n"
"contains more than two files of a given type, the others are ignored.\n\n"
"The program requires the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A list of two raw dark images\n"
"    %-21s A list of two raw dome flat image\n"
"    %-21s Optional master bad pixel map or\n"
"    %-21s Optional master confidence map\n"
"\n";

/**@{*/

/**
    \ingroup recipelist
    \defgroup vircam_detector_noise vircam_detector_noise
    \brief Work out the read noise and gain of a detector 

    \par Name:
        vircam_detector_noise
    \par Purpose:
        Work out the read noise and gain of a detector
    \par Description:
        The input dome flat exposures and dark exposures are used to 
        determine the read noise and gain of a detector. The method uses
        two exposures of each. The dome flats and the dark frames should be
	done with the same exposure times. A simple statistical method is
        used to work out the desired values. See section 2.4 of the VIRCAM
	Data Reduction Library Design document for further information.
    \par Language:
        C
    \par Parameters:
        - \b thr (float): The rejection threshold in numbers of sigmas.
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b DARK_GAIN (required): A list of two raw dark images. If any more
             are found in the list, they will be ignored.
        - \b FLAT_LAMP_GAIN (required): A list of two raw dome flat images. If 
             any more are found in the list, they will be ignored.
        - \b MASTER_BPM or \b MASTER_CONF (optional): A master bad pixel mask.
             If this is omitted then all pixels are assumed to be good.
    \par Output Products:
        The following list gives the output data products that are generated
        by this recipe.
        - An output readnoise/gain file (\b READGAIN_TABLE). The exact 
	  content of the columns is described in chapter 5 of the DRLD.
    \par Output QC Parameters:
        - \b READNOISE
            The readout noise of the detector in electrons
        - \b GAIN
            The gain of the detector in electrons per ADU.
    \par Notes
        None.
    \par Fatal Error Conditions:
        - NULL input frameset
        - Input frameset headers incorrect meaning that RAW and CALIB frame
          can't be distinguished
        - Insufficient dark or flat exposures
        - Unable to save data products
    \par Non-Fatal Error Conditions:
        - Missing bad pixel or confidence map. All pixels considered to be
	  good during the stats phase.
    \par Conditions Leading To Dummy Products:
        - Image extensions wouldn't load.
        - The detector for the current image extension is flagged dead
        - Unphysical statistical result
    \par Author:
        Jim Lewis, CASU
    \par Code Reference: 
        vircam_detector_noise.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_detector_noise_description,
		   VIRCAM_NOISE_DARK_RAW,VIRCAM_NOISE_FLAT_RAW,VIRCAM_CAL_BPM,
		   VIRCAM_CAL_CONF);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_detector_noise",
                    "VIRCAM recipe to determine readout noise and gain",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_detector_noise_create,
                    vircam_detector_noise_exec,
                    vircam_detector_noise_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_detector_noise_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Fill in the rejection threshold parameter */

    p = cpl_parameter_new_value("vircam.vircam_detector_noise.thresh",
                                CPL_TYPE_DOUBLE,
                                "Rejection threshold in sigma above background",                                "vircam.vircam_detector_noise",5.0);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"thresh");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_detector_noise.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_detector_noise",
                                1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_detector_noise_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_detector_noise_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_detector_noise(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_detector_noise(cpl_parameterlist *parlist,
				 cpl_frameset *framelist) {
    const char *fctid="vircam_detector_noise";
    cpl_parameter *p;
    int nlab,j,jst,jfn,retval,nx,ny,live;
    long nptsdark;
    float meandark1,meandome1,meandark2,meandome2,sigdark,lcut,hcut;
    float meandarkdiff,sigdarkdiff;
    float meandomediff,sigdomediff,gain,readnoise;
    cpl_frame *dark1,*dark2,*dome1,*dome2,*frame;
    cpl_propertylist *plist;
    unsigned char *bpm;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
	cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
	return(-1);
    }

    /* Initialise a few things */

    vircam_detector_noise_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_detector_noise.thresh");
    vircam_detector_noise_config.thresh = (float)cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist,"vircam.vircam_detector_noise.extenum");
    vircam_detector_noise_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
	cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
	vircam_detector_noise_tidy();
	return(-1);
    }

    /* Get dark and dome flat frames. Make sure there are at least 2 of each */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
					   &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frameset");
	BUZZ_OFF
    }
    if ((ps.darklist = vircam_frameset_subgroup(framelist,ps.labels,nlab,
						VIRCAM_NOISE_DARK_RAW)) == NULL) {
	cpl_msg_error(fctid,"Cannot find dark frames in input frameset");
	BUZZ_OFF
    }
    if (cpl_frameset_get_size(ps.darklist) < 2) {
	cpl_msg_error(fctid,"Dark frameset doesn't have enough frames");
	BUZZ_OFF
    }
    if ((ps.domelist = vircam_frameset_subgroup(framelist,ps.labels,nlab,
						VIRCAM_NOISE_FLAT_RAW)) == NULL) {
	cpl_msg_error(fctid,"Cannot find dome flat frames in input frameset");
	BUZZ_OFF
    }
    if (cpl_frameset_get_size(ps.domelist) < 2) {
	cpl_msg_error(fctid,"Dome flat frameset doesn't have enough frames");
	BUZZ_OFF
    }

    /* Check to see if there is a master bad pixel map. If there isn't one 
       then look for a confidence map */

    ps.master_mask = vircam_mask_define(framelist,ps.labels,nlab);

    /* Set up some convenience variables */

    dark1 = cpl_frameset_get_frame(ps.darklist,0);
    dark2 = cpl_frameset_get_frame(ps.darklist,1);
    dome1 = cpl_frameset_get_frame(ps.domelist,0);
    dome2 = cpl_frameset_get_frame(ps.domelist,1);
    
    /* Set up sorted frameset */

    ps.sorted = cpl_frameset_new();
    cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(dome1));
    cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(dome2));
    cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(dark1));
    cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(dark2));

    /* Get a propertylist for the primary */

    ps.ph = cpl_propertylist_load(cpl_frame_get_filename(dome1),0);

    /* Sort a frameset to use for the save routine. This is so that the
       header info (which is taken from the first frame in the list) comes
       from one of the dome images */

    ps.sorted = cpl_frameset_new();
    for (j = 0; j < cpl_frameset_get_size(ps.domelist); j++) {
	frame = cpl_frameset_get_frame(ps.domelist,j);
	cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(frame));
    }
    for (j = 0; j < cpl_frameset_get_size(ps.darklist); j++) {
	frame = cpl_frameset_get_frame(ps.darklist,j);
	cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(frame));
    }
    for (j = 0; j < cpl_frameset_get_size(framelist); j++) {
	frame = cpl_frameset_get_frame(framelist,j);
	if (strcmp(cpl_frame_get_tag(frame),VIRCAM_NOISE_FLAT_RAW) && 
	    strcmp(cpl_frame_get_tag(frame),VIRCAM_NOISE_DARK_RAW)) 
	    cpl_frameset_insert(ps.sorted,cpl_frame_duplicate(frame));
    }

    /* Loop for each of the image extensions */

    vircam_exten_range(vircam_detector_noise_config.extenum,
		       (const cpl_frame *)dark1,&jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_detector_noise_tidy();
	return(-1);
    }
    for (j = jst; j <= jfn; j++) {
	cpl_msg_info(fctid,"Beginning work on extension %d",j);
	isfirst = (j == jst);
	vircam_detector_noise_config.readnoise = 0.0;
	vircam_detector_noise_config.gain = 0.0;
	dummy = 1;

        /* Get the full set of images */

        ps.darkim1 = cpl_image_load(cpl_frame_get_filename(dark1),
				    CPL_TYPE_FLOAT,0,j);
        ps.darkim2 = cpl_image_load(cpl_frame_get_filename(dark2),
				    CPL_TYPE_FLOAT,0,j);
        ps.domeim1 = cpl_image_load(cpl_frame_get_filename(dome1),
				    CPL_TYPE_FLOAT,0,j);
        ps.domeim2 = cpl_image_load(cpl_frame_get_filename(dome2),
				    CPL_TYPE_FLOAT,0,j);
	if (ps.darkim1 == NULL || ps.darkim2 == NULL || ps.domeim1 == NULL ||
	    ps.domeim2 == NULL) {
	    cpl_error_reset();
	    cpl_msg_error(fctid,"NULL image input for extension %d",j);
            retval = vircam_detector_noise_save(framelist,parlist);
	    freeimage(ps.darkim1);
	    freeimage(ps.darkim2);
	    freeimage(ps.domeim1);
	    freeimage(ps.domeim2);
	    continue;
	}

	/* Get some an extension propertylist */

	ps.eh = cpl_propertylist_load(cpl_frame_get_filename(dome1),j);

	/* Check that the current detectors are live */

	plist = cpl_propertylist_load(cpl_frame_get_filename(dark1),j);
	vircam_pfits_get_detlive(plist,&live);
	if (! live) {
	    cpl_msg_warning(fctid,"First dark image detector not live");
            retval = vircam_detector_noise_save(framelist,parlist);
	    cpl_propertylist_delete(plist);
	    freeimage(ps.darkim1);
	    freeimage(ps.darkim2);
	    freeimage(ps.domeim1);
	    freeimage(ps.domeim2);
	    freepropertylist(ps.eh);
	    continue;
	}
	cpl_propertylist_delete(plist);
	plist = cpl_propertylist_load(cpl_frame_get_filename(dark2),j);
	vircam_pfits_get_detlive(plist,&live);
	if (! live) {
	    cpl_msg_warning(fctid,"Second dark image detector not live");
            retval = vircam_detector_noise_save(framelist,parlist);
	    cpl_propertylist_delete(plist);
	    freeimage(ps.darkim1);
	    freeimage(ps.darkim2);
	    freeimage(ps.domeim1);
	    freeimage(ps.domeim2);
	    freepropertylist(ps.eh);
	    continue;
	}
	cpl_propertylist_delete(plist);
	plist = cpl_propertylist_load(cpl_frame_get_filename(dome1),j);
	vircam_pfits_get_detlive(plist,&live);
	if (! live) {
	    cpl_msg_warning(fctid,"First dome image detector not live");
            retval = vircam_detector_noise_save(framelist,parlist);
	    cpl_propertylist_delete(plist);
	    freeimage(ps.darkim1);
	    freeimage(ps.darkim2);
	    freeimage(ps.domeim1);
	    freeimage(ps.domeim2);
	    freepropertylist(ps.eh);
	    continue;
	}
	cpl_propertylist_delete(plist);
	plist = cpl_propertylist_load(cpl_frame_get_filename(dome2),j);
	vircam_pfits_get_detlive(plist,&live);
	if (! live) {
	    cpl_msg_warning(fctid,"Second dome image detector not live");
            retval = vircam_detector_noise_save(framelist,parlist);
	    cpl_propertylist_delete(plist);
	    freeimage(ps.darkim1);
	    freeimage(ps.darkim2);
	    freeimage(ps.domeim1);
	    freeimage(ps.domeim2);
	    freepropertylist(ps.eh);
	    continue;
	}
	cpl_propertylist_delete(plist);

	/* Get the data array sizes */

	nx = cpl_image_get_size_x(ps.darkim1);
	ny = cpl_image_get_size_y(ps.darkim1);
	nptsdark = (long)(nx*ny);

	/* Load the mask */

        if (vircam_mask_load(ps.master_mask,j,nx,ny) == VIR_FATAL) {
            cpl_msg_info(fctid,"Unable to load mask image %s[%d]",
                         vircam_mask_get_filename(ps.master_mask),j);
            cpl_msg_info(fctid,"Forcing all pixels to be good from now on");
            vircam_mask_force(ps.master_mask,nx,ny);
        }
        bpm = vircam_mask_get_data(ps.master_mask);

        /* Get the mean of the first dark and the first dome */

	vircam_medmad((float *)cpl_image_get_data(ps.darkim1),bpm,
			       nptsdark,&meandark1,&sigdark);
	sigdark *= 1.48;
	lcut = meandark1 - vircam_detector_noise_config.thresh*sigdark;
	hcut = meandark1 + vircam_detector_noise_config.thresh*sigdark;
	vircam_medmadcut((float *)cpl_image_get_data(ps.darkim1),bpm,
			 nptsdark,lcut,hcut,&meandark1,&sigdark);
	vircam_medmad((float *)cpl_image_get_data(ps.domeim1),bpm,
			       nptsdark,&meandome1,&sigdark);
	sigdark *= 1.48;
	lcut = meandome1 - vircam_detector_noise_config.thresh*sigdark;
	hcut = meandome1 + vircam_detector_noise_config.thresh*sigdark;
	vircam_medmadcut((float *)cpl_image_get_data(ps.domeim1),bpm,
			 nptsdark,lcut,hcut,&meandome1,&sigdark);

        /* Get the mean of the second dark and the second dome */

	vircam_medmad((float *)cpl_image_get_data(ps.darkim2),bpm,
			       nptsdark,&meandark2,&sigdark);
	sigdark *= 1.48;
	lcut = meandark2 - vircam_detector_noise_config.thresh*sigdark;
	hcut = meandark2 + vircam_detector_noise_config.thresh*sigdark;
	vircam_medmadcut((float *)cpl_image_get_data(ps.darkim2),bpm,
			 nptsdark,lcut,hcut,&meandark2,&sigdark);
	vircam_medmad((float *)cpl_image_get_data(ps.domeim2),bpm,
			       nptsdark,&meandome2,&sigdark);
	sigdark *= 1.48;
	lcut = meandome2 - vircam_detector_noise_config.thresh*sigdark;
	hcut = meandome2 + vircam_detector_noise_config.thresh*sigdark;
	vircam_medmadcut((float *)cpl_image_get_data(ps.domeim2),bpm,
			 nptsdark,lcut,hcut,&meandome2,&sigdark);

        /* Form a difference image with each the of the dark and dome pairs */

        cpl_image_subtract(ps.darkim1,ps.darkim2);
	cpl_image_subtract(ps.domeim1,ps.domeim2);

        /* Now measure the mean and sigma of each of the difference images */
    
	vircam_medmad((float *)cpl_image_get_data(ps.darkim1),bpm,
			       nptsdark,&meandarkdiff,&sigdarkdiff);
	sigdarkdiff *= 1.48;
	lcut = meandarkdiff - vircam_detector_noise_config.thresh*sigdarkdiff;
	hcut = meandarkdiff + vircam_detector_noise_config.thresh*sigdarkdiff;
	vircam_medmadcut((float *)cpl_image_get_data(ps.darkim1),bpm,
			 nptsdark,lcut,hcut,&meandarkdiff,&sigdarkdiff);
	sigdarkdiff *= 1.48;
	vircam_medmad((float *)cpl_image_get_data(ps.domeim1),bpm,
			       nptsdark,&meandomediff,&sigdomediff);
	sigdomediff *= 1.48;
	lcut = meandomediff - vircam_detector_noise_config.thresh*sigdomediff;
	hcut = meandomediff + vircam_detector_noise_config.thresh*sigdomediff;
	vircam_medmadcut((float *)cpl_image_get_data(ps.domeim1),bpm,
			 nptsdark,lcut,hcut,&meandomediff,&sigdomediff);
	sigdomediff *= 1.48;

        /* Work out gain */

        gain = ((meandome1 + meandome2) - (meandark1 + meandark2))/
		(sigdomediff*sigdomediff - sigdarkdiff*sigdarkdiff);
	vircam_detector_noise_config.gain = gain;

        /* Now the read noise */

        readnoise = gain*sigdarkdiff/sqrt(2.0);
	vircam_detector_noise_config.readnoise = readnoise;
	dummy = 0;
    
        /* Save the products */

        retval = vircam_detector_noise_save(framelist,parlist);
	if (retval != 0) {
	    cpl_msg_error(fctid,"Error saving results");
	    BUZZ_OFF
	}

        /* Tidy up */

	freeimage(ps.darkim1);
	freeimage(ps.darkim2);
	freeimage(ps.domeim1);
	freeimage(ps.domeim2);
	vircam_mask_clear(ps.master_mask);
	freepropertylist(ps.eh);
    }
    vircam_detector_noise_tidy();
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_detector_noise_save(cpl_frameset *framelist, 
				      cpl_parameterlist *parlist) {
    const char *fctid = "vircam_detector_noise_save";
    const char *outpaf = "vircam_detector_noise";
    const char *outfile = "detector_noise.fits";
    const char *recipeid = "vircam_detector_noise";
    cpl_propertylist *plist,*p;
    cpl_table *o;

    /* If this is the first time through then create the output image */
    
    if (isfirst) {

        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_READGAINFILE);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

	/* Fiddle with the primary header */

	plist = cpl_propertylist_duplicate(ps.ph);
	ps.phupaf = vircam_paf_phu_items(plist);
        vircam_dfs_set_product_primary_header(plist,product_frame,ps.sorted,
                                              parlist,(char *)recipeid,
                                              "PRO-1.15");

        /* 'Save' the PHU image */

        if (cpl_image_save(NULL,outfile,CPL_BPP_8_UNSIGNED,plist,
                           CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product PHU");
            cpl_frame_delete(product_frame);
	    cpl_propertylist_delete(plist);
            return(-1);
        }
        cpl_propertylist_delete(plist);
        cpl_frameset_insert(framelist,product_frame);
    }

    /* Create results header */

    plist = cpl_propertylist_duplicate(ps.eh);

    /* Now add the new data... */

    cpl_propertylist_update_float(plist,"ESO QC READNOISE",
				  vircam_detector_noise_config.readnoise);
    cpl_propertylist_set_comment(plist,"ESO QC READNOISE",
				 "Calculated detector readnoise");
    cpl_propertylist_update_float(plist,"ESO QC GAIN",
				  vircam_detector_noise_config.gain);
    cpl_propertylist_set_comment(plist,"ESO QC GAIN",
				 "Calculated detector gain");

    /* Now fiddle with the extension header */

    vircam_dfs_set_product_exten_header(plist,product_frame,ps.sorted,
					parlist,(char *)recipeid,
					"PRO-1.15");
    if (dummy)
	vircam_dummy_property(plist);

    /* Create the output table */

    o = cpl_table_new(1);
    cpl_table_new_column(o,"EXTNAME",CPL_TYPE_STRING);
    cpl_table_new_column(o,"READNOISE",CPL_TYPE_FLOAT);
    cpl_table_new_column(o,"GAIN",CPL_TYPE_FLOAT);

    /* Write the values to the table */

    cpl_table_set_string(o,"EXTNAME",0,
			 cpl_propertylist_get_string(plist,"EXTNAME"));
    cpl_table_set_float(o,"READNOISE",0,
			vircam_detector_noise_config.readnoise);
    cpl_table_set_float(o,"GAIN",0,vircam_detector_noise_config.gain);

    /* 'Save' the image */

    if (cpl_table_save(o,NULL,plist,outfile,CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(fctid,"Cannot save product");
        cpl_propertylist_delete(plist);
	cpl_frame_delete(product_frame);
	freetable(o);
	return(-1);
    }
    freetable(o);

    /* Write the PAF */

    p = vircam_paf_req_items(plist);
    vircam_merge_propertylists(p,ps.phupaf);
    if (vircam_paf_print((char *)outpaf,"VIRCAM/vircam_detector_noise",
			 "QC file",p) != VIR_OK)
        cpl_msg_warning(fctid,"Unable to write PAF\n");
    cpl_propertylist_delete(p);

    /* Tidy and exit */

    freepropertylist(plist);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_detector_noise_init(void) {
    ps.labels = NULL;
    ps.darklist = NULL;
    ps.domelist = NULL;
    ps.darkim1 = NULL;
    ps.darkim2 = NULL;
    ps.domeim1 = NULL;
    ps.domeim2 = NULL;
    ps.master_mask = NULL;
    ps.sorted = NULL;
    ps.ph = NULL;
    ps.eh = NULL;
    ps.phupaf = NULL;
    return;
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_detector_noise_tidy(void) {
    freespace(ps.labels);
    freeframeset(ps.darklist);
    freeframeset(ps.domelist);
    freeframeset(ps.sorted);
    freeimage(ps.darkim1);
    freeimage(ps.darkim2);
    freeimage(ps.domeim1);
    freeimage(ps.domeim2);
    freemask(ps.master_mask);
    freeframeset(ps.sorted);
    freepropertylist(ps.ph);
    freepropertylist(ps.eh);
    freepropertylist(ps.phupaf);
    return;
}

/**@}*/


/*

$Log: vircam_detector_noise.c,v $
Revision 1.48  2007/10/25 18:39:22  jim
Altered to remove some lint messages

Revision 1.47  2007/10/15 12:52:39  jim
Fixed little cockup which had duplicate entries in the ps structure

Revision 1.46  2007/09/07 13:32:12  jim
uses a sorted framelist to ensure that the correct information is given
to the output product header

Revision 1.45  2007/09/06 21:37:53  jim
fixed call to vircam_dfs_setup_product_ routines to use the full input
frameset

Revision 1.44  2007/07/18 15:35:41  jim
Added better error handling for missing or corrupt mask extensions

Revision 1.43  2007/07/09 13:21:55  jim
Modified to use new version of vircam_exten_range

Revision 1.42  2007/06/13 08:11:27  jim
Modified docs to reflect changes in DFS tags

Revision 1.41  2007/04/04 10:36:18  jim
Modified to use new dfs tags

Revision 1.40  2007/03/29 12:19:38  jim
Little changes to improve documentation

Revision 1.39  2007/03/06 12:30:17  jim
Now writes results to a table in addtion to a header and a paf.

Revision 1.38  2007/03/01 12:41:49  jim
Modified slightly after code checking

Revision 1.37  2007/02/25 06:26:35  jim
Plugged a few memory leaks

Revision 1.36  2007/02/15 06:59:37  jim
Added ability to write QC paf files

Revision 1.35  2007/02/07 10:12:39  jim
Removed calls to vircam_ndit_correct as this is now no longer necessary

Revision 1.34  2007/02/06 13:11:12  jim
Fixed entry for PRO dictionary in cpl_dfs_set_product_header

Revision 1.33  2007/02/06 12:38:59  jim
Creates a sorted version of the input frameset so that a dome flat is first
in the list. This is done so that the header of the output product comes
from one of the dome flats and so that you can see which filter it was done
with.

Revision 1.32  2007/02/06 11:59:38  jim
Added call to vircam_medmadcut for a better estimate of dispersion

Revision 1.31  2007/01/17 23:56:12  jim
Plugged possible memory leak

Revision 1.30  2006/12/14 14:46:32  jim
Fixed typo

Revision 1.29  2006/12/11 22:47:49  jim
Fixed QC header names

Revision 1.28  2006/11/27 12:16:33  jim
Modified to change location in the header where the results are written

Revision 1.27  2006/11/10 09:20:29  jim
Fixed _save routine so that an extension name is written out to the file
header

Revision 1.26  2006/09/29 11:19:31  jim
changed aliases on parameter names

Revision 1.25  2006/09/09 16:49:39  jim
Header comment update

Revision 1.24  2006/09/08 09:17:27  jim
Modified to flag dummy results

Revision 1.23  2006/09/04 23:02:14  jim
Modified to deal with det live issues. Also does a better job of dealing
with duff input

Revision 1.22  2006/06/20 19:07:01  jim
Corrects for ndit != 1

Revision 1.21  2006/06/15 09:58:58  jim
Minor changes to docs

Revision 1.20  2006/06/13 21:27:10  jim
Changed output product to a MEF with null images and readnoise/gain estimates
in the header. I HATE this solution

Revision 1.19  2006/06/09 11:26:25  jim
Small changes to keep lint happy

Revision 1.18  2006/05/04 11:56:02  jim
changed filename extension in default output file

Revision 1.17  2006/05/04 11:53:14  jim
Fixed the way the _save routine works to be more consistent with the
standard CPL way of doing things

Revision 1.16  2006/04/27 09:46:01  jim
Modified DFS frame types to conform to new dictionary

Revision 1.15  2006/03/22 14:02:51  jim
cosmetic changes to keep lint happy

Revision 1.14  2006/03/22 12:13:51  jim
Modified to use new vircam_mask capability

Revision 1.13  2006/03/15 10:43:40  jim
Fixed a few things

Revision 1.12  2006/01/23 10:36:39  jim
Now allows a CPM or a BPM to be used as the mask

Revision 1.11  2005/12/14 22:19:12  jim
fixed docs

Revision 1.10  2005/12/09 09:47:58  jim
Many changes to add more documentation

Revision 1.9  2005/12/02 10:45:37  jim
The tags used in the sof are now written to the description string in the
constructor. This is so that if they change in the vircam_dfs.h file, they
aren't then hardcopied into each of the recipes...

Revision 1.8  2005/11/29 16:10:58  jim
Added clipping into stats for difference images

Revision 1.7  2005/11/25 15:20:29  jim
Fixed bug in mathematical description

Revision 1.6  2005/11/25 09:56:14  jim
Tidied up some more documentation

Revision 1.5  2005/11/23 14:57:40  jim
A bit of tidying in response to splint messages

Revision 1.4  2005/11/21 16:16:23  jim
Added some better garbage collection

Revision 1.3  2005/11/07 13:14:18  jim
Added some error trapping and some docs

Revision 1.2  2005/08/09 11:09:39  jim
Replaced dodgy call to cpl_framelist_delete with correct cpl_frameset_delete

Revision 1.1.1.1  2005/08/05 08:29:09  jim
Initial import


*/
