/* $Id: vircam_crosstalk_analyse.c,v 1.3 2007/04/04 10:36:18 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/04/04 10:36:18 $
 * $Revision: 1.3 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>
#include <math.h>

#include "vircam_utils.h"
#include "vircam_pfits.h"
#include "vircam_dfs.h"
#include "vircam_mods.h"
#include "vircam_stats.h"
#include "vircam_fits.h"
#include "vircam_mask.h"
#include "vircam_channel.h"

/* Function prototypes */

static int vircam_crosstalk_analyse_create(cpl_plugin *) ;
static int vircam_crosstalk_analyse_exec(cpl_plugin *) ;
static int vircam_crosstalk_analyse_destroy(cpl_plugin *) ;
static int vircam_crosstalk_analyse(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_crosstalk_analyse_save(cpl_frameset *framelist, 
					   cpl_parameterlist *parlist);
static void vircam_crosstalk_analyse_init(void);
static void vircam_crosstalk_analyse_tidy(int level);

/* Static global variables */

static struct {

    /* Input */

    int         extenum;


} vircam_crosstalk_analyse_config;


static struct {
    int               *labels;
} ps;

static char vircam_crosstalk_analyse_description[] =
"vircam_crosstalk_analyse -- VIRCAM crosstalk analysis.\n\n"
"Dummy recipe\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"\n";


/**@{*/

/**
    \ingroup recipelist
    \defgroup vircam_crosstalk_analyse vircam_crosstalk_analyse
    \brief Analyse images from a crosstalk template and work out the crosstalk 
     matrix.

    \par Name: 
        vircam_crosstalk_analyse
    \par Purpose: 
        Dummy crosstalk analysis routine
    \par Description: 
        Dummy crosstalk analysis routine
    \par Language:
        C
    \par Parameters:
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO CATG value.
    \par Output Products:
        The following list gives the output data products that are generated
        by this recipe. The word in bold gives the PRO CATG keyword value for
        each product:
    \par Output QC Parameters:
    \par Notes
        None.
    \par Fatal Error Conditions:
    \par Non-Fatal Error Conditions:
    \par Conditions Leading To Dummy Products:
    \par Author:
        Jim Lewis, CASU
    \par Code Reference: 
        vircam_crosstalk_analyse.c
*/

/* Function code */


/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_crosstalk_analyse_description);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_crosstalk_analyse",
                    "VIRCAM crosstalk analysis routine",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_crosstalk_analyse_create,
                    vircam_crosstalk_analyse_exec,
                    vircam_crosstalk_analyse_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_crosstalk_analyse_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_crosstalk_analyse.extenum",
			        CPL_TYPE_INT,
			        "Extension number to be done, 0 == all",
			        "vircam.vircam_crosstalk_analyse",
			        1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);
        
    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_crosstalk_analyse_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    return(vircam_crosstalk_analyse(recipe->parameters,recipe->frames));
}
				
/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_crosstalk_analyse_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else 
	return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here 
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_crosstalk_analyse(cpl_parameterlist *parlist, 
				      cpl_frameset *framelist) {
    const char *fctid="vircam_crosstalk_analyse";    

    /* Initialise some things */

    vircam_crosstalk_analyse_init();
    cpl_msg_info(fctid,"This is a dummy recipe");
    vircam_crosstalk_analyse_tidy(1);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_crosstalk_analyse_save(cpl_frameset *framelist, 
					   cpl_parameterlist *parlist) {
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_crosstalk_analyse_init(void) {
    ps.labels = NULL;
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_crosstalk_analyse_tidy(int level) {


    if (level == 1)
	return;
    freespace(ps.labels);

}

/**@}*/

/* 

$Log: vircam_crosstalk_analyse.c,v $
Revision 1.3  2007/04/04 10:36:18  jim
Modified to use new dfs tags

Revision 1.2  2007/03/29 12:19:38  jim
Little changes to improve documentation

Revision 1.1  2007/01/10 22:10:10  jim
Added as dummy


*/


