#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>


struct io_header_1
{
  int npart[6];                 /*!< number of particles of each type in this file */
  double mass[6];               /*!< mass of particles of each type. If 0, then the masses are explicitly
                                 stored in the mass-block of the snapshot file, otherwise they are omitted */
  double time;                  /*!< time of snapshot file */
  double redshift;              /*!< redshift of snapshot file */
  int flag_sfr;                 /*!< flags whether the simulation was including star formation */
  int flag_feedback;            /*!< flags whether feedback was included (obsolete) */
  unsigned int npartTotal[6];   /*!< total number of particles of each type in this snapshot. This can be
                                 different from npart if one is dealing with a multi-file snapshot. */
  int flag_cooling;             /*!< flags whether cooling was included  */
  int num_files;                /*!< number of files in multi-file snapshot */
  double BoxSize;               /*!< box-size of simulation in case periodic boundaries were used */
  double Omega0;                /*!< matter density in units of critical density */
  double OmegaLambda;           /*!< cosmological constant parameter */
  double HubbleParam;           /*!< Hubble parameter in units of 100 km/sec/Mpc */
  int flag_stellarage;          /*!< flags whether the file contains formation times of star particles */
  int flag_metals;              /*!< flags whether the file contains metallicity values for gas and star
                                 particles */
  
  
  char     fill[256- 6*4- 6*8- 2*8- 2*4- 6*4- 2*4 - 4*8 - 2*4];  /* fills to 256 Bytes */
} header, header1, header2;


int run;

int     NumPart, Ngas, Nhalo, Ndisk, Nbulge, Nstars;
int     NumPart1, Ngas1, Nhalo1, Ndisk1, Nbulge1, Nstars1;
int     NumPart2, Ngas2, Nhalo2, Ndisk2, Nbulge2, Nstars2;

int     NumParttot, Ngastot, Nhalotot, Ndisktot, Nbulgetot, Nstarstot;
int     NumParttot1, Ngastot1, Nhalotot1, Ndisktot1, Nbulgetot1, Nstarstot1;
int     NumParttot2, Ngastot2, Nhalotot2, Ndisktot2, Nbulgetot2, Nstarstot2;

int     Ngashwhw, Nhalohw, Ndiskhw, Nbulgehw, Nstarshw;
int     Ngashwhw1, Nhalohw1, Ndiskhw1, Nbulgehw1, Nstarshw1;
int     Ngashwhw2, Nhalohw2, Ndiskhw2, Nbulgehw2, Nstarshw2;

double  Mgas, Mhalo, Mdisk, Mbulge, Mstars;
double  Mgas1, Mhalo1, Mdisk1, Mbulge1, Mstars1;
double  Mgas2, Mhalo2, Mdisk2, Mbulge2, Mstars2;




struct particle_data
{
  float  Pos[3];
  float  Vel[3];
  float  Mass;
  int    Type;
  
  float  Rho, U, Temp, Ne, Nh, Hsml, Sfr, SAge;
} *P, *P1, *P2;




int *Id, *Id1, *Id2;




double  Time, Redshift, BoxSize, Omega0, OmegaLambda, HubbleParam;
double  Time1, Redshift1, BoxSize1, Omega01, OmegaLambda1, HubbleParam1;
double  Time2, Redshift2, BoxSize2, Omega02, OmegaLambda2, HubbleParam2;

int     FlagSfr, Flagfeedback, FlagCooling, NumFiles, FlagAge, FlagMetals, flag_entr_ics;
int     FlagSfr1, Flagfeedback1, FlagCooling1, NumFiles1, FlagAge1, FlagMetals1, flag_entr_ics1;
int     FlagSfr2, Flagfeedback2, FlagCooling2, NumFiles2, FlagAge2, FlagMetals2, flag_entr_ics2;

double phi1;
double theta1;
double phi2;
double theta2;

int load_file1(char *fname, int files);
int load_file2(char *fname, int files);
int allocate_memory(void);
int allocate_memory_1(void);
int allocate_memory_2(void);
int save_particles(char *fname);
int merge(void);
int separate(double Rstart, double Rmin, double e);
int rotate_1(void);
int rotate_2(void);



/* Here we load a snapshot file. It can be distributed
 * onto several files (for files>1).
 * The particles are brought back into the order
 * implied by their ID's.
 * A unit conversion routine is called to do unit
 * conversion, and to evaluate the gas temperature.
 */
int main(int argc, char **argv)
{
  char input_fname1[200], input_fname2[200], output_fname[200];
  int  type, files;
  double Rstart,Rmin,e;
  
  if (argc < 11) {
    printf("Enter 1st galaxy filename, 2nd galaxy filename, output filename, start distance, pericentric distance, eccentricity, phi1, theta1, phi2, theta2!\n");
    exit(0);
  }
  else {
    sprintf(input_fname1, "%s", argv[1]);
    sprintf(input_fname2, "%s", argv[2]);
    sprintf(output_fname, "%s", argv[3]);
    Rstart = atof(argv[4]);
    Rmin   = atof(argv[5]);
    e      = atof(argv[6]);
    phi1   = atof(argv[7]);
    theta1 = atof(argv[8]);
    phi2   = atof(argv[9]);
    theta2 = atof(argv[10]);
  }
  
  files=1;                               /* number of files per snapshot */
  
  load_file1(input_fname1, files);
  load_file2(input_fname2, files);
  
 
  printf("*****************************************\n");
  printf("Number of Particles in 1 - Gas   = %i\n", Ngas1);
  printf("Number of Particles in 1 - Halo  = %i\n", Nhalo1);
  printf("Number of Particles in 1 - Disk  = %i\n", Ndisk1);
  printf("Number of Particles in 1 - Bulge = %i\n", Nbulge1);
  printf("Number of Particles in 1 - Stars = %i\n", Nstars1);
  printf("Number of Particles in 1 - Total = %i\n", NumPart1);
  printf("*****************************************\n");
  printf("Number of Particles in 2 - Gas   = %i\n", Ngas2);
  printf("Number of Particles in 2 - Halo  = %i\n", Nhalo2);
  printf("Number of Particles in 2 - Disk  = %i\n", Ndisk2);
  printf("Number of Particles in 2 - Bulge = %i\n", Nbulge2);
  printf("Number of Particles in 2 - Stars = %i\n", Nstars2);
  printf("Number of Particles in 2 - Total = %i\n", NumPart2);
  printf("*****************************************\n");
  
  rotate_1();
  rotate_2();
  separate(Rstart,Rmin,e);
  merge();
  
  save_particles(output_fname);
  
}



/* this routine loads particle data from Gadget's default
 * binary file format. Reads file1.
 */
int load_file1(char *fname, int files)
{
  FILE *fd;
  char   buf[200];
  int    i,j,k,dummy,ntot_withmasses;
  int    t,n,off,pc,pc_new,pc_sph;
  
#define SKIP fread(&dummy, sizeof(dummy), 1, fd);
  
  for(i=0, pc=1; i<files; i++, pc=pc_new)
  {
    if(files>1)
      sprintf(buf,"%s.%d",fname,i);
    else
      sprintf(buf,"%s",fname);
    
    if(!(fd=fopen(buf,"r")))
    {
      printf("can't open file `%s`\n",buf);
      exit(0);
    }
    
    printf("reading `%s' ...\n",buf); fflush(stdout);
    
    SKIP;
    fread(&header1, sizeof(header1), 1, fd);
    SKIP;
    
    if(files==1)
    {
      for(k=0, NumPart1=0, ntot_withmasses=0; k<5; k++)
        NumPart1+= header1.npart[k];
      Ngas1=   header1.npart[0];
      Nhalo1=  header1.npart[1];
      Ndisk1=  header1.npart[2];
      Nbulge1= header1.npart[3];
      Nstars1= header1.npart[4];
    }
    else
    {
      for(k=0, NumPart1=0, ntot_withmasses=0; k<5; k++)
        NumPart1+= header1.npartTotal[k];
      Ngas1=   header1.npartTotal[0];
      Nhalo1=  header1.npartTotal[1];
      Ndisk1=  header1.npartTotal[2];
      Nbulge1= header1.npartTotal[3];
      Nstars1= header1.npartTotal[4];
    }
    
    for(k=0, ntot_withmasses=0; k<5; k++)
    {
      if(header1.mass[k]==0)
        ntot_withmasses+= header1.npart[k];
      Mgas1=   header1.mass[0];
      Mhalo1=  header1.mass[1];
      Mdisk1=  header1.mass[2];
      Mbulge1= header1.mass[3];
      Mstars1= header1.mass[4];
    }
    
    Time1= header1.time;
    Redshift1= header1.redshift;
    FlagSfr1= header1.flag_sfr;
    Flagfeedback1= header1.flag_feedback;
    Ngastot1= header1.npartTotal[0];
    Nhalotot1= header1.npartTotal[1];
    Ndisktot1= header1.npartTotal[2];
    Nbulgetot1= header1.npartTotal[3];
    Nstarstot1= header1.npartTotal[4];
    FlagCooling1= header1.flag_cooling;
    NumFiles1= header1.num_files;
    BoxSize1= header1.BoxSize;
    Omega01= header1.Omega0;
    OmegaLambda1= header1.OmegaLambda;
    HubbleParam1= header1.HubbleParam;
    FlagAge1= header1.flag_stellarage;
    FlagMetals1= header1.flag_metals;
    NumParttot1=0;
    for(k=0;k<6;k++) NumParttot1+=header1.npartTotal[k];
    
    
    if(i==0)
      allocate_memory_1();
    
    SKIP;
    for(k=0,pc_new=pc;k<6;k++)
    {
      for(n=0;n<header1.npart[k];n++)
      {
        fread(&P1[pc_new].Pos[0], sizeof(float), 3, fd);
        pc_new++;
      }
    }
    SKIP;
    
    SKIP;
    for(k=0,pc_new=pc;k<6;k++)
    {
      for(n=0;n<header1.npart[k];n++)
      {
        fread(&P1[pc_new].Vel[0], sizeof(float), 3, fd);
        pc_new++;
      }
    }
    SKIP;
    
    
    SKIP;
    for(k=0,pc_new=pc;k<6;k++)
    {
      for(n=0;n<header1.npart[k];n++)
      {
        fread(&Id1[pc_new], sizeof(int), 1, fd);
        pc_new++;
      }
    }
    SKIP;
    
    
    if(ntot_withmasses>0)
      SKIP;
    for(k=0, pc_new=pc; k<6; k++)
    {
      for(n=0;n<header1.npart[k];n++)
      {
        P1[pc_new].Type=k;
        
        if(header1.mass[k]==0)
          fread(&P1[pc_new].Mass, sizeof(float), 1, fd);
        else
          P1[pc_new].Mass= header1.mass[k];
        pc_new++;
      }
    }
    if(ntot_withmasses>0)
      SKIP;
    
    
    if(header1.npart[0]>0)
    {
      SKIP;
      for(n=0, pc_sph=pc; n<header1.npart[0];n++)
      {
        fread(&P1[pc_sph].U, sizeof(float), 1, fd);
        pc_sph++;
      }
      SKIP;
      
      SKIP;
      for(n=0, pc_sph=pc; n<header1.npart[0];n++)
      {
        fread(&P1[pc_sph].Rho, sizeof(float), 1, fd);
        pc_sph++;
      }
      SKIP;
      
      if(header1.flag_cooling)
      {
        SKIP;
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          fread(&P1[pc_sph].Ne, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
        
        SKIP;
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          fread(&P1[pc_sph].Nh, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
      }
      else
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          P1[pc_sph].Ne= 1.0;
          P1[pc_sph].Nh= 0.0;
          pc_sph++;
        }
      
      SKIP;
      for(n=0, pc_sph=pc; n<header1.npart[0];n++)
      {
        fread(&P1[pc_sph].Hsml, sizeof(float), 1, fd);
        pc_sph++;
      }
      SKIP;
      
      if(header1.flag_sfr)
      {
        SKIP;
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          fread(&P1[pc_sph].Sfr, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
      }
      else
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          P1[pc_sph].Sfr= 0.0;
          pc_sph++;
        }
      
      if(header1.flag_stellarage)
      {
        SKIP;
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          fread(&P1[pc_sph].SAge, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
      }
      else
        for(n=0, pc_sph=pc; n<header1.npart[0];n++)
        {
          P1[pc_sph].SAge= 0.0;
          pc_sph++;
        }
      
    }
    
    fclose(fd);
  }
  return 0;
}


/* this routine loads particle data from Gadget's default
 * binary file format. Reads file2.
 */
int load_file2(char *fname, int files)
{
  FILE *fd;
  char   buf[200];
  int    i,j,k,dummy,ntot_withmasses;
  int    t,n,off,pc,pc_new,pc_sph;
  
#define SKIP fread(&dummy, sizeof(dummy), 1, fd);
  
  for(i=0, pc=1; i<files; i++, pc=pc_new)
  {
    if(files>1)
      sprintf(buf,"%s.%d",fname,i);
    else
      sprintf(buf,"%s",fname);
    
    if(!(fd=fopen(buf,"r")))
    {
      printf("can't open file `%s`\n",buf);
      exit(0);
    }
    
    printf("reading `%s' ...\n",buf); fflush(stdout);
    
    SKIP;
    fread(&header2, sizeof(header2), 1, fd);
    SKIP;
    
    if(files==1)
    {
      for(k=0, NumPart2=0, ntot_withmasses=0; k<5; k++)
        NumPart2+= header2.npart[k];
      Ngas2=   header2.npart[0];
      Nhalo2=  header2.npart[1];
      Ndisk2=  header2.npart[2];
      Nbulge2= header2.npart[3];
      Nstars2= header2.npart[4];
    }
    else
    {
      for(k=0, NumPart2=0, ntot_withmasses=0; k<5; k++)
        NumPart2+= header2.npartTotal[k];
      Ngas2=   header2.npartTotal[0];
      Nhalo2=  header2.npartTotal[1];
      Ndisk2=  header2.npartTotal[2];
      Nbulge2= header2.npartTotal[3];
      Nstars2= header2.npartTotal[4];
    }
    
    for(k=0, ntot_withmasses=0; k<5; k++)
    {
      if(header2.mass[k]==0)
        ntot_withmasses+= header2.npart[k];
      Mgas2=   header2.mass[0];
      Mhalo2=  header2.mass[1];
      Mdisk2=  header2.mass[2];
      Mbulge2= header2.mass[3];
      Mstars2= header2.mass[4];
    }
    
    Time2= header2.time;
    Redshift2= header2.redshift;
    FlagSfr2= header2.flag_sfr;
    Flagfeedback2= header2.flag_feedback;
    Ngastot2= header2.npartTotal[0];
    Nhalotot2= header2.npartTotal[1];
    Ndisktot2= header2.npartTotal[2];
    Nbulgetot2= header2.npartTotal[3];
    Nstarstot2= header2.npartTotal[4];
    FlagCooling2= header2.flag_cooling;
    NumFiles2= header2.num_files;
    BoxSize2= header2.BoxSize;
    Omega02= header2.Omega0;
    OmegaLambda2= header2.OmegaLambda;
    HubbleParam2= header2.HubbleParam;
    FlagAge2= header2.flag_stellarage;
    FlagMetals2= header2.flag_metals;
    NumParttot2=0;
    for(k=0;k<6;k++) NumParttot2+=header2.npartTotal[k];
    
    
    if(i==0)
      allocate_memory_2();
    
    SKIP;
    for(k=0,pc_new=pc;k<6;k++)
    {
      for(n=0;n<header2.npart[k];n++)
      {
        fread(&P2[pc_new].Pos[0], sizeof(float), 3, fd);
        pc_new++;
      }
    }
    SKIP;
    
    SKIP;
    for(k=0,pc_new=pc;k<6;k++)
    {
      for(n=0;n<header2.npart[k];n++)
      {
        fread(&P2[pc_new].Vel[0], sizeof(float), 3, fd);
        pc_new++;
      }
    }
    SKIP;
    
    
    SKIP;
    for(k=0,pc_new=pc;k<6;k++)
    {
      for(n=0;n<header2.npart[k];n++)
      {
        fread(&Id2[pc_new], sizeof(int), 1, fd);
        pc_new++;
      }
    }
    SKIP;
    
    
    if(ntot_withmasses>0)
      SKIP;
    for(k=0, pc_new=pc; k<6; k++)
    {
      for(n=0;n<header2.npart[k];n++)
      {
        P2[pc_new].Type=k;
        
        if(header2.mass[k]==0)
          fread(&P2[pc_new].Mass, sizeof(float), 1, fd);
        else
          P2[pc_new].Mass= header2.mass[k];
        pc_new++;
      }
    }
    if(ntot_withmasses>0)
      SKIP;
    
    
    if(header2.npart[0]>0)
    {
      SKIP;
      for(n=0, pc_sph=pc; n<header2.npart[0];n++)
      {
        fread(&P2[pc_sph].U, sizeof(float), 1, fd);
        pc_sph++;
      }
      SKIP;
      
      SKIP;
      for(n=0, pc_sph=pc; n<header2.npart[0];n++)
      {
        fread(&P2[pc_sph].Rho, sizeof(float), 1, fd);
        pc_sph++;
      }
      SKIP;
      
      if(header2.flag_cooling)
      {
        SKIP;
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          fread(&P2[pc_sph].Ne, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
        
        SKIP;
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          fread(&P2[pc_sph].Nh, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
      }
      else
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          P2[pc_sph].Ne= 1.0;
          P2[pc_sph].Nh= 0.0;
          pc_sph++;
        }
      
      SKIP;
      for(n=0, pc_sph=pc; n<header2.npart[0];n++)
      {
        fread(&P2[pc_sph].Hsml, sizeof(float), 1, fd);
        pc_sph++;
      }
      SKIP;
      
      if(header2.flag_sfr)
      {
        SKIP;
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          fread(&P2[pc_sph].Sfr, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
      }
      else
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          P2[pc_sph].Sfr= 0.0;
          pc_sph++;
        }
      
      if(header2.flag_stellarage)
      {
        SKIP;
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          fread(&P2[pc_sph].SAge, sizeof(float), 1, fd);
          pc_sph++;
        }
        SKIP;
      }
      else
        for(n=0, pc_sph=pc; n<header2.npart[0];n++)
        {
          P2[pc_sph].SAge= 0.0;
          pc_sph++;
        }
      
    }
    
    fclose(fd);
  }
  return 0;
}



/* this routine allocates the memory for the
 * particle data.
 */
int allocate_memory(void)
{
  printf("allocating memory...\n");
  
  if(!(P=malloc(NumPart*sizeof(struct particle_data))))
  {
    fprintf(stderr,"failed to allocate memory.\n");
    exit(0);
  }
  
  P--;   /* start with offset 1 */
  
  if(!(Id=malloc(NumPart*sizeof(int))))
  {
    fprintf(stderr,"failed to allocate memory.\n");
    exit(0);
  }
  
  Id--;   /* start with offset 1 */
  
  printf("allocating memory...done\n");
  return 0;
}

/* this routine allocates the memory for the
 * particle data in file1.
 */
int allocate_memory_1(void)
{
  printf("allocating memory in file1...\n");
  
  if(!(P1=malloc(NumPart1*sizeof(struct particle_data))))
  {
    fprintf(stderr,"failed to allocate memory.\n");
    exit(0);
  }
  
  P1--;   /* start with offset 1 */
  
  if(!(Id1=malloc(NumPart1*sizeof(int))))
  {
    fprintf(stderr,"failed to allocate memory.\n");
    exit(0);
  }
  
  Id1--;   /* start with offset 1 */
  
  printf("allocating memory in file1...done\n");
  return 0;
}


/* this routine allocates the memory for the
 * particle data.
 */
int allocate_memory_2(void)
{
  printf("allocating memory in file2...\n");
  
  if(!(P2=malloc(NumPart2*sizeof(struct particle_data))))
  {
    fprintf(stderr,"failed to allocate memory.\n");
    exit(0);
  }
  
  P2--;   /* start with offset 1 */
  
  if(!(Id2=malloc(NumPart2*sizeof(int))))
  {
    fprintf(stderr,"failed to allocate memory.\n");
    exit(0);
  }
  
  Id2--;   /* start with offset 1 */
  
  printf("allocating memory in file2...done\n");
  return 0;
}





int save_particles(char *fname)
{
  FILE *fd;
  int i,d,k;
  int nwithmass;
  float xyz[3];
  double t;
  int blklen;
#define BLKLEN fwrite(&blklen, sizeof(blklen), 1, fd);
  
  
  if(!(fd=fopen(fname,"w")))
  {
    printf("error opening file %s\n",fname);
    exit(0);
  }
  
  printf("saveing initial conditions to file `%s'\n\n",fname);
  
  blklen=sizeof(header);
  BLKLEN;
  fwrite(&header, sizeof(header), 1, fd);
  BLKLEN;
  
  
  blklen=3*(Ngas+Nhalo+Ndisk+Nbulge+Nstars)*sizeof(float);
  BLKLEN;
  for(i=1;i<=NumPart;i++)
  {
    xyz[0]=P[i].Pos[0];
    xyz[1]=P[i].Pos[1];
    xyz[2]=P[i].Pos[2];
    fwrite(xyz,sizeof(float),3,fd);
  }
  BLKLEN;
  
  blklen=3*(Ngas+Nhalo+Ndisk+Nbulge+Nstars)*sizeof(float);
  BLKLEN;
  for(i=1;i<=NumPart;i++)
  {
    xyz[0]=P[i].Vel[0];
    xyz[1]=P[i].Vel[1];
    xyz[2]=P[i].Vel[2];
    fwrite(xyz,sizeof(float),3,fd);
  }
  BLKLEN;
  
  
  blklen=(Ngas+Nhalo+Ndisk+Nbulge+Nstars)*sizeof(int);
  BLKLEN;
  for(i=1;i<=NumPart;i++)
  {
    fwrite(&Id[i],sizeof(int),1,fd);  // ID
  }
  BLKLEN;
  
  
  nwithmass=0;
  for(k=0; k<6; k++)
  {
    if (header.mass[k]==0) nwithmass+= header.npart[k];
  }
  
  blklen=nwithmass*sizeof(float);
  if(nwithmass>0) BLKLEN;
  
  if(header.mass[0]==0)
  {
    for(i=1;i<=Ngas;i++)
    {
      xyz[0]=P[i].Mass;
      fwrite(xyz,sizeof(float),1,fd);
    }
  }
  
  if(header.mass[1]==0)
  {
    for(i=Ngas+1;i<=Ngas+Nhalo;i++)
    {
      xyz[0]=P[i].Mass;
      fwrite(xyz,sizeof(float),1,fd);
    }
  }
  
  if(header.mass[2]==0)
  {
    for(i=Ngas+Nhalo+1;i<=Ngas+Nhalo+Ndisk;i++)
    {
      xyz[0]=P[i].Mass;
      fwrite(xyz,sizeof(float),1,fd);
    }
  }
  
  if(header.mass[3]==0)
  {
    for(i=Ngas+Nhalo+Ndisk+1;i<=Ngas+Nhalo+Ndisk+Nbulge;i++)
    {
      xyz[0]=P[i].Mass;
      fwrite(xyz,sizeof(float),1,fd);
    }
  }
  
  if(header.mass[4]==0)
  {
    for(i=Ngas+Nhalo+Ndisk+Nbulge+1;i<=Ngas+Nhalo+Ndisk+Nbulge+Nstars;i++)
    {
      xyz[0]=P[i].Mass;
      fwrite(xyz,sizeof(float),1,fd);
    }
  }
  
  if(nwithmass>0) BLKLEN;
  
  if(Ngas)
  {
    blklen=(Ngas)*sizeof(float);
    BLKLEN;
    for(i=1;i<=Ngas;i++)
    {
      xyz[0]= P[i].U;
      fwrite(xyz, sizeof(float), 1, fd);
    }
    BLKLEN;
    
    BLKLEN;
    for(i=1;i<=Ngas;i++)
    {
      xyz[0]= P[i].Rho;
      fwrite(xyz, sizeof(float), 1, fd);
    }
    BLKLEN;
    
    if(header1.flag_cooling){
      
      BLKLEN;
      for(i=1;i<=Ngas;i++)
      {
        xyz[0]= P[i].Ne;
        fwrite(xyz, sizeof(float), 1, fd);
      }
      BLKLEN;
      
      BLKLEN;
      for(i=1;i<=Ngas;i++)
      {
        xyz[0]= P[i].Nh;
        fwrite(xyz, sizeof(float), 1, fd);
      }
      BLKLEN;
      
    }
    
    BLKLEN;
    for(i=1;i<=Ngas;i++)
    {
      xyz[0]= P[i].Hsml;
      fwrite(xyz, sizeof(float), 1, fd);
    }
    BLKLEN;
    
    if(header1.flag_sfr){
      
      BLKLEN;
      for(i=1;i<=Ngas;i++)
      {
        xyz[0]= P[i].Sfr;
        fwrite(xyz, sizeof(float), 1, fd);
      }
      BLKLEN;
      
    }
    
    
    if(header1.flag_stellarage){
      
      BLKLEN;
      for(i=1;i<=Ngas;i++)
      {
        xyz[0]= P[i].SAge;
        fwrite(xyz, sizeof(float), 1, fd);
      }
      BLKLEN;
      
    }
    
    
    
  }
  
  fclose(fd);
  return 0;
}




int merge(void)
{
  
  int i,j;
  
  Ngas= Ngas1+Ngas2;
  Nhalo= Nhalo1+Nhalo2;
  Ndisk= Ndisk1+Ndisk2;
  Nbulge= Nbulge1+Nbulge2;
  Nstars= Nstars1+Nstars2;
  NumPart= NumPart1+NumPart2;
  if (NumPart!=Ngas+Nhalo+Ndisk+Nbulge+Nstars) printf("ERROR: Something went wrong with the numbers of particles\n");
  
  allocate_memory();
  
  header.npart[0]= Ngas;
  header.npart[1]= Nhalo;
  header.npart[2]= Ndisk;
  header.npart[3]= Nbulge;
  header.npart[4]= Nstars;
  header.npart[5]= 0;
  
  if (header1.mass[0]!=header2.mass[0]) header.mass[0]=0;
  else header.mass[0]=header1.mass[0];
  if (header1.mass[1]!=header2.mass[1]) header.mass[1]=0;
  else header.mass[1]=header1.mass[1];
  if (header1.mass[2]!=header2.mass[2]) header.mass[2]=0;
  else header.mass[2]=header1.mass[2];
  if (header1.mass[3]!=header2.mass[3]) header.mass[3]=0;
  else header.mass[3]=header1.mass[3];
  if (header1.mass[4]!=header2.mass[4]) header.mass[4]=0;
  else header.mass[4]=header1.mass[4];
  if (header1.mass[5]!=header2.mass[5]) header.mass[5]=0;
  else header.mass[5]=header1.mass[5];
  
  Time= Time1;
  Redshift= Redshift1;
  FlagSfr= FlagSfr1;
  Flagfeedback= Flagfeedback1;
  
  Ngastot= Ngastot1+Ngastot2;
  Nhalotot= Nhalotot1+Nhalotot2;
  Ndisktot= Ndisktot1+Ndisktot2;
  Nbulgetot= Nbulgetot1+Nbulgetot2;
  Nstarstot= Nstarstot1+Nstarstot2;
  NumParttot= NumParttot1+NumParttot2;
  if (NumParttot!=Ngastot+Nhalotot+Ndisktot+Nbulgetot+Nstarstot) printf("ERROR: Something went wrong with the numbers of particles (total)\n");
  
  header.npartTotal[0]= Ngas;
  header.npartTotal[1]= Nhalo;
  header.npartTotal[2]= Ndisk;
  header.npartTotal[3]= Nbulge;
  header.npartTotal[4]= Nstars;
  header.npartTotal[5]= 0;
  
  FlagCooling= FlagCooling1;
  NumFiles= NumFiles1;
  BoxSize= BoxSize2;
  Omega0= Omega01;
  OmegaLambda= OmegaLambda1;
  HubbleParam= HubbleParam1;
  
  header.time= Time;
  header.redshift= Redshift;
  header.num_files= NumFiles;
  header.flag_sfr=FlagSfr;
  header.flag_feedback=Flagfeedback;
  header.flag_cooling=FlagCooling;
  header.BoxSize= BoxSize;
  header.Omega0= Omega0;
  header.OmegaLambda= OmegaLambda;
  header.HubbleParam= HubbleParam;
  
  
  
  j=1;
  for (i=1; i<=Ngas1; i++,j++) P[j] = P1[i];
  for (i=1; i<=Ngas2; i++,j++) P[j] = P2[i];
  for (i=1+Ngas1; i<=Ngas1+Nhalo1; i++,j++) P[j] = P1[i];
  for (i=1+Ngas2; i<=Ngas2+Nhalo2; i++,j++) P[j] = P2[i];
  for (i=1+Ngas1+Nhalo1; i<=Ngas1+Nhalo1+Ndisk1; i++,j++) P[j] = P1[i];
  for (i=1+Ngas2+Nhalo2; i<=Ngas2+Nhalo2+Ndisk2; i++,j++) P[j] = P2[i];
  for (i=1+Ngas1+Nhalo1+Ndisk1; i<=Ngas1+Nhalo1+Ndisk1+Nbulge1; i++,j++) P[j] = P1[i];
  for (i=1+Ngas2+Nhalo2+Ndisk2; i<=Ngas2+Nhalo2+Ndisk2+Nbulge2; i++,j++) P[j] = P2[i];
  for (i=1+Ngas1+Nhalo1+Ndisk1+Nbulge1; i<=Ngas1+Nhalo1+Ndisk1+Nbulge1+Nstars1; i++,j++) P[j] = P1[i];
  for (i=1+Ngas2+Nhalo2+Ndisk2+Nbulge2; i<=Ngas2+Nhalo2+Ndisk2+Nbulge2+Nstars2; i++,j++) P[j] = P2[i];
  
  j=1;
  for (i=1; i<=Ngas1; i++,j++) Id[j] = Id1[i];
  for (i=1; i<=Ngas2; i++,j++) Id[j] = Id2[i]+NumPart1;
  for (i=1+Ngas1; i<=Ngas1+Nhalo1; i++,j++) Id[j] = Id1[i];
  for (i=1+Ngas2; i<=Ngas2+Nhalo2; i++,j++) Id[j] = Id2[i]+NumPart1;
  for (i=1+Ngas1+Nhalo1; i<=Ngas1+Nhalo1+Ndisk1; i++,j++) Id[j] = Id1[i];
  for (i=1+Ngas2+Nhalo2; i<=Ngas2+Nhalo2+Ndisk2; i++,j++) Id[j] = Id2[i]+NumPart1;
  for (i=1+Ngas1+Nhalo1+Ndisk1; i<=Ngas1+Nhalo1+Ndisk1+Nbulge1; i++,j++) Id[j] = Id1[i];
  for (i=1+Ngas2+Nhalo2+Ndisk2; i<=Ngas2+Nhalo2+Ndisk2+Nbulge2; i++,j++) Id[j] = Id2[i]+NumPart1;
  for (i=1+Ngas1+Nhalo1+Ndisk1+Nbulge1; i<=Ngas1+Nhalo1+Ndisk1+Nbulge1+Nstars1; i++,j++) Id[j] = Id1[i];
  for (i=1+Ngas2+Nhalo2+Ndisk2+Nbulge2; i<=Ngas2+Nhalo2+Ndisk2+Nbulge2+Nstars2; i++,j++) Id[j] = Id2[i]+NumPart1;
  
  return 0;
  
}


int separate(double Rstart, double Rmin, double e)
{
  
  int    i;
  double p,phi,m,m1,m2,k,G,vphi,vrad;
  double phistart=0.0;
 
  if (e<=(1.0-Rmin/Rstart)/(1.0+Rmin/Rstart)){
    printf("Ellipticity must be larger than %f\n",(1.0-Rmin/Rstart)/(1.0+Rmin/Rstart));
    exit(0);
  }

  
  p= Rmin*(1+e);
  phi= acos((p/Rstart-1.0)/e);
  printf("%f %f %f %f %f %f %f\n",Rmin,e,(1+e),p,Rmin/(1.0-e),p/Rstart-1,phi/M_PI*180.0);
  
  
  m1=0;
  m2=0;
  
  for (i=1; i<=NumPart1; i++) m1+=P1[i].Mass;
  for (i=1; i<=NumPart2; i++) m2+=P2[i].Mass;
  
  m=m1*m2/(m1+m2);
  
  G=43007.1;
  k=(m1+m2)*G;
  
  vrad = sqrt(k/p)*e*sin(phi);
  vphi = sqrt(k/p)*(1.0+e*cos(phi));
  
  printf("Masses of galaxies+halos: %f and %f [10^10M_sun]\n",m1,m2);
  printf("Vrad = %f   Vtan = %f\n",vrad,vphi);
  
  for (i=1; i<=NumPart1; i++)
  {
    P1[i].Pos[0]+= -m2/(m1+m2)*Rstart*cos(phistart);
    P1[i].Pos[1]+= +m2/(m1+m2)*Rstart*sin(phistart);
    P1[i].Pos[2]+= 0;
    P1[i].Vel[0]+= +m2/(m1+m2)*(vrad*cos(phistart)-vphi*sin(phistart));
    P1[i].Vel[1]+= -m2/(m1+m2)*(vrad*sin(phistart)+vphi*cos(phistart));
    P1[i].Vel[2]+= 0;
  }
		
  
  for (i=1; i<=NumPart2; i++)
  {
    P2[i].Pos[0]+= +m1/(m1+m2)*Rstart*cos(phistart);
    P2[i].Pos[1]+= -m1/(m1+m2)*Rstart*sin(phistart);
    P2[i].Pos[2]+= 0;
    P2[i].Vel[0]+= -m1/(m1+m2)*(vrad*cos(phistart)-vphi*sin(phistart));
    P2[i].Vel[1]+= +m1/(m1+m2)*(vrad*sin(phistart)+vphi*cos(phistart));
    P2[i].Vel[2]+= 0;
  }
  
  
  printf("x/y offset  : %f %f\n",Rstart*cos(phi),Rstart*sin(phi));
  
  return 0;
}



int rotate_1(void)
{
  
  int    i;
  double x,y,z,u,v,w;
  
  
  theta1= theta1*2*M_PI/360;
  phi1=   phi1*2*M_PI/360;
  
  printf("%f %f\n",cos(phi1),phi1);
  
  for (i=1; i<=NumPart1; i++)
  {
    x= P1[i].Pos[0];
    y= P1[i].Pos[1];
    z= P1[i].Pos[2];
    u= P1[i].Vel[0];
    v= P1[i].Vel[1];
    w= P1[i].Vel[2];
    
    P1[i].Pos[0]= cos(phi1)*(cos(theta1)*x+sin(theta1)*z)-sin(phi1)*y;
    P1[i].Pos[1]= sin(phi1)*(cos(theta1)*x+sin(theta1)*z)+cos(phi1)*y;
    P1[i].Pos[2]= -sin(theta1)*x+cos(theta1)*z;
    P1[i].Vel[0]= cos(phi1)*(cos(theta1)*u+sin(theta1)*w)-sin(phi1)*v;
    P1[i].Vel[1]= sin(phi1)*(cos(theta1)*u+sin(theta1)*w)+cos(phi1)*v;
    P1[i].Vel[2]= -sin(theta1)*u+cos(theta1)*w;
    
  }
  
  
  return 0;
}



int rotate_2(void)
{
  
  int    i;
  double x,y,z,u,v,w;
  
  
  theta2= theta2*2*M_PI/360;
  phi2=   phi2*2*M_PI/360;
  
  printf("%f %f\n",cos(phi2),phi2);
  
  for (i=1; i<=NumPart2; i++)
  {
    x= P2[i].Pos[0];
    y= P2[i].Pos[1];
    z= P2[i].Pos[2];
    u= P2[i].Vel[0];
    v= P2[i].Vel[1];
    w= P2[i].Vel[2];
    
    P2[i].Pos[0]= cos(phi2)*(cos(theta2)*x+sin(theta2)*z)-sin(phi2)*y;
    P2[i].Pos[1]= sin(phi2)*(cos(theta2)*x+sin(theta2)*z)+cos(phi2)*y;
    P2[i].Pos[2]= -sin(theta2)*x+cos(theta2)*z;
    P2[i].Vel[0]= cos(phi2)*(cos(theta2)*u+sin(theta2)*w)-sin(phi2)*v;
    P2[i].Vel[1]= sin(phi2)*(cos(theta2)*u+sin(theta2)*w)+cos(phi2)*v;
    P2[i].Vel[2]= -sin(theta2)*u+cos(theta2)*w;
    
  }
  
  return 0;
  
}





