# IRAF script: photrans_rc1
#
#  gets gain from CIRSI data using photon transfer method.
#
#  Mike Hoenig, December '98
#
#  17-12-98: uses reset-corrected images and just one run number.
#  21-12-98: added option to use a sub-range of images
#  11-01-99: now subtracts 2 rc'ed frames from each other to avoid flat field


procedure photrans_rc2 (runno1, runno2)

string runno1   {prompt="First run number to use"}
string runno2   {prompt="Second run number to use"}
string chipquad {"c1q2", prompt="Chip and quadrant (eg. 'c1q2')"}
string avg      {"mean", prompt="Type of average to be used by GSTAT",
                 enum="mean|midpt|mode"}
string xrang   	{"*", prompt="Range of x-coordinates to be used"}		 
string yrang   	{"*", prompt="Range of y-coordinates to be used"}
int first	{1, prompt="Starting sequence no. of image to use"}		 
int last	{0, prompt="Final sequence no. of image to use (0 for all)"}		 
bool deleteim	{yes, prompt="Delete reset-corrected images?"}

begin


    # Define local variables

    string l_runno1  	    # run no. 1
    string l_runno2  	    # run no. 2
    string l_chipquad	    # chip, quadrant
    string l_avg	    # GSTAT average to use
    bool l_deleteim 	    # delete images at end?
    string l_xrang  	    # x range
    string l_yrang  	    # y range

    file imglist1  	    # filename of 1st list of images
    file imglist2  	    # filename of 2nd list of images
    string immask1  	    # expression for 1st list of images
    string immask2  	    # expression for 2nd list of images
    int noimages1  	    # no. of input images - 1
    int noimages2  	    # no. of input images - 2
    string imgname1 	    # name of input image in 1st list
    string imgname2 	    # name of input image in 2nd list
    string rootimgname	    # the image filename, without sequence no. (used in rc)
    string rootimgname1     # the image filename, without sequence no.
    string rootimgname2     # dto. for run no. 2
    string imgname  	    # temp. file for scanning for filenames
    real imavg	    	    # average level of image
    real variance   	    # the variance from the - images
    file datafile	    # the final data file (for plotting)
    string lab_x     	    # x-axis label (complex string concatenation possible)
    string lab_y     	    # y-axis label
    string lab_title	    # title of graph
    string pfs1     	    # temp. string while scanning output from polyfit
    string pfs2     	    # dto.
    real slope	    	    # slope from curve fit
    real gain	    	    # 1/slope
    file fitfile    	    # the file containing the fitted data
    string fs1	    	    # dummy string used for scanning
    string gain_form	    # the string containing the formatted value of the gain
    string imgname_sec	    # the image name string, including the [X:X,X:X] info
    real rdnoise    	    # the read noise
    string rdnoise_form     # formatted string of rd. noise
    file rsetim    	    # the name of the reset image.
    int idot	    	    # the position of the . in the filename
    string rcimgname	    # name of the reset corrected image.
    int n   	    	    # counter for image no. in sequence
    string nstr     	    # same as above but formatted
    file rcimgslist1 	    # the list of rc'ed images
    file rcimgslist2 	    # the list of rc'ed images
    file rcimgslist1_2 	    # the list of rc'ed images - 1 + 2 pasted together
    string rs1	    	    # temp. string used to parse reset image
    int l_first, l_last     # first and last sequence nos.
    file plusnamlist	    # list of output + names
    file minusnamlist	    # dto. for -
    int i1  	    	    # integer used for substringing in the filename
    int i2  	    	    # dto.
    string plusname 	    # the output name of the imarith + file
    string minusname	    # the output name of the imarith - file
    file plusminuslist	    # the above two pasted together
    string plusimg  	    # temp. string used while scanning plusminuslist
    string minusimg 	    # dto.
    
    l_runno1 = runno1
    l_runno2 = runno2
    l_chipquad = chipquad
    l_avg = avg
    l_deleteim = deleteim
    l_xrang = xrang
    l_yrang = yrang
    l_first = first
    l_last = last

    # trying to make paste a native task in this program... will it work?
    
    task $paste = "$foreign"
    
    
    # Make sure STSDAS is loaded (needed for GSTAT)

    if (!defpac("stsdas"))
    	{
    	stsdas motd-
    	}


    imglist1=mktemp("filenames1")//".list"
    imglist2=mktemp("filenames2")//".list"
    if (access(imglist1)) delete (imglist1, ver-, >& "dev$null")
    if (access(imglist2)) delete (imglist2, ver-, >& "dev$null")
    
    # The following construction is the best I could do, the only thing it doesn't
    # catch is files without _any_ prefix - which should hopefully be a thing of the
    # past anyway.
    
    immask1="*_"//l_runno1//"_"//l_chipquad//"_???.fits"
    immask2="*_"//l_runno2//"_"//l_chipquad//"_???.fits"
    
    sections (immask1, option="fullname", > imglist1)
    noimages1 = sections.nimages
    sections (immask2, option="fullname", > imglist2)
    noimages2 = sections.nimages
    
    if (noimages1!=noimages2)
    	{
	print ("")
	print ("ERROR: There seems to be a mismatch between the number of images")
	print (" for each run number!")
        print ("run no. "//l_runno1//": "//noimages1//" images")
        print ("run no. "//l_runno2//": "//noimages2//" images")
        print ("Please check your input files again...")
	print ("")
	beep
	bye
	}

    print ("")
    print ("Total no. of images per run number is: ", noimages2)

    if ((l_first == 0) || (l_first < 1)) l_first = 1
    if ((l_last == 0) || (l_last > noimages2)) l_last = noimages2
    
    print ("Using sequence numbers ", l_first, "to ", l_last, "for analysis.")
    
    # Now perform reset correction!
    # Subtract first image in list from all the subsequent ones...
    # NB. need to do this twice over, once for each run number.
    
    print ("")
    print ("Performing reset correction ...")

    # First run number:
    
    # Get name of reset image - NB. assuming this will always be the first image
    # in the list
    
    type (imglist1) | scan (rs1)
    rsetim = rs1
    
    print ("")
    print ("1) Run number "//l_runno1)
    print ("Reset image is: ", rsetim)
    # Next, go down the list one by one and subtract the reset image from each
    
    rcimgslist1=mktemp("rcfilenames1")//".list"
    if (access(rcimgslist1)) delete (rcimgslist1, ver-, >& "dev$null")

    n=1
    
    list=imglist1
    while (fscan (list, imgname) != EOF)
        {
	# be sure to exclude reset image & check for limiting sequence nos.
	
	if ((imgname != rsetim) && (n >= l_first) && (n <= l_last))
	    {
	    # Parse out the sequence number (last 3 digits before the .)
	
	    idot = stridx (".", imgname)
	    rootimgname = substr(imgname, 1, (idot-4))
	
	    # Add a leading zero to the image number if required:
	
	    if (strlen(str(n))==1)
	    	{
	    	nstr="0"//str(n)
	    	}
	    else nstr=str(n)    
	
	    rcimgname = rootimgname//"rc"//nstr//"01.fits"
	
	    if (access(rcimgname)) delete (rcimgname, ver-, >& "dev$null")
	
	    print ("imarith "//imgname//" - "//rsetim//" "//rcimgname)
	    imarith (imgname, "-", rsetim, rcimgname)
	
	    print (rcimgname, >> rcimgslist1)
	    }
	    
	n+=1
	}
    
    # Second run number:
    
    # Get name of reset image - NB. assuming this will always be the first image
    # in the list
    
    type (imglist2) | scan (rs1)
    rsetim = rs1
    
    print ("")
    print ("2) Run number "//l_runno2)
    print ("Reset image is: ", rsetim)

    # Next, go down the list one by one and subtract the reset image from each
    
    rcimgslist2=mktemp("rcfilenames2")//".list"
    if (access(rcimgslist2)) delete (rcimgslist2, ver-, >& "dev$null")

    n=1
    
    list=imglist2
    while (fscan (list, imgname) != EOF)
        {
	# be sure to exclude reset image & check for limiting sequence nos.
	
	if ((imgname != rsetim) && (n >= l_first) && (n <= l_last))
	    {
	    # Parse out the sequence number (last 3 digits before the .)
	
	    idot = stridx (".", imgname)
	    rootimgname = substr(imgname, 1, (idot-4))
	
	    # Add a leading zero to the image number if required:
	
	    if (strlen(str(n))==1)
	    	{
	    	nstr="0"//str(n)
	    	}
	    else nstr=str(n)    
	
	    rcimgname = rootimgname//"rc"//nstr//"01.fits"
	
	    if (access(rcimgname)) delete (rcimgname, ver-, >& "dev$null")
	
	    print ("imarith "//imgname//" - "//rsetim//" "//rcimgname)
	    imarith (imgname, "-", rsetim, rcimgname)
	
	    print (rcimgname, >> rcimgslist2)
	    }
	    
	n+=1
	}
    
    print ("")
    print ("Reset correction done.")
    
    # Next, subtract one set of images from the other!
    
    # Paste the two lists of filenames together...
    # should work since I defined paste as a foreign task?
    
    rcimgslist1_2=mktemp("rcfilenames1_2")//".list"
    if (access(rcimgslist1_2)) delete (rcimgslist1_2, ver-, >& "dev$null")

    paste (rcimgslist1, rcimgslist2, >> rcimgslist1_2)
    
    # Do the required imarith work
    
    print ("")
    print ("Performing imarith's ...")
    
    plusnamlist=mktemp("plusnames")//".list"
    minusnamlist=mktemp("minusnames")//".list"
    if (access(plusnamlist)) delete (plusnamlist, ver-, >& "dev$null")
    if (access(minusnamlist)) delete (minusnamlist, ver-, >& "dev$null")

    list=rcimgslist1_2
    while (fscan (list, imgname1, imgname2) != EOF)
        {
	print ("Now processing images: ", imgname1, " and ", imgname2)
	
        # clever construction (and it works too!) to get rid of the file prefix:
	# 1. get first char. in run no. and its position in the filename
	
	i2 = stridx (substr(l_runno2, 1, 1), imgname2)
	
	# 2. then lift out the substring of the filename starting with the run no.
	
	rootimgname2=substr(imgname2, i2, (strlen(imgname2)))
	
	# same again for other image:

	i1 = stridx (substr(l_runno1, 1, 1), imgname1)
	rootimgname1=substr(imgname1, i1, (strlen(imgname1)))
	
	# now JUST DO IT
	
	plusname="photrans_"//l_runno1//"+"//rootimgname2
	minusname="photrans_"//l_runno2//"-"//rootimgname1
	if (access(plusname)) delete (plusname, ver-, >& "dev$null")
	if (access(minusname)) delete (minusname, ver-, >& "dev$null")
	
	imarith (imgname1, "+", imgname2, plusname)
	print (plusname, >> plusnamlist)
	
	imarith (imgname2, "-", imgname1, minusname)
	print (minusname, >> minusnamlist)
	}
    
    print ("imarith's done.")
    
    # Now, get the image statistics with gstat

    print ("")
    print ("Performing image statistics ...")
    
    plusminuslist=mktemp("plusminus")//".list"
    if (access(plusminuslist)) delete (plusminuslist, ver-, >& "dev$null")

    paste (plusnamlist, minusnamlist, >> plusminuslist)
    
    datafile="photrans.data"
    if (access(datafile)) delete (datafile, ver-, >& "dev$null")
    
    # Now, get signal level and variance from the images...
    # *new improved* loop!

    list=plusminuslist
    while (fscan (list, plusimg, minusimg) != EOF)
    	{
    	# first, get average from + images
	
	imgname_sec=plusimg//"["//l_xrang//","//l_yrang//"]"

	gstatistics (imgname_sec, fields=l_avg, >& "dev$null")
	
        if (l_avg=="mean") imavg=gstpar.mean
        else if (l_avg=="midpt") imavg=gstpar.midpt
        else if (l_avg=="mode") imavg=gstpar.mode
	
	# next, get stddev from - images
	
	imgname_sec=minusimg//"["//l_xrang//","//l_yrang//"]"

	gstatistics (imgname_sec, fields="stddev", >& "dev$null")

	variance = (gstpar.stddev)**2
	
	print (imavg/2, variance/2, >> datafile)
	}
	
    # Ready to do the fit...
    
    # First run through polyfit - get slope
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=no) |
    scan (pfs1, pfs2)
    
    slope=real(pfs2)
    gain=1/slope
    
    # NB. get the read noise in electrons here (* by gain)
    # also checking for >0
    
    if ((real(pfs1))>0)
    	{
    	rdnoise=(sqrt(real(pfs1)))*gain
	}
    else rdnoise=0
    
    print ("")
    print ("    slope = ", pfs2)
    print ("==> gain  = ", gain)
    print ("    y-intercept = ", pfs1)
    print ("==> read noise = ", rdnoise)

    # second run through polyfit - get fitted x, y
    
    fitfile=mktemp("fitted")//".list"
    if (access(fitfile)) delete (fitfile, ver-, >& "dev$null")
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=yes, >>fitfile)
    
    # ...and plot them with sgraph!
    
    print ("")
    print ("Plotting ...")
    
    # First run through sgraph - plot data points
    
    # clever way of creating a formatted value for the gain and read noise:
    printf ("%.2f\n", gain) | scan (fs1)
    gain_form=fs1
    
    if (rdnoise!=0)
    	{
    	printf ("%.2f\n", rdnoise) | scan (fs1)
    	rdnoise_form=fs1
	}
    else rdnoise_form="N/A"	
    
    # Create labels:
    lab_x=l_avg//" of ((rc#"//l_runno1//" + rc#"//l_runno2//") / 2)"
    lab_y="(variance of (rc#"//l_runno2//" - rc#"//l_runno1//")) / 2"
    lab_title="RC"//l_first//"-"//l_last//" #"//l_runno1//"&"//l_runno2//" "//l_chipquad//" "//"["//l_xrang//","//l_yrang//"]  Gain="//gain_form//" Rd.noise="//rdnoise_form
    
    sgraph (datafile, device="stdgraph", append=no, fill=yes, pointmo=yes, 
    	    marker="circle", szmarke=7, wl=0, wr=0, wb=0, wt=0, logx=no,
	    logy=no, box=yes, grid=no, ticklab=yes, xlabel=lab_x, ylabel=lab_y,  
	    title=lab_title, sysid=yes)
    
    # Second run through sgraph - get fitted data to plot curve
    
    sgraph (fitfile, append=yes, pointmode=no, pattern="dashed")
    
    # Clean up
    
    print ("")
    print ("Cleaning up ...")
    
    if (l_deleteim)
    	{
	imdelete("@"//rcimgslist1, ver-, >& "dev$null")
	imdelete("@"//rcimgslist2, ver-, >& "dev$null")
	}
    delete (fitfile, ver-, >& "dev$null")
    delete ("@"//plusnamlist, ver-, >& "dev$null")
    delete ("@"//minusnamlist, ver-, >& "dev$null")
    delete (plusnamlist, ver-, >& "dev$null")
    delete (minusnamlist, ver-, >& "dev$null")
    delete (plusminuslist, ver-, >& "dev$null")
    delete (imglist1, ver-, >& "dev$null")
    delete (imglist2, ver-, >& "dev$null")
    delete (rcimgslist1, ver-, >& "dev$null")
    delete (rcimgslist2, ver-, >& "dev$null")
    delete (rcimgslist1_2, ver-, >& "dev$null")
    
    print ("")    


end
