# IRAF script: photrans_rc1
#
#  gets gain from CIRSI data using photon transfer method.
#
#  Mike Hoenig, December '98
#
#  17-12-98: uses reset-corrected images and just one run number.
#  21-12-98: added option to use a sub-range of images
#  25-01-99: added option not to reset-correct (for pre-rc'ed images)


procedure photrans_rc1 (runno)

string runno    {prompt="Run number to use"}
string chipquad {"c1q2", prompt="Chip and quadrant (eg. 'c1q2')"}
bool rcorr  	{yes, prompt="Reset-correct images?"}
string avg      {"mean", prompt="Type of average to be used by GSTAT",
                 enum="mean|midpt|mode"}
string xrang   	{"*", prompt="Range of x-coordinates to be used"}		 
string yrang   	{"*", prompt="Range of y-coordinates to be used"}
int first	{1, prompt="Starting sequence no. of image to use"}		 
int last	{0, prompt="Final sequence no. of image to use (0 for all)"}		 
bool deleteim	{yes, prompt="Delete reset-corrected images?"}

begin


    # Define local variables

    string l_runno  	    # run no.
    string l_chipquad	    # chip, quadrant
    string l_avg	    # GSTAT average to use
    bool l_deleteim 	    # delete images at end?
    string l_xrang  	    # x range
    string l_yrang  	    # y range
    bool l_rcorr

    file imglist  	    # filename of list of images
    string immask  	    # expression for list of images
    int noimages  	    # no. of input images
    string rootimgname      # the image filename, without sequence no.
    string imgname  	    # temp. file for scanning for filenames
    real imavg	    	    # average level of image
    real variance   	    # the variance from the - images
    real var_div_2  	    # the above divided by 2
    file datafile	    # the final data file (for plotting)
    string lab_x     	    # x-axis label (complex string concatenation possible)
    string lab_y     	    # y-axis label
    string lab_title	    # title of graph
    string pfs1     	    # temp. string while scanning output from polyfit
    string pfs2     	    # dto.
    real slope	    	    # slope from curve fit
    real gain	    	    # 1/slope
    file fitfile    	    # the file containing the fitted data
    string fs1	    	    # dummy string used for scanning
    string gain_form	    # the string containing the formatted value of the gain
    string imgname_sec	    # the image name string, including the [X:X,X:X] info
    real rdnoise    	    # the read noise
    string rdnoise_form     # formatted string of rd. noise
    file rsetim    	    # the name of the reset image.
    int idot	    	    # the position of the . in the filename
    string rcimgname	    # name of the reset corrected image.
    int n   	    	    # counter for image no. in sequence
    string nstr     	    # same as above but formatted
    file rcimgslist 	    # the list of rc'ed images
    string rs1	    	    # temp. string used to parse reset image
    int l_first, l_last     # first and last sequence nos.
    
    l_runno = runno
    l_chipquad = chipquad
    l_avg = avg
    l_deleteim = deleteim
    l_xrang = xrang
    l_yrang = yrang
    l_first = first
    l_last = last
    l_rcorr = rcorr


    # Make sure STSDAS is loaded (needed for GSTAT)

    if (!defpac("stsdas"))
    	{
    	stsdas motd-
    	}


    imglist=mktemp("filenames")//".list"
    if (access(imglist)) delete (imglist, ver-, >& "dev$null")
    
    # The following construction is the best I could do, the only thing it doesn't
    # catch is files without _any_ prefix - which should hopefully be a thing of the
    # past anyway.
    
    immask="*_"//l_runno//"_"//l_chipquad//"_???.fits"
    
    print ("")
    print (immask)
    
    sections (immask, option="fullname", > imglist)
    noimages = sections.nimages

    print ("")
    print ("Total no. of images in this run number is: ", noimages)

    if (l_first < 1) l_first = 1
    if ((l_last == 0) || (l_last > noimages)) l_last = noimages
    
    print ("Using sequence numbers ", l_first, "to ", l_last, "for analysis.")
    
    if (l_rcorr)
    	{
    	# Now perform reset correction!
    	# Subtract first image in list from all the subsequent ones...
    
    	# Get name of reset image - NB. assuming this will always be the first 
	# image in the list
    
    	type (imglist) | scan (rs1)
    	rsetim = rs1
    
    	print ("")
    	print ("Reset image is: ", rsetim)
    	print ("")
    	print ("Performing reset correction ...")

    	# Next, go down the list one by one and subtract the reset image from 
	# each
    
    	rcimgslist=mktemp("rcfilenames")//".list"
    	if (access(rcimgslist)) delete (rcimgslist, ver-, >& "dev$null")

    	n=1
    
    	list=imglist
    	while (fscan (list, imgname) != EOF)
            {
	    # be sure to exclude reset image & check for limiting sequence nos.
	
	    if ((imgname != rsetim) && (n >= l_first) && (n <= l_last))
	    	{
	    	# print ("Now reset correcting: ", imgname)
	
	    	# Parse out the sequence number (last 3 digits before the .)
	
	    	idot = stridx (".", imgname)
	    	rootimgname = substr(imgname, 1, (idot-4))
	
	    	# Add a leading zero to the image number if required:
	
	    	if (strlen(str(n))==1)
	    	    {
	    	    nstr="0"//str(n)
	    	    }
	    	else nstr=str(n)    
	
	    	rcimgname = rootimgname//"rc"//nstr//"01.fits"
	
	    	if (access(rcimgname)) delete (rcimgname, ver-, >& "dev$null")
	
	    	print ("imarith "//imgname//" - "//rsetim//" "//rcimgname)
	    	imarith (imgname, "-", rsetim, rcimgname)
	
	    	print (rcimgname, >> rcimgslist)
	    	}
	    
	    n+=1
	    }
    
    	print ("Reset correction done.")
	}

    else
    	# dummy rc routine:
    	{
    	rcimgslist=mktemp("rcfilenames")//".list"
    	if (access(rcimgslist)) delete (rcimgslist, ver-, >& "dev$null")

	type (imglist) | scan (rs1)
    	rsetim = rs1
	
	n=1
	
	list=imglist
    	while (fscan (list, imgname) != EOF)
	    {
	    if ((imgname != rsetim) && (n >= l_first) && (n <= l_last))
	    	{
	    	print (imgname, >> rcimgslist)
	    	}
	    n+=1
	    }
	}    

    # Now get the image statistics with gstat

    print ("")
    print ("Performing image statistics ...")

    datafile="photrans.data"
    if (access(datafile)) delete (datafile, ver-, >& "dev$null")

    # Now, get signal level and variance from the images...
    # *new improved* loop!

    list=rcimgslist
    while (fscan (list, imgname) != EOF)
    	{
    	imgname_sec=imgname//"["//l_xrang//","//l_yrang//"]"

	gstatistics (imgname_sec, fields=l_avg//",stddev", >& "dev$null")
	
        if (l_avg=="mean") imavg=gstpar.mean
        else if (l_avg=="midpt") imavg=gstpar.midpt
        else if (l_avg=="mode") imavg=gstpar.mode
	
	variance = (gstpar.stddev)**2
	var_div_2 = variance / 2
	
	print (imavg, var_div_2, >> datafile)
	}
	
    # ...do the fit...
    
    # First run through polyfit - get slope
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=no) |
    scan (pfs1, pfs2)
    
    slope=real(pfs2)
    gain=1/slope
    
    # NB. get the read noise in electrons here (* by gain)
    # also checking for >0
    
    if ((real(pfs1))>0)
    	{
    	rdnoise=(sqrt(real(pfs1)))*gain
	}
    else rdnoise=0
    
    print ("")
    print ("    slope = ", pfs2)
    print ("==> gain  = ", gain)
    print ("    y-intercept = ", pfs1)
    print ("==> read noise = ", rdnoise)

    # second run through polyfit - get fitted x, y
    
    fitfile=mktemp("fitted")//".list"
    if (access(fitfile)) delete (fitfile, ver-, >& "dev$null")
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=yes, >>fitfile)
    
    # ...and plot them with sgraph!
    
    print ("")
    print ("Plotting ...")
    
    # First run through sgraph - plot data points
    
    # clever way of creating a formatted value for the gain and read noise:
    printf ("%.2f\n", gain) | scan (fs1)
    gain_form=fs1
    
    if (rdnoise!=0)
    	{
    	printf ("%.2f\n", rdnoise) | scan (fs1)
    	rdnoise_form=fs1
	}
    else rdnoise_form="N/A"	
    
    # Create labels:
    lab_x=l_avg
    lab_y="variance  / 2"
    lab_title="RC "//l_first//"-"//l_last//" Run#"//l_runno//" "//l_chipquad//" "//"["//l_xrang//","//l_yrang//"]  Gain="//gain_form//" Rd.noise="//rdnoise_form
    
    sgraph (datafile, device="stdgraph", append=no, fill=yes, pointmo=yes, 
    	    marker="circle", szmarke=7, wl=0, wr=0, wb=0, wt=0, logx=no,
	    logy=no, box=yes, grid=no, ticklab=yes, xlabel=lab_x, ylabel=lab_y,  
	    title=lab_title, sysid=yes)
    
    # Second run through sgraph - get fitted data to plot curve
    
    sgraph (fitfile, append=yes, pointmode=no, pattern="dashed")
    
    # Clean up
    
    print ("")
    print ("Cleaning up ...")
    
    if ((l_deleteim) && (l_rcorr))
    	{
	imdelete("@"//rcimgslist, ver-, >& "dev$null")
	}
    delete (imglist, ver-, >& "dev$null")
    delete (fitfile, ver-, >& "dev$null")
    delete (rcimgslist, ver-, >& "dev$null")
    
    print ("")    


end
