# IRAF script: photrans_gen
#
#  gets gain from CIRSI data using photon transfer method.
#  -this version can take a list of any files as an input; so more general.
#
#  Mike Hoenig, 15-01-99


procedure photrans_gen (infiles)

file infiles	{prompt="Input files"}
string avg      {"mean", prompt="Type of average to be used by GSTAT",
                 enum="mean|midpt|mode"}
string xrang   	{"*", prompt="Range of x-coordinates to be used"}		 
string yrang   	{"*", prompt="Range of y-coordinates to be used"}

begin


    # Define local variables

    file l_infiles  	    # input files
    string l_avg	    # GSTAT average to use
    string l_xrang  	    # x range
    string l_yrang  	    # y range

    file imglist  	    # filename of list of images
    int noimages  	    # no. of input images
    string imgname  	    # temp. file for scanning for filenames
    real imavg	    	    # average level of image
    real variance   	    # the variance from the - images
    file datafile	    # the final data file (for plotting)
    string lab_x     	    # x-axis label (complex string concatenation possible)
    string lab_y     	    # y-axis label
    string lab_title	    # title of graph
    string pfs1     	    # temp. string while scanning output from polyfit
    string pfs2     	    # dto.
    real slope	    	    # slope from curve fit
    real gain	    	    # 1/slope
    file fitfile    	    # the file containing the fitted data
    string fs1	    	    # dummy string used for scanning
    string gain_form	    # the string containing the formatted value of the gain
    string imgname_sec	    # the image name string, including the [X:X,X:X] info
    real rdnoise    	    # the read noise
    string rdnoise_form     # formatted string of rd. noise
    
    l_infiles = infiles
    l_avg = avg
    l_xrang = xrang
    l_yrang = yrang


    # Make sure STSDAS is loaded (needed for GSTAT)

    if (!defpac("stsdas"))
    	{
    	stsdas motd-
    	}


    imglist=mktemp("filenames")//".list"
    if (access(imglist)) delete (imglist, ver-, >& "dev$null")
    
    sections (l_infiles, option="fullname", > imglist)
    noimages = sections.nimages

    print ("")
    print ("Total no. of images is: ", noimages)

    # Now get the image statistics with gstat

    print ("")
    print ("Now performing image statistics ...")

    datafile="photrans.data"
    if (access(datafile)) delete (datafile, ver-, >& "dev$null")

    # Now, get signal level and variance from the images...
    # *new improved* loop!

    list=imglist
    while (fscan (list, imgname) != EOF)
    	{
    	imgname_sec=imgname//"["//l_xrang//","//l_yrang//"]"

	gstatistics (imgname_sec, fields=l_avg//",stddev", >& "dev$null")
	
        if (l_avg=="mean") imavg=gstpar.mean
        else if (l_avg=="midpt") imavg=gstpar.midpt
        else if (l_avg=="mode") imavg=gstpar.mode
	
	variance = (gstpar.stddev)**2
	
	print (imavg, variance/2, >> datafile)
	}
	
    # ...do the fit...
    
    # First run through polyfit - get slope
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=no) |
    scan (pfs1, pfs2)
    
    slope=real(pfs2)
    gain=1/slope
    
    # NB. get the read noise in electrons here (* by gain)
    # also checking for >0
    
    if ((real(pfs1))>0)
    	{
    	rdnoise=(sqrt(real(pfs1)))*gain
	}
    else rdnoise=0
    
    print ("")
    print ("    slope = ", pfs2)
    print ("==> gain  = ", gain)
    print ("    y-intercept = ", pfs1)
    print ("==> read noise = ", rdnoise)

    # second run through polyfit - get fitted x, y
    
    fitfile=mktemp("fitted")//".list"
    if (access(fitfile)) delete (fitfile, ver-, >& "dev$null")
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=yes, >>fitfile)
    
    # ...and plot them with sgraph!
    
    print ("")
    print ("Plotting ...")
    
    # First run through sgraph - plot data points
    
    # clever way of creating a formatted value for the gain and read noise:
    printf ("%.2f\n", gain) | scan (fs1)
    gain_form=fs1
    
    if (rdnoise!=0)
    	{
    	printf ("%.2f\n", rdnoise) | scan (fs1)
    	rdnoise_form=fs1
	}
    else rdnoise_form="N/A"	
    
    # Create labels:
    lab_x=l_avg
    lab_y="variance  / 2"
    lab_title=l_infiles//"["//l_xrang//","//l_yrang//"]  Gain="//gain_form//"  Read noise="//rdnoise_form
    
    sgraph (datafile, device="stdgraph", append=no, fill=yes, pointmo=yes, 
    	    marker="circle", szmarke=7, wl=0, wr=0, wb=0, wt=0, logx=no,
	    logy=no, box=yes, grid=no, ticklab=yes, xlabel=lab_x, ylabel=lab_y,  
	    title=lab_title, sysid=yes)
    
    # Second run through sgraph - get fitted data to plot curve
    
    sgraph (fitfile, append=yes, pointmode=no, pattern="dashed")
    
    # Clean up
    
    print ("")
    print ("Cleaning up ...")
    
    delete (imglist, ver-, >& "dev$null")
    delete (fitfile, ver-, >& "dev$null")
    
    print ("")    


end
