# IRAF script: photrans_ds
#
#  gets gain from CIRSI data using photon transfer method.
#  -this version can take a list of any files as an input; so more general.
#
#  Mike Hoenig, 15-01-99
#
#  19-01-99: adding option to subtract _values_ of the dark images
#  21-01-99: finally finished! (along with rcimages)
#  22-01-99: added option for using certain sequence nos. only


procedure photrans_ds (flats,darks)

file flats	{prompt="Flat-field exposures"}
file darks	{prompt="Dark frames"}
bool flatrc 	{no, prompt="Reset-correct flats?"}
bool darkrc 	{no, prompt="Reset-correct darks?"}
string avg      {"mean", prompt="Type of average to be used on images",
                 enum="mean|midpt|mode"}
string xrang   	{"*", prompt="Range of x-coordinates to be used"}		 
string yrang   	{"*", prompt="Range of y-coordinates to be used"}
int first   	{2, prompt="Starting sequence no. of image to use"}
int last        {0, prompt="Final sequence no. of image to use (0 for last)"}             
bool delrcim	{no, prompt="Delete reset-corrected images?"}

begin


    # Define local variables

    file l_flats  	    # input flats
    file l_darks  	    # input darks
    bool l_flatrc,l_darkrc  # reset-correct flats/darks?
    string l_avg	    # GSTAT average to use
    string l_xrang  	    # x range
    string l_yrang  	    # y range
    bool l_delrcim  	    # delete rc'ed images?
    int l_first, l_last     # first and last sequence nos.

    file fimglist,dimglist  # filename of list of images
    int nofimages,nodimages # no. of input images
    real flatavg,darkavg    # average level of image
    real fvariance,dvariance # the variance from the - images
    file datafile	    # the final data file (for plotting)
    string lab_x     	    # x-axis label (complex string concatenation possible)
    string lab_y     	    # y-axis label
    string lab_title	    # title of graph
    string pfs1,pfs2        # temp. string while scanning output from polyfit
    real slope	    	    # slope from curve fit
    real gain	    	    # 1/slope
    file fitfile    	    # the file containing the fitted data
    string fs1	    	    # dummy string used for scanning
    string gain_form	    # the string containing the formatted value of the gain
    string sflat_sec,sdark_sec # the image name string, including the [X:X,X:X] info
    real rdnoise    	    # the read noise
    string rdnoise_form     # formatted string of rd. noise
    string check    	    # everything in _rcimages go OK?
    file frcimglist 	    # list of rc'ed flats
    file drcimglist 	    # list of rc'ed darks
    file flatimglist	    # final list of flats to use
    file darkimglist	    # same for darks
    file flatdarkimglist    # ...and both of them pasted together into one
    string sflat,sdark      # temp. string used when scanning the above
    string frctext,drctext  # RC label (or none) in image caption
    int n   	    	    # counter for image no. in sequence
    
    
    l_flats = flats
    l_darks = darks
    l_flatrc = flatrc
    l_darkrc = darkrc
    l_avg = avg
    l_xrang = xrang
    l_yrang = yrang
    l_delrcim = delrcim
    l_first = first
    l_last = last


    # Make paste a native task in this program...

    task $paste = "$foreign"


    # Make sure STSDAS is loaded (needed for GSTAT)

    if (!defpac("stsdas"))
    	{
    	stsdas motd-
    	}

    
    fimglist=mktemp("flats")//".list"
    if (access(fimglist)) delete (fimglist, ver-, >& "dev$null")
    dimglist=mktemp("darks")//".list"
    if (access(dimglist)) delete (dimglist, ver-, >& "dev$null")

    sections (l_flats, option="fullname", > fimglist)
    nofimages = sections.nimages

    sections (l_darks, option="fullname", > dimglist)
    nodimages = sections.nimages

    if (nodimages!=nofimages)
    	{
    	print ("")
    	print ("ERROR: There seems to be a mismatch between the numbers of images!")
    	print (nofimages//" flats, but")
    	print (nodimages//" darks.")
    	print ("Please check your input files again...")
    	print ("")
    	beep
    	bye
    	}

    print ("")
    print ("Total no. of images is: ", nofimages)
    
    # Perform reset correction if necessary
    
    if (l_flatrc)
    	{
    	frcimglist=mktemp("rcflats")//".list"
    	if (access(frcimglist)) delete (frcimglist, ver-, >& "dev$null")

	unlearn rcimages
	print ("")
	print ("Reset-correcting flats ...")
	rcimages (l_flats, frcimglist, sub1st=yes, rccorr=yes, silent=yes)
	check = rcimages.check
	if (check != "OK")
	    {
	    beep
	    print ("")
	    print ("ERROR: problem while performing reset correction")
	    print ("Exiting ...")
	    print ("")
	    bye
	    }
	}
    
    # But if no reset correction, at least get rid of that reset frame!
    
    else
    	{
	unlearn rcimages
	print ("")
	print ("Discarding reset flat frame ...")
	rcimages (l_flats, fimglist, sub1st=yes, rccorr=no, silent=yes)
	check = rcimages.check
	if (check != "OK")
	    {
	    beep
	    print ("")
	    print ("ERROR: problem while discarding reset frame")
	    print ("Exiting ...")
	    print ("")
	    bye
	    }
    	}
		    
	
    # again for darks...

    if (l_darkrc)
    	{
    	drcimglist=mktemp("rcdarks")//".list"
    	if (access(drcimglist)) delete (drcimglist, ver-, >& "dev$null")

	unlearn rcimages
	print ("")
	print ("Reset-correcting darks ...")
	rcimages (l_darks, drcimglist, sub1st=yes, rccorr=yes, silent=yes)
	check = rcimages.check
	if (check != "OK")
	    {
	    beep
	    print ("")
	    print ("ERROR: problem while performing reset correction")
	    print ("Exiting ...")
	    print ("")
	    bye
	    }
	}

    else
    	{
	unlearn rcimages
	print ("")
	print ("Discarding reset dark frame ...")
	rcimages (l_darks, dimglist, sub1st=yes, rccorr=no, silent=yes)
	check = rcimages.check
	if (check != "OK")
	    {
	    beep
	    print ("")
	    print ("ERROR: problem while discarding reset frame")
	    print ("Exiting ...")
	    print ("")
	    bye
	    }
    	}	    
	
    # Now perform dark "subtraction" along with the gstats
    
    print ("")   
    print ("Performing image statistics and subtraction of dark levels ...")
    
    if (l_first < 2) l_first = 2
    if ((l_last == 0) || (l_last > nofimages)) l_last = nofimages
    
    print ("Using sequence numbers ", l_first, "to ", l_last, "for analysis.")
    print ("")
    
    flatimglist=mktemp("flatimgs")//".list"
    if (access(flatimglist)) delete (flatimglist, ver-, >& "dev$null")
    darkimglist=mktemp("darkimgs")//".list"
    if (access(darkimglist)) delete (darkimglist, ver-, >& "dev$null")
    flatdarkimglist=mktemp("flatdarkimgs")//".list"
    if (access(flatdarkimglist)) delete (flatdarkimglist, ver-, >& "dev$null")
    
    # use which images... rc'ed or non-rc'ed?
    
    if (l_flatrc) flatimglist = frcimglist
    else flatimglist = fimglist

    if (l_darkrc) darkimglist = drcimglist
    else darkimglist = dimglist
    	
    # Paste list of images together into one file to make life easier
    
    paste (flatimglist, darkimglist, >> flatdarkimglist)

    # Now get stats and subtract dark values
    
    datafile="photrans.data"
    if (access(datafile)) delete (datafile, ver-, >& "dev$null")

    n=2
    
    list=flatdarkimglist
    while (fscan (list, sflat, sdark) != EOF)
    	{
        # check whether sequence no. is in bounds
	
	if ((n >= l_first) && (n <= l_last))
	    {
	    print ("Now processing- flat: "//sflat//"  dark: "//sdark)
	
	    sflat_sec=sflat//"["//l_xrang//","//l_yrang//"]"

	    gstatistics (sflat_sec, fields=l_avg//",stddev", >& "dev$null")

            if (l_avg=="mean") flatavg=gstpar.mean
            else if (l_avg=="midpt") flatavg=gstpar.midpt
            else if (l_avg=="mode") flatavg=gstpar.mode
	
	    fvariance = (gstpar.stddev)**2
	
    	    sdark_sec=sdark//"["//l_xrang//","//l_yrang//"]"

	
	    gstatistics (sdark_sec, fields=l_avg//",stddev", >& "dev$null")

            if (l_avg=="mean") darkavg=gstpar.mean
            else if (l_avg=="midpt") darkavg=gstpar.midpt
            else if (l_avg=="mode") darkavg=gstpar.mode
	
	    dvariance = (gstpar.stddev)**2
	
	
	    print ((flatavg-darkavg), (fvariance-dvariance)/2, >> datafile)
	    }
	    
	n+=1    
	}
			
    # Now ready to fit and plot!
    
    print ("")
    print ("Plotting ...")
    
    # First run through polyfit - get slope
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=no) |
    scan (pfs1, pfs2)
    
    slope=real(pfs2)
    gain=1/slope
    
    # NB. get the read noise in electrons here (* by gain)
    # also checking for >0
    
    if ((real(pfs1))>0)
    	{
    	rdnoise=(sqrt(real(pfs1)))*gain
	}
    else rdnoise=0
    
    print ("")
    print ("    slope = ", pfs2)
    print ("==> gain  = ", gain)
    print ("    y-intercept = ", pfs1)
    print ("==> read noise = ", rdnoise)

    # second run through polyfit - get fitted x, y
    
    fitfile=mktemp("fitted")//".list"
    if (access(fitfile)) delete (fitfile, ver-, >& "dev$null")
    
    polyfit (datafile, order=1, weighting="uniform", verbose=no, listdata=yes, >>fitfile)
    
    # ...now plot with sgraph!
    
    # First run through sgraph - plot data points
    
    # clever way of creating a formatted value for the gain and read noise:
    printf ("%.2f\n", gain) | scan (fs1)
    gain_form=fs1
    
    if (rdnoise!=0)
    	{
    	printf ("%.2f\n", rdnoise) | scan (fs1)
    	rdnoise_form=fs1
	}
    else rdnoise_form="N/A"	
    
    # Create labels:
    if (l_flatrc) frctext="(RC)"
    else frctext=""
    if (l_darkrc) drctext="(RC)"
    else drctext=""

    lab_x=l_avg
    lab_y="variance  / 2"
    lab_title=l_first//"-"//l_last//" "//l_flats//frctext//"-"//l_darks//drctext//" ["//l_xrang//","//l_yrang//"]  Gain="//gain_form//" R.n="//rdnoise_form
    
    unlearn sgraph
    
    sgraph (datafile, device="stdgraph", append=no, fill=yes, pointmo=yes, 
    	    marker="circle", szmarke=7, wl=0, wr=0, wb=0, wt=0, logx=no,
	    logy=no, box=yes, grid=no, ticklab=yes, xlabel=lab_x, ylabel=lab_y,  
	    title=lab_title, sysid=yes)
    
    # Second run through sgraph - get fitted data to plot curve
    
    sgraph (fitfile, append=yes, pointmode=no, pattern="dashed")
    
    # Clean up
    
    print ("")
    print ("Cleaning up ...")
    
    if ((l_flatrc) && (l_delrcim))
        {
        imdelete("@"//frcimglist, ver-, >& "dev$null")
        }
    if ((l_darkrc) && (l_delrcim))
        {
        imdelete("@"//drcimglist, ver-, >& "dev$null")
        }
    if (l_flatrc)
    	{
	delete (frcimglist, ver-, >& "dev$null")
    	}
    if (l_darkrc)
    	{
	delete (drcimglist, ver-, >& "dev$null")
    	}
    delete (fimglist, ver-, >& "dev$null")
    delete (dimglist, ver-, >& "dev$null")
    delete (flatimglist, ver-, >& "dev$null")
    delete (darkimglist, ver-, >& "dev$null")
    delete (flatdarkimglist, ver-, >& "dev$null")
    delete (fitfile, ver-, >& "dev$null")
    
    print ("")    


end
