/* initwcs.c -- set approximate WCS from RA/DEC/SCALE/NAXIS keywords */

/*
 * RA,DEC keywords in raw data indicate telescope focal plane center,
 * not chip center.  use the known chip offsets from the telescope pointing
 * position to estimate the image RA,DEC center, and call put_wcs()
 * with the image RA,DEC center to write a rough WCS for the image.
 *
 * initwcs.c has built-in defaults for the chip offsets and rotations for
 * both du Pont and INT data, but in the case that these are not correct
 * the user can create a file called "chipoffsets" containing chip_number,
 * ra_offset, dec_offset, and rotation (degrees), eg for INT:
 * 1 -0.128  0.128  0.0
 * 2 -0.128 -0.128  0.0
 * 3  0.128 -0.128  0.0
 * 4  0.128  0.128  0.0
 *
 * initwcs assumes that RA,DEC keywords refer to telescope pointing
 * position, so do not run initwcs multiple times on the same image.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "irdr.h"

#define DEGRAD 0.017453293
#define NCHIP 4

/* OFFSETS (from telescope field center to chip center in degrees) */

static double LCO_ra_chip_offset [NCHIP] = {-0.054, -0.054, 0.054, 0.054};
static double LCO_dec_chip_offset [NCHIP] = {0.054, -0.054, -0.054, 0.054};
static double INT_ra_chip_offset [NCHIP] = {0.128, 0.128, -0.128, -0.128};
static double INT_dec_chip_offset [NCHIP] = {0.128, -0.128, -0.128, 0.128};

/* ROTATIONS (chip rotations in degrees) */

static double LCO_chip_rotation [NCHIP] = {0.3, -0.4, -0.2, -0.3};
static double INT_chip_rotation [NCHIP] = {0.2, 0.35, 0.85, 0.75};

/* precession routine from WCSTools: */

extern void fk5prec(double epoch0, double epoch1, double *ra, double *dec);

static void read_offsets(void);

int main(int argc, char *argv[])
{
    int i, nx, ny, chip = -1;
    double ra, dec, scale, epoch0 = 0.0;
    char *tel;

    if (argc < 2)
        eprintf("Usage: %s *.fits\n", argv[0]);

    read_offsets();
 
    for (i = 1; i < argc; i++) {
        get_key_int(argv[i], "CHIP", &chip);

        if (chip < 1 || chip > NCHIP)
            eprintf("%s: unexpected chip number: %d\n", chip);

        if (get_key_int(argv[i], "NAXIS1", &nx) < 0)
            eprintf("%s: failed reading NAXIS1\n", argv[0]);

        if (get_key_int(argv[i], "NAXIS2", &ny) < 0)
            eprintf("%s: failed reading NAXIS2\n", argv[0]);

        if (get_wcs(argv[i], &ra, &dec, &scale))
            eprintf("%s: get_wcs failed\n", argv[0]);
  
        if (get_key_double(argv[i], "EQUINOX", &epoch0) < 0 || epoch0 < 1990) {
            fprintf(stderr, "WARN: fail reading EQUINOX, assume 2000.0\n");
            epoch0 = 2000.0;
        }

        if ((tel = get_key_str(argv[i], "TELESCOP")) == NULL)
            eprintf("%s: failed reading TELESCOP keyword\n", argv[0]);

        if (strncmp("INT", tel, 3) == 0) {                            /* INT */
            ra  += INT_ra_chip_offset[chip-1] / cos(DEGRAD * fabs(dec));
            dec += INT_dec_chip_offset[chip-1];
            put_key_double(argv[i], "CHIPROT", INT_chip_rotation[chip-1]);

        } else if (strncmp("LCO", tel, 3) == 0) {                     /* LCO */
            ra  += LCO_ra_chip_offset[chip-1] / cos(DEGRAD * fabs(dec));
            dec += LCO_dec_chip_offset[chip-1];
            put_key_double(argv[i], "CHIPROT", LCO_chip_rotation[chip-1]);

        } else {
            eprintf("FITS key TELESCOP should begin with LCO or INT");
        }

        fk5prec(epoch0, 2000.0, &ra, &dec);

        put_wcs(argv[i], ra, dec, scale, nx, ny);

        put_key_double(argv[i], "EQUINOX", 2000.0);
    }

    return 0;
}

/* read chip offsets and rotations from the file "chipoffsets" if it exists */
static void read_offsets(void)
{
    int chipno;
    float dra, ddec, rot;
    char line[256], *fn = "chipoffsets";
    FILE *fp;

    if ((fp = fopen(fn, "r")) == NULL)
        return;

    fprintf(stderr, "reading chip offsets...\n");

    while (fgets(line, sizeof(line), fp) != NULL) {
        if (sscanf(line, "%d %f %f %f", &chipno, &dra, &ddec, &rot) != 4)
            eprintf("read_offsets: check list format\n");

        if (chipno < 1 || chipno > NCHIP)
            eprintf("read_offsets: bad chip no.: %d\n", chipno);

        /* just overwrite both sets */
        LCO_ra_chip_offset[chipno-1] = INT_ra_chip_offset[chipno-1] = dra;
        LCO_dec_chip_offset[chipno-1] = INT_dec_chip_offset[chipno-1] = ddec;
        LCO_chip_rotation[chipno-1] = INT_chip_rotation[chipno-1] = rot;
    }
}
