/* Program zerosub
**
** Extracts the zeroth read from all other extensions in a multi 
** extension fits file
*/

#include <stdlib.h>
#include "fitswrap.h"

int main(int argc, char* argv[])
{

  int status=0;                     /* Error handle for cfitsio */
  int fpixel=1;                     /* First pixel to read */
  int nread=0;                      /* Number of pixels to read */     
  int nullvall=0;                   /* Markers for bad pixels */
  int anynull=0;                    /* as used by cfitsio */

  int nextend=0;                    /* Total number of extensions in file */
  int nreads=0;                     /* Number of reads in each loop */
  int currndr=0;                    /* Current read no */
  int zexn=0;                       /* Extension zeroth read is in */
  int nimset=NIMSET;                /* Number of images per extension */
  int nsci=0;                       /* Number of science extension in file */  
  float ztime=0;                    /* Time for the zeroth extension */
  float newtime=0;                  /* Actual time for the readout */
  
  typedef float* float_pointer;
  
  float_pointer* data=0;        /* Array for the pixel data */
  float* subdata=0;             /* Array for the subtracted data values*/
  float* timeobs=0;             /* Array for observation times */
  int nx=0;                     /* Assume these are the same for all */
  int ny=0;                     /* reads for now */
  int bitpix=0;
  int i=0;                      /* Loop variables */
  int j=0;

  fitsfile *fptr;               /* fits object as defined by cfitsio */


  if (argc < 2) {
    fprintf(stderr, 
	    "Usage: %s ME-FITS_file\n", 
	    argv[0]);
    exit(1);
  }

  printf("zerosub: Subtracting the zeroth read from all subsequent reads,\nbut leaving the reads before the zeroth read unchanged.\n");


  /* open the file */

  if ( fits_open_file(&fptr, argv[1], READWRITE, &status) )
    
   printerror ( status ) ;

  /* Get info from header */
  
  
  if ( fits_read_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
      
   printerror ( status ) ;

  /*printf("Total number of extensions %i\n",nextend);*/

  if ( fits_read_key(fptr, TINT, "NREADS", &nreads, NULL, &status) )
    
    printerror ( status ) ;

  nsci=nextend / nimset;
  zexn=-100;
  
  /*Loop over the extension and find the one that contains CURR_NDR=nread*/
  for(i=0;i<nsci;i++){
    
    if(fits_movnam_hdu(fptr, ANY_HDU, "SCI", i+1, &status))
      printerror(status);
    if(fits_read_key(fptr, TINT, "CURR_NDR", &currndr, NULL, &status) )
      printerror ( status );
    if((currndr+1)==nreads){
      zexn=i;
    }
  }

  if(zexn==-100){
    fprintf(stderr,"zerosub: Zeroth read is in loop 1, read %i, which is not in this MEF file\n",nreads);
    exit(1);
  }

  /*printf("zerosub: Zeroth read is in extension SCI,%i\n", zexn+1);*/

  /* Allocate memory for the data arrays */
  
  if ( (data=(float_pointer *)malloc(nsci*sizeof(float_pointer))) == NULL ){
    
    printf("zerosub: Memory allocation for data pointers failed!");
    exit(1);
  }
  
  if ( (timeobs=(float *)malloc(nsci*sizeof(float))) == NULL ){
    
    printf("zerosub: Memory allocation for timeobs pointers failed!");
    exit(1);
  }

  /* Loop over the science extensions reading the data in */

  for ( i=0; i<nsci; i++ ){

    /* Get data from the first header which is assumed to be the same 
	   for all images in the set */

    if ( i==0 ){

      /* Move to the first science extension header */

      if ( fits_movnam_hdu(fptr, ANY_HDU, "SCI", 1, &status) )
    
	printerror ( status ) ;

      if ( fits_read_key(fptr, TINT, "BITPIX", &bitpix, NULL, &status) )
	  
	printerror ( status ) ;
	
      if ( fits_read_key(fptr, TINT, "NAXIS1", &nx, NULL, &status) )
	  
	printerror ( status ) ;
      if ( fits_read_key(fptr, TINT, "NAXIS2", &ny, NULL, &status) )
      
	printerror ( status ) ;

      nread=nx*ny;

    } else {

      /* Move to the next header */

      if ( fits_movnam_hdu(fptr, ANY_HDU, "SCI", i+1, &status) )
 
	printerror ( status ) ;
      
    }

    /* Allocate memory for the data array */
    
    if ( (*(data+i)=(float *)malloc(nx*ny*sizeof(float))) == NULL ){
    
      printf("zerosub: Memory allocation for data pointer failed!");
      exit(1);
    }

      
    /* read in the data */
      
    if ( fits_read_key(fptr, TFLOAT, "TIME_OBS", (timeobs+i), NULL, 
			 &status) )
    
      printerror ( status ) ;
  
    if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, *(data+i), 
			 &anynull, &status) )
      
      printerror( status );

    /*printf("timeobs for extension %i is %f, data is %f\n",i+1,
     *(timeobs+i), **(data+i)); */

  }
  
  /* Allocate memory for the temporary zeroth read subtracted data */

  if ( (subdata=(float *)malloc(nx*ny*sizeof(float))) == NULL ){
    
    printf("zerosub: Memory allocation for subdata array failed!");
    exit(1);
  }

  /* subtract off the zeroth read from all images except
     those before the zeroth read and write back to the file */
      
  ztime=*(timeobs+zexn);
  printf("zerosub: Zeroth read, readout after %f milli seconds\n",ztime);
     

  for ( i=0; i<nsci; i++ ){

    /* Calculate the new values */

    newtime=*(timeobs+i)-ztime; 
    
    /* Only change extensions after (and including) the zeroth extension */

    if ( i<=zexn ){

      for ( j=0; j<nx*ny; j++ ){
	  
	*(subdata+j)= *( *(data+i)+j ) - *( *(data+zexn)+j ); 
   
      }
    
    } else {

      for ( j=0; j<nx*ny; j++ ){
	  
	*(subdata+j)= *( *(data+i)+j ); 
   
      }

    }

    if ( i==0 ){

      /* Move to the first extension header */

      if ( fits_movnam_hdu(fptr, ANY_HDU, "SCI", 1, &status) )
       
	printerror ( status ) ;
      
    } else {
      
      /* Move to the next header */
     
      if ( fits_movrel_hdu(fptr, nimset, NULL, &status) )
	
	printerror ( status ) ;
      
    }

    /* Update the fits file */
    
    if ( fits_update_key(fptr, TFLOAT, "SAMPTIME", &newtime, NULL, &status) )
     
      printerror ( status ) ;

    if ( fits_write_img(fptr, TFLOAT, fpixel, nread, subdata, &status) )
	
      printerror( status );
    
  }
  
  if (fits_close_file(fptr, &status) )

      printerror ( status ) ;

  free(timeobs);
  free(subdata);
  
  for( i=0; i<nsci; i++ )
    free(*(data+i));
  
  free(data);
  
  return 0;
}







