/* gain.c
**
** C Program which expects a list of .ima files from do cirpass which 
** need flatfielding
** The program flat fields the data, calculates the mean and standard 
** deviation across the pixels, fits a straight line to this data
** using numerical recipes fitting routines to determine
** the gain and readnoise at each pixel
**
** A. J. Dean 26th Oct, 2000
**
*/


/* The mean and var from this have been tested and do agree with 
** the values from imcombine on the same data !!
** AJD 8th Jan, 2001
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "fitswrap.h"
#include "nrroutines.h"
#include "cpgplot.h"      /* for the graphics output */

/* static funcion definitions */

static char* fgets_nonewline(char *s, int n, FILE *iop);
static int lines_in_file(FILE *iop);

int main(int argc, char* argv[])
{


  FILE *fp;               /* File pointer */
  char* inputfiles;       /* array for file names to stack */
  
  int nread=0;

  int nextend=0;
  int nsci=0;
  int nimset=NIMSET;

  int num=0;  /* Number of images in input list - number to be averaged ... */
  int i=0;
  int j=0;
  int k=0;
  int h=0;
  int level=0;
  int doflat=0;


  long axes[2]={0};
  int bitpix=0;

  char* outputdevice=NULL;
  
  const char* exts[]={"GAIN","READNOISE","CHI2"};
  float* data[3]={NULL}; /* !!! Make the size of this array  !!!
			  * !!! the same as exts             !!! */
  
  typedef float* float_pointer;  
  
  float* image_data=NULL;
  float* flat=NULL;
  double* means=NULL;
  float* mean_errs=NULL;
  double* vars=NULL;
  float* var_errs=NULL;

  float* temp_means=NULL;
  float* temp_vars=NULL;
  float* temp_mean_err=NULL;
  float* temp_var_err=NULL;

  float* gain=NULL;
  float* readnoise=NULL;
  float* chi2image=NULL;

  double* sum=NULL;
  double* sq_sum=NULL;

  double mean=0;
  double mean_sq_sum=0;

  float a, b, siga, sigb, chi2, q;
  int mwt=0;

  float fitx[2]={0};
  float fity[2]={0};
  float xminval=0;
  float xmaxval=0;
  float yminval=0;
  float ymaxval=0;
  char title[100];
  char pm=(char)177; /* The plus / minus symbol */
  
  int xpos=0;
  int ypos=0;

  char mean_name[200]={0};  /* Names for output mean and var images */
  char var_name[200]={0};

# if defined(DEBUG)
  float tempout[1048576]={0};
# endif

  /* Check for appropriate input */
  if ( argc < 7 ){
   
    printf("Usage: %s filelist flatfield|none outputfile xpix ypix int_data_filename [file.ps] \n", 
	   argv[0]);
    exit(1);

  }

  if ( 8 == argc ){
 
    if ( (outputdevice=(char *)malloc((strlen(argv[7])+4)*sizeof(char))) 
	 == NULL ){
      
      printf("plot: memory allocation for output file failed\n");
      exit(1);
      
    }
    
    strcpy(outputdevice,argv[7]);
    strcat(outputdevice,"/PS");
    
  } else if ( 7 == argc ){
    
    if ( (outputdevice=(char *)malloc(9*sizeof(char))) == NULL ){
      
      printf("plot: memory allocation for output file failed\n");
      exit(1);
      
    }
    
    strcpy(outputdevice, "/xwindow");
   
  }

  sscanf(argv[4],"%i",&xpos);
  sscanf(argv[5],"%i",&ypos);

  /*  if ( (xpos > 255) ||  (ypos > 255) ){
    printf("x or y is > 255 which is outside their range\n");
    exit(1);
  }
  */

  /* open the input file and find out how many names there are */
  if ( (fp = fopen(argv[1],"r")) == NULL ){
    printf("Cannot open file containing list of files to be stacked\n");
    exit(1);
  }

  num=lines_in_file(fp);

  /* Allocate memory for the file names */  
  if ( (inputfiles=(char *)malloc(num*FNAMELENGTH*sizeof(char))) == NULL ){
    
    printf("stack: Memory allocation for file name arrays failed!");
    exit(1);
  }

  rewind(fp);
  
  for ( i=0; i<num; i++ ){
    
    if ( fgets_nonewline( (inputfiles+i*FNAMELENGTH),FNAMELENGTH,fp) == NULL ){
      printf("Unable to read the input file list!!\n");
      exit(1);
    }
    
  }
  
  /* Close the input file containing the file list to be read in */
  fclose(fp); 

  /* Read keys from 1st image in list - assume same otherwise
  ** it doens't make sense to be using this data anyway
  */
  readkeys_MEF(inputfiles, axes, &nextend, &bitpix);

  printf("Imsize = %lix%li\n",*(axes),*(axes+1));

  nread=*(axes) * *(axes+1);
  nsci= nextend / nimset;
  printf("nextend = %i nimset = %i nsci = %i\n", nextend, nimset, nsci);

  /* Find out if the data is to be flatfielded first */
  if( strcmp(argv[2],"none") == 0 ){
    doflat=0;
    printf("The data will not be flatfielded\n");
  } else {
    doflat=1;
    printf("The data will be flatfielded with %s\n",argv[2]);
    flat=readfits_float(argv[2], 0);
  }


  
  /* Allocate space for the output gain + readnoise + chi2*/
  if ( (gain=(float *)malloc(sizeof(float)*nread)) == NULL || 
       (readnoise=(float *)malloc(sizeof(float)*nread)) == NULL ||
       (chi2image=(float *)malloc(sizeof(float)*nread)) == NULL ){
    printf("gain: Memory allocation for output arrays failed\n");
    exit(1);
  }
  
  if ( (means=(double *)malloc(sizeof(double)*(nsci-1)*nread )) == NULL || 
       (mean_errs=(float *)malloc(sizeof(float)*(nsci-1)*nread)) == NULL ||
       (vars =(double *)malloc(sizeof(double)*(nsci-1)*nread )) == NULL ||
       (var_errs=(float *)malloc(sizeof(float)*(nsci-1)*nread )) == NULL ||
       (sum=(double *)malloc(sizeof(double)*nread )) == NULL ||
       (sq_sum=(double *)malloc(sizeof(double)*nread )) == NULL ){
    printf("gain: Memory allocation for fitting arrays failed\n");
    exit(1);
  }

  /* Calculate the means and vars at each pixel (miss off zero level)*/  
  for(level=0; level<nsci-1; level++){ /* Loop over intensity levels */
    printf("Calculating mean / var for science extension %i\n",
	   nsci-level-1);
    
    /* Clear sums */
    for(h=0; h<nread; h++){
      *(sum+h)=0;
      *(sq_sum+h)=0;
    }
      
    for(k=0; k<num; k++){ /* Looping over images in input list */
     
      /* Open image and reorder to count up */
      image_data=readfits_float_extver( (inputfiles+k*FNAMELENGTH), "SCI", 
				  nsci-level-1);

      /* Flat field if necessary */
      if(doflat){
	for(h=0; h<nread; h++){
	  *(image_data+h) = ( *(flat+h) == 0 ? *(image_data+h) : (double)*(image_data+h) / (double)*(flat+h) );
	}
      }
	
      for(j=0; j<axes[1]; j++){
	
	for(i=0; i<axes[0]; i++){
	  
	  /* Accumulate sums */
	  *(sum+i+j*axes[0]) += (double)*(image_data+i+j*axes[0]);
	  *(sq_sum+i+j*axes[0]) += (double)( *(image_data+i+j*axes[0]) * 
	    (double)*(image_data+i+j*axes[0]) );

# if defined(DEBUG)
	  if(i==686 && j==452)
	    printf("Data for pixel 687 453 is %f sq is %f sq_sum is %f\n",
		   *(image_data+i+j*axes[0]), 
		   (*(image_data+i+j*axes[0]) * *(image_data+i+j*axes[0]) ),
		   *(sq_sum+i+j*axes[0]));
# endif

	}

      }
	
      free(image_data);
    
    }

# if defined(DEBUG)
    printf("Pixel 687 453 sum is %f, sq_sum is %f\n", *(sum+686+452*axes[0]),
	   *(sq_sum+686+452*axes[0]));
# endif

    for(j=0; j<axes[1]; j++){
	
      for(i=0; i<axes[0]; i++){
	
	mean=*(sum+i+j*axes[0]) / (double)num; 
	*(means+level*nread+i+j*axes[0]) = mean; 
	
	mean_sq_sum = *(sq_sum+i+j*axes[0]) / (double)num;
	
	*(vars+level*nread+i+j*axes[0]) = ((double)num / (double)(num-1)) * ( (double)mean_sq_sum - (double)(mean * mean) );

# if defined(DEBUG)
	if( i==686 && j==452 )
	  printf("mean for pixel 685 453 is %f stddev is %f\n", mean, 
		 sqrt(*(vars+level*nread+i+j*axes[0])));
# endif
	
      }

    }
#if defined(DEBUG)    
    if(level==5)
      for(i=0; i<1024*1024;i++)
	*(tempout+i)=*(vars+level*nread+i); 
# endif
    
  }
  
  /* Numerical recipes + fitsio expecting floats from here */

  /* Allocate memory for temporary fitting arrays - nrroutines needs these
   * to run from 1, so need 1 more than number of points which is nsci-1 */
   if ( (temp_means=(float *)malloc(sizeof(float)*(nsci) )) == NULL ||  
	(temp_vars =(float *)malloc(sizeof(float)*(nsci) )) == NULL ||
	(temp_mean_err=(float *)malloc(sizeof(float)*(nsci) )) == NULL ||
	(temp_var_err=(float *)malloc(sizeof(float)*(nsci) )) == NULL ){
     printf("gain: Memory allocation for fitting arrays failed\n");
     exit(1);
  }

  /* Open graphics device. */      
   /*    if(cpgbeg(0, outputdevice, -3, 3) != 1)
	return 1; 
   */
  
   if(cpgbeg(0, outputdevice, 1, 1) != 1)
     return 1; 
   
  printf("gain: Fitting the slope to find the gain and readnoise ");
  printf("for each pixel\n");

  for(i=0; i<nread; i++){
  
    for(j=0; j<nsci-1; j++){ /* numerical recipes from 1 not 0 and have 1
			      * less data point */  

      *(temp_means+j+1)=(float)*( means + j*nread + i);
      *(temp_vars+j+1) =(float)*( vars + j*nread + i);
      
      *(temp_mean_err+j+1) = ( sqrt(*(temp_vars+j+1) / (float)num) );

      *(temp_var_err+j+1)= 2 * ( 1 / sqrt( 2*(float)num - 2 ) ) *
	                   ( *(temp_vars+j+1) );

# if defined(DEBUG)			   
      if (i==463534)
	printf("mean err = %f, var err = %f\n", *(temp_mean_err+j+1),
	       *(temp_var_err+j+1));
# endif
    
    }
  
    /* Do the straight line fitting for each pixel and put into output file 
     * arrays (this is the Numerical Recipes in C fitting code)
     */
    mwt=1; /* Know the standard deviations */
    
    fit(temp_means, temp_vars, nsci-1, temp_mean_err, mwt, &a, &b, 
	&siga, &sigb, &chi2, &q); 
    *(gain+i) = ( b==0.0 ? 0.0 : 1/(b) );
    *(readnoise+i) = ( b==0.0 ? 0.0 : (float)sqrt((1/(b))*(a)) );
    *(chi2image+i) = chi2;
     
    /* if( i == (xpos+1024*ypos) || 
	i == ((xpos+1024*ypos)+256) || 
	i == ((xpos+1024*ypos)+512) ||
	i == ((xpos+1024*ypos)+1024*256) || 
	i == ((xpos+1024*ypos)+1024*256+256) || 
	i == ((xpos+1024*ypos)+1024*256+512) ||
	i == ((xpos+1024*ypos)+1024*512) || 
	i == ((xpos+1024*ypos)+1024*512+256) || 
	i == ((xpos+1024*ypos)+1024*512+512) ){
    */
  
    if( i == ((xpos-1)+1024*(ypos-1)) ){

# if defined(DEBUG)	  
      /* print the output on screen */
      for(k=1; k<=nsci-1; k++){
	printf("mean = %f var = %f chi2 = %f\n",*(temp_means+k),
	       *(temp_vars+k),chi2);
      }      
# endif

      /* Make up a straight line graph of the fit */
      xminval=*(temp_means+1)*0.85;
      xmaxval=*(temp_means+nsci-1)*1.15;
      yminval=*(temp_vars+1)*0.85;
      ymaxval=*(temp_vars+nsci-1)*1.15;
      
      fitx[0]=xminval;
      fity[0]=a+b*fitx[0];
      fitx[1]=xmaxval;
      fity[1]=a+b*fitx[1];      
      
      /* Plot the output on screen */
      /* Define coordinate range of graph and draw axes. */
      cpgenv(xminval, xmaxval, yminval, ymaxval, 0, 0);
    
      sprintf(title,"Gain = %f %c %f e/DN, Read noise = %f", 1/(b), pm, 
	      (1/(b))*(sigb)/(b), sqrt((1/(b))*(a)));
      
      /* Label the axes */
      cpglab("Mean", "Variance", title); 
      
      /*  Plot the line graph. */
      cpgsci(5); /* Blue lines */ 
      cpgline(2, fitx, fity);
      
      cpgsci(1); /* White points */
      cpgpt(nsci-1,(temp_means+1),(temp_vars+1),5);
       
    }
  
  }
 
  /* Close the graphics device */
  cpgend();

  /* Write out the data */
  data[0]=gain;
  data[1]=readnoise;
  data[2]=chi2image;
  
  write_ME_FITS(argv[3], exts, data, 3, axes);

  /* Write out the mean and var data */

  for(i=0; i<nsci-1; i++){

    sprintf(mean_name,"%s_mean_%i.fits",argv[6],i+1);
    sprintf(var_name,"%s_var_%i.fits",argv[6],i+1);   
     
    writefits_double_basic(mean_name, axes, (means + i*nread));
    writefits_double_basic(var_name, axes,  (vars + i*nread));
  }


# if defined(DEBUG)
  writefits_float_basic("outfile.fits", axes, tempout); 
# endif  
 
  /* Clean up */
  free(outputdevice);
  free(inputfiles);
  if(doflat)
    free(flat);

  free(means);
  free(mean_errs);
  free(vars);
  free(var_errs);

  
  free(gain);
  free(readnoise);
  free(chi2image);

  free(sum);
  free(sq_sum);

  free(temp_means);
  free(temp_vars);
  free(temp_mean_err);
  free(temp_var_err);

  return 0;

}

/* This is the same code as the standard library function fgets, 
** except for the -- in the second to last line,
** which makes it overwrite the newline with a trailing 0.
**/

char *fgets_nonewline(char *s, int n, FILE *iop)
{

  register int c=0;
  register char *cs;

  cs = s;
  while (--n > 0 && ( c = getc(iop)) !=EOF)
    if ((*cs++ = c) == '\n')
      break;
  *(--cs) = '\0';
  return ( c == EOF && cs == s ) ? NULL : s;

}

int lines_in_file(FILE *iop)
{
  register int c=0;
  int lines=0;
  
  while ( ( c = getc(iop)) !=EOF) 
    if ( c == '\n')
      lines++;

  return lines;
}




