/* plotraw.c -- plots the pixel as it integrates from
** a stacked cirpass data cube,
**
** Andrew J. Dean. 4th Aug, 2000.   
**
** Version 3 of plotpix which will work on a raw stacked datacube (no zerosub)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fitswrap.h"     /* for printerror and cfitsio */
#include "cpgplot.h"      /* for the graphics output */

/* Start of main code */

int main(int argc, char* argv[])
{

  /* Define parameters for fitsio */
  int status=0;                  /* Error handle for cfitsio */
  fitsfile *fptr;                /* fits input object as defined by cfitsio */
 
  int fpixel=1;                  /* First pixel to read */
  int ntoread=0;                   /* Number of pixels to read */     
  int nullvall=0;                /* Markers for bad pixels */
  int anynull=0; 
  char comment[FLEN_COMMENT];    /* Fits header comment */
  
  /* Local variables */
  float* data;                   /* Big array for whole image data */
  float* sci;                    /* Fitting arrays for cosmic ray routines */
  float* time;                   /* and data quality */
	
  int xpix=0;                    /* xpixel to examine */
  int ypix=0;                    /* ypixel to examine */
  int boxsize=0;                 /* size of pixel box to examine */
  int pixel=0;                   /* Pixel counter */

  int nx=0;                      /* Dimensions of image */
  int ny=0;
  int nloops=0;                  /* Number of loops in exposure */
  int nreads=0;                  /* Number of reads in each loop */
  int nsci=0;                    /* Number of science extensions in file */
  int nextend=0;                 /* Number of extensions in file */
  int nimset=NIMSET;             /* Number of extensions per image */
  int nbytes;                    /* bytes needed for data */
  int i=0;                       /* Loop variables */
  int j=0;
  int k=0;
  short nsamp=0;                 /* counter */     
  int numsamps=0;                /* Number of integrating pixel points */    
  char title[40];
  char* outputdevice=NULL;       /* output device */

  /* Check for appropriate input */
  if (argc < 5) {
    fprintf(stderr, 
            "Usage: %s scidata.fits xpix ypix boxsize [output.ps]\n", 
            argv[0]);
    exit(1);
  }
 
  sscanf(argv[2],"%i",&xpix);
  sscanf(argv[3],"%i",&ypix);   
  sscanf(argv[4],"%i",&boxsize);
   
  if ( argc==6 ){

    if ( (outputdevice=(char *)malloc((strlen(argv[5])+4)*sizeof(char))) 
	 == NULL ){
      
      printf("plot: memory allocation for output file failed\n");
      exit(1);

    }

    strcpy(outputdevice,argv[5]);
    strcat(outputdevice,"/PS");

  } else if ( argc == 5 ){

    if ( (outputdevice=(char *)malloc(9*sizeof(char))) == NULL ){
  
      printf("plot: memory allocation for output file failed\n");
      exit(1);

    }

    strcpy(outputdevice, "/xwindow");

  }

  /*  printf("plot: plotting all loops\n"); */

  /* Open the input file */
  if ( fits_open_file(&fptr, argv[1], READONLY, &status) ){
    printf("Failed to open file\n");
    printerror ( status ) ;
  }

  
  /* Can probably delete nloops + possibly nreads !!!!!!! */
  /* Read dimensions of image, nx and ny, and find number of loops and reads */
  if ( fits_read_key(fptr, TINT, "NLOOPS", &nloops, &comment[0], &status) )
      
    printerror ( status ) ; 

  if ( fits_read_key(fptr, TINT, "NREADS", &nreads, &comment[0], &status) )
      
    printerror ( status ) ;
  
  if ( fits_read_key(fptr, TINT, "NEXTEND", &nextend, &comment[0], &status) )
      
    printerror ( status ) ;

  /* Move to the first science extension to get the image dimensions */
  fits_movnam_hdu(fptr, ANY_HDU, "SCI", 1, &status);
 
  if ( fits_read_key(fptr, TINT, "NAXIS1", &nx, &comment[0], &status) )
      
    printerror ( status ) ; 

  if ( fits_read_key(fptr, TINT, "NAXIS2", &ny, &comment[0], &status) )
      
    printerror ( status ) ; 
  
  /* printf("nx is %i, ny is %i, nloops is %i, nreads is %i \n", nx, ny, 
            nloops, nreads); */

  nsci=nextend/nimset; /* Number of science extensions in image */
  
  if ( nsci <=1 ){
    printf( "plotpix: Not enough data to make a sensible plot!\n" );
    exit(1);
  }

  ntoread=nx*ny; /* Number of pixels to read in */  
  /*  numsamps=(int)((nsci-nreads)/nreads);*/ /* Number useful science extensions */
  numsamps=nsci;
  nsamp=numsamps-1; /* As above but for zero based indexing */
  nbytes=nx*ny*numsamps*sizeof(float); /* Memory needed for the data array */

  /* Allocate memory for local fitting arrays */
  if ( (sci=(float *)malloc(numsamps*boxsize*boxsize*sizeof(float))) == NULL ){
    
    printf("cosmicray: Memory allocation for sci fitting arrays failed!");
    exit(1);
  }

  if ( (time=(float *)malloc(numsamps*sizeof(float))) == NULL ){

    printf("cosmicray: Memory allocation for time fitting arrays failed!");
    exit(1);
  }
  
  /* Use lots of memory and read the data in! */
  if ( (data=(float *)malloc(nbytes)) == NULL ){

   printf("cosmicray: Memory allocation for data failed!");
   exit(1);
  }
  
  fpixel=1;

  /* Loop over the science extensions in the input data and
  ** place into float data array
  */
  for(i=0; i<numsamps; i++){

    /* printf("i is %i, reading header %i\n",i,1+i*nreads); */

    /* Read in the science data, this is in datanumbers */
    fits_movnam_hdu(fptr, ANY_HDU, "sci", 1+i, &status);  /* was 1+i*nreads */
    
    if ( fits_read_img(fptr, TFLOAT, fpixel, ntoread, &nullvall, 
		       &data[i*ntoread], &anynull, &status) )

    printerror( status );
    
    /* Read in the times of the science data, use nsamp to reorder
    ** to count up 
    */
    if ( fits_read_key(fptr, TFLOAT, "TIME_OBS", &time[nsamp], 
		       &comment[0], &status) )
      
     printerror ( status ) ; 
    
    /* PixCel outputs the time in milliseconds so convert to seconds */
    
    time[nsamp]/=1000;
       
    nsamp--;
    
  }

  /* Open graphics device. */      
  if(cpgbeg(0, outputdevice, -boxsize, boxsize) != 1)
    return 1; 


  /* Loop over image array, assembling the data, and plotting
  **
  */
  pixel=0;
 
  for( i=ypix-1; i<ypix-1+boxsize; i++ ){
    
    for ( j=xpix-1; j<xpix-1+boxsize; j++ ){
  
      /* create fitting arrays for this pixel reordering to integrate up*/
      nsamp=0;

      /* Fill the fitting arrays with all the data, reordering to integrate up
      */
      /* for ( k=((nsci-nreads)/nreads)-1; k>=0; k-- ){ */
      
      for ( k=nsci-1; k>=0; k-- ){
	sci[nsamp+pixel*numsamps]=data[k*nx*ny + i*nx + j]; 
	                                          
	if ( i==ypix-1 && j==xpix-1 ) { 
	  printf("pixel %i,%i in science extension %i has time %f and sci %f.\n",j+1,i+1,k+1,time[nsamp],sci[nsamp+pixel*numsamps]); 
	} /* was k*nreads+1 */
	nsamp++;
      }

      /* Plot the output on screen */
      /* Define coordinate range of graph and draw axes. */
      cpgenv(time[0], time[nsamp-1], sci[pixel*numsamps], 
	     sci[nsamp-1+pixel*numsamps], 0, 0);
      
      /* Label the axes */
      sprintf(title,"Pixel x=%i y=%i integration with time",j+1,i+1);
      cpglab("Time (s)", "Counts", title);
     
      /*  Plot the line graph. */
      cpgsci(5); /* Blue lines */
      cpgline(nsamp, time, (sci+pixel*numsamps));
      cpgsci(1); /* White axes */

      pixel++;

    }

  }
      
  /* Close the graphics device */
  cpgend();

 
  /* Free memory allocated locally */
  free (sci);
  free (time);
    
  if ( fits_close_file(fptr, &status) )
    printerror( status );
   
  free (data);
  
  /* successful return */

  return 0;

}






