/* stack_average.c 
** 
** Adapted from stack.c to average the n-1 last reads of a loop
**
** A.J. Dean 3rd Aug, 2002
**
**
** c program to create multi-extension FITS files from input FITS files
** The first file in the list is placed into the last extension, 
** the second file in the next-to-last extension etc.
**
** Each file in the list should have the same header keyworks.
** 
** Written by AJ Dean. 19th June, 2000. Based on stack_read.cl by 
** RAJ 10 March, 2000.
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include "cirpass.h"
#include "fitswrap.h"
#include "unfit_code.h"
#include "libaddfibreinfo.h"

/* static funcion definitions */
static char* fgets_nonewline(char *s, int n, FILE *iop);
static float* get_keys_and_data(const char* fn,  char* extkeys[], 
				float* extkeyval, int numkeys);

/* Main code */
int main(int argc, char* argv[])
{

  /* Local Variables */
  FILE *fp;               /* File pointer */
  char* inputfiles;       /* array for file names to stack */
  
  int num=0;              /* number of files to stack */
  int i=0;                /* loop variables */
  int j=0;
  int k=0;
  int ver=0;              /* Extension version */
  int filenum=0;
 
  typedef float* pointer_to_float;   /* place to store the pixcel values */
  pointer_to_float* scidata;         /* in the input images */

                          /* keywords to delete from primary header */
  char* delkeys[]={"NAXIS1", "NAXIS2", "LOOP","READ","CURR_NDR","TIME_OBS",
		   "SAMPTIME"};
  
  char* extkeys[]={"LOOP", "READ", "CURR_NDR", "TIME_OBS", "SAMPTIME"};

  char* infokeys[]={"NLOOPS","NREADS"};

  int* infodata=NULL;
  int NLOOPS=0;
  int NREADS=0;

  float* average=NULL;
  float* tempstorage=NULL;

  pointer_to_float* extkeyval;

  int numdelkeys=0;
  int numextkeys=0;
  int numinfokeys=0;

  /* Variables for cfitsio */
  int status=0;                   /* Error handle for cfitsio */
  fitsfile *fptr;                 /* fits object as defined by cfitsio */
  fitsfile *fptr_first;             
  int  fpixel, nelements;
  int bitpix   =  0;              /* bits per pixel in output ME FITS file */
  int shortimg =  SHORT_IMG;      /* bits per pixel for short image (for dq) */
  long naxis=2;                   /* 2D images */ 
  long axes[2];                   /* contains image dimensions */
  int nextend=0;         /* Number of extensions in stacked ME FITS files */
  int updatenextend=0;   /* Has a table extension been added ? */
  int trueextend=1;      /* Used for cfitsio */
  char fitscard[81];     /* Header card */
  
  /* Debug variables */

# if defined(DEBUG_STACK)
 int j=0;
# endif /* DEBUG_STACK */

  /* check for appropriate input */
  if ( argc < 5 ) {
    fprintf(stderr, 
	    "Usage: %s file:filelist numinlist outfile configfile\n", 
	    argv[0]);
    exit(1);
  }

  /* convert the number to type int so can use it in loops */
  sscanf(argv[2],"%i",&num);
 
  numdelkeys=(sizeof(delkeys)/sizeof(delkeys[0]));
  numextkeys=(sizeof(extkeys)/sizeof(extkeys[0]));
  numinfokeys=(sizeof(infokeys)/sizeof(infokeys[0]));

  /* Allocate memory for the file names */  
  if ( (inputfiles=(char *)malloc(num*FNAMELENGTH*sizeof(char))) == NULL ){
    
    printf("stack: Memory allocation for file name arrays failed!");
    exit(1);
  }

  /* Allocate memory for the info keys */  
  if ( (infodata=(int *)malloc(numinfokeys*sizeof(int))) == NULL ){
    
    printf("stack: Memory allocation for info keys failed!");
    exit(1);
  }

  /* open the input file and read in the files to stack */
  if ( (fp = fopen(argv[1],"r")) == NULL ){
    printf("Cannot open file containing list of files to be stacked\n");
    exit(1);
  }
    
  for ( i=0; i<num; i++ ){
    
    if ( fgets_nonewline(&inputfiles[i*FNAMELENGTH],FNAMELENGTH,fp) == NULL ){
      printf("Unable to read the input file list!!\n");
      exit(1);
    }

# if defined(DEBUG_STACK)       
    printf("stack:Opening file %s\n",&inputfiles[i*FNAMELENGTH]);
# endif /* DEBUG_STACK */
    
  } 

  /* Close the input file containing the file list to be read in */
  fclose(fp); 
  

  /* Find out nloops and nreads as this will determine how much stacking
   * will be needed, get this from first image
   */

  read_int_keys(&inputfiles[0], infokeys, infodata, numinfokeys);

  NLOOPS=*(infodata);
  NREADS=*(infodata+1);

  /* Allocate memory for the image data
  ** contain the addresses of the actual data read in using fitswrap
  */
  if ( (scidata=(pointer_to_float *)malloc(num*sizeof(pointer_to_float))) == NULL ){
    
    printf("stack: Memory allocation for data pointers failed!");
    exit(1);
  }
  
  num=NLOOPS+1;
  
  /* Find out the image sizes  */
  if( readkeys(&inputfiles[0],axes, &bitpix) ){
    printf("stack: Failed to read header keywords from %s\n",
	   inputfiles);
    exit(1);
  }
    
  /* Reset bitpix to float image */
  bitpix=FLOAT_IMG;

# if defined(DEBUG_STACK)     
  printf("Image %s, nx is %ld, ny is %ld\n",&inputfiles[i*FNAMELENGTH],
	 axes[0],axes[1]);
# endif /* DEBUG_STACK */

  /* Allocate memory for the output sci data*/
  for(i=0; i<num; i++){
    if ( (*(scidata+i)=(float *)malloc(axes[0]*axes[1]*sizeof(float))) == NULL ){
      
      printf("stack: Memory allocation for data pointers failed!");
      exit(1);
    }
  }

  /* Allocate memory for the average data */
  if ( (average=(float *)malloc(axes[0]*axes[1]*sizeof(float))) == NULL ){
    
    printf("stack: Memory allocation for data pointers failed!");
    exit(1);
  }

  /* Allocate memory for the keywords to add to the extensions */
  if ( (extkeyval=(pointer_to_float *)malloc(num*sizeof(pointer_to_float))) == NULL ){
    
    printf("stack: Memory allocation for extension keywords pointers failed!");
    exit(1);
  }
  
  for ( i=0; i<num; i++ ){

    if ( (*(extkeyval+i)=(float *)malloc(numextkeys*sizeof(float))) == NULL ){
    
      printf("stack: Memory allocation for extension keywords failed!");
      exit(1);
    }
  
  }

  for(k=0; k<=NLOOPS; k++){

    for(j=0; j<axes[0]*axes[1]; j++){
      *(average+j)=0.0;
    }

    printf("stack_average: averaging loop %i\n",k);

    for(i=0; i<NREADS-1; i++){

      filenum=(k*NREADS)+1+i;

# if defined(DEBUG_STACK)     
      printf("File number to average %i, name=%s\n",filenum,&inputfiles[filenum*FNAMELENGTH]);
# endif /* DEBUG_STACK */

      tempstorage=get_keys_and_data(&inputfiles[((k*NREADS)+1+i)*FNAMELENGTH],
				    extkeys, *(extkeyval+k), numextkeys); 
      /* unfit the data ********!!!!!!!!!!!!!!*********** remove eventually */
      /*
	# if defined(DEBUG_STACK) 
	printf("stack:Unfitting the data\n");
        # endif 
    
	if( unfit( *(scidata+i), PIXEL_SHIFT) ){
	printf("stack: unfit failed!!");
	exit(1);
	}
      */
      
      for(j=0; j<axes[0]*axes[1]; j++){
	*(average+j)+=*(tempstorage+j);
      }
      
      free(tempstorage);
      
    }
 
    for(j=0; j<axes[0]*axes[1]; j++){
      *(*(scidata+k)+j) = *(average+j)/((float)NREADS-1.0);
    }
    
  }
        
  /* Create the multi-extension fits file */
  if ( fits_create_file(&fptr, argv[3], &status) ) 
    printerror( status );      /* call printerror if error occurs */


  /* Create the primary header */
  
  /* by copying over the header from the first file in the list*/

  if ( fits_open_file(&fptr_first, inputfiles, READONLY, &status) )
    printerror( status );
  
  if ( fits_copy_hdu(fptr_first, fptr, 0, &status) )
    printerror( status );

  if ( fits_close_file(fptr_first, &status) )
    printerror( status );
  
  /* and then sorting  out the keywords */
  
  i=0;              /* used for no data in primary header */

  if ( fits_update_key(fptr, TINT, "NAXIS", &i, NULL, &status) )
    printerror( status );    

  /* Pixel doesn't output a EXTEND keyword so need to add it, however, fits
   * standard compliant files will have this keyword so it should be updated
   */
  fits_read_card(fptr, "EXTEND", fitscard, &status);

  if( status == KEY_NO_EXIST ){
 
    status=0;
    if ( fits_insert_key_log(fptr,"EXTEND", TRUE, 
			     "File may contain extensions", &status) )
      printerror( status );
  
  } else {
    
    status=0;
    trueextend=TRUE;

    if ( fits_update_key(fptr,TLOGICAL,"EXTEND", &trueextend, 
		       "File may contain extensions", &status))
      printerror( status );

  }
  
  nextend=num*NIMSET;
  
  if ( fits_insert_key_lng(fptr, "NEXTEND", nextend, NULL, &status) )
    printerror( status );
  
  /* delete unwanted keywords */
  for( i=0; i<numdelkeys; i++ ){  

  if ( fits_delete_key(fptr, delkeys[i], &status) )
    printerror( status );

  }
  
  /* Add in the fibre info - need to close the file first to avoid having
   * multiple copies open
   */

  if ( fits_close_file(fptr, &status) )
    printerror( status );

  if ( !addfibreinfo(argv[3], argv[4]) ){
  
    /* need to update nextend */
    updatenextend=1;
    nextend+=1;

  } else {

    printf("stack: libaddfibreinfo failed.\nstack: No fibre configuration data will be written!!\n");

  }

  if ( fits_open_file(&fptr, argv[3], READWRITE, &status) )
    printerror( status );

  
  if(updatenextend){

    if ( fits_update_key(fptr, TLONG, "NEXTEND", &nextend, NULL, &status) )
      printerror( status );  
  }
  
  /* Back to normal stacking */

  /* Loop over and add in all extensions and basic headers */
  for ( i=num-1; i>=0; i--){
   
 
  /* create the new extension with the correct keywords  */
  /* these will mainly be the keywords for the last read of the loop */

  /* science images */
    if ( fits_create_img(fptr,  bitpix, naxis, axes, &status) )
      printerror( status );          
    
    if ( fits_update_key(fptr, TSTRING, "EXTNAME", "SCI", NULL, &status) )
      printerror( status );      
    
    ver=num-i;
    
    if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
      printerror( status );  

    for ( k=0; k<numextkeys; k++ ){

      if ( fits_update_key(fptr, TFLOAT, extkeys[k], (*(extkeyval+i)+k), 
			   NULL, &status) )
      printerror( status ); 

    }
    
    /* write the array of floats to the FITS file */
    fpixel = 1;                           /* first pixel to write      */
    nelements = axes[0] * axes[1];      /* number of pixels to write */
    
    if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, *(scidata+i), 
			&status) )
      printerror( status );            
    
    /* Error arrays */
    if ( fits_create_img(fptr,  bitpix, naxis, axes, &status) )
      printerror( status );          
    
    if ( fits_update_key(fptr, TSTRING, "EXTNAME", "VAR", NULL, &status) )
      printerror( status );      

    if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
      printerror( status );  
    
    /* Data quality arrays */
    if ( fits_create_img(fptr,  shortimg, naxis, axes, &status) )
      printerror( status );          
    
    if ( fits_update_key(fptr, TSTRING, "EXTNAME", "DQ", NULL, &status) )
      printerror( status );      

    if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
      printerror( status ); 
    
    /* Time arrays */
    if ( fits_create_img(fptr,  bitpix, naxis, axes, &status) )
      printerror( status );          
    
    if ( fits_update_key(fptr, TSTRING, "EXTNAME", "TIME", NULL, &status) )
      printerror( status );      

    if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
      printerror( status ); 


  }

  if ( fits_close_file(fptr, &status) )
    printerror( status );
    
  /* Free memory allocated locally */
  for ( i=0; i<num; i++ ){
    free(*(scidata+i));
    free(*(extkeyval+i));
  }

  free(infodata);
  free(average);
  free(inputfiles);
  free(scidata);
  free(extkeyval);

  /* succesfull return */
  return 0;

}

/* Static functions */

/* This is the same code as the standard library function fgets, 
** except for the -- in the second to last line,
** which makes it overwrite the newline with a trailing 0.
**/

char *fgets_nonewline(char *s, int n, FILE *iop)
{

  register int c=0;
  register char *cs;

  cs = s;
  while (--n > 0 && ( c = getc(iop)) !=EOF)
    if ((*cs++ = c) == '\n')
      break;
  *(--cs) = '\0';
  return ( c == EOF && cs == s ) ? NULL : s;

}

float* get_keys_and_data(const char* fn, char* extkeys[], float* extkeyval,
			 int numkeys)
{  

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  long nx, ny;                         /* Dimensions of image */
  int nbytes;                          /* Size of array required for data */
    
  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
  int i=0;
  float* data;

 if ( fits_open_file(&fptr, fn, READONLY, &status) )
    
   printerror ( status ) ;

 /* Get info from header */
 if ( fits_read_key(fptr, TLONG, "NAXIS1", &nx, NULL, &status) ){
   printf("Failed to read naxis1 from header\n");
   printerror ( status ) ; 
 }

 if ( fits_read_key(fptr, TLONG, "NAXIS2", &ny, NULL, &status) ){
   printf("Failed to read naxis2 from header\n");
   printerror ( status ) ;
 }
 
 for( i=0; i<numkeys; i++ ){
 
   
   if ( fits_read_key(fptr, TFLOAT, extkeys[i], (extkeyval+i), NULL, &status) ){
     printf("Failed to read %s from header\n",extkeys[i]);
     printerror ( status ) ;
   }
   
# if defined(DEBUG_STACK)     
   printf("key %s has value %f\n",extkeys[i],*(extkeyval+i));
# endif /* DEBUG_STACK */
     
 }
 
 /* Find out size of array required and allocate necessary memory */
 nbytes=nx*ny*sizeof(float);
 nread=nx*ny;

 /* Read in the data */
 if ( (data=(float *)malloc(nbytes)) == NULL ){

   printf("stack,getkeys_and_data: Memory allocation for data failed!");
   exit(1);
 }

 if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, &status) )
   printerror( status );
 

 /* Close the file freeing memory allocated for fitsfile fptr */
 if ( fits_close_file(fptr, &status) )
      printerror ( status );

    return data;

}







