/* peaktrace.c 
**
** Follows a peak across an image and returns its trace
**
** A. J. Dean. 12th Nov, 2001
**
*/

#include <stdio.h>
#include <stdlib.h>
#include "fitswrap.h"
#include "mathutils.h"

#define LOST -999

int main(int argc, char* argv[]){

  int xinitpos=0; /* Place to start tracing from */
  int yinitpos=0; 
  int scanwidth=0; /* How far either side of initial peak to go when tracing */
  int width=0;     /* Total width either side of peak */
  int ystart=0;    /* Area around trace in which to locate the peak */
  float background=0;/* Background in image */
  float sigma=0; /* Sigma above background which indicates we have a trace */
  float datamin=0; /* Used when checking there is a peak in the trace */
  float datamax=0;

  int recovered=1; /* Bool type marker used if traces are lost */

  
  float* data=NULL; /* For the fits data */
  long axes[2]={0}; 

  double* outdata=NULL;
  float* tempval=NULL;
  float* temppos=NULL;

  int i=0; /* Loop variables */
  int j=0;

  FILE* outfile;

  if (argc < 8) {
    
    fprintf(stderr, 
	    "Usage: %s image_file outfile xinitpos yinitpos scan(half)width sigma background_noise\n", 
           argv[0]);
    exit(1);
  
  } 
 
  sscanf(argv[3],"%i",&xinitpos);
  sscanf(argv[4],"%i",&yinitpos); 
  sscanf(argv[5],"%i",&scanwidth);
  sscanf(argv[6],"%f",&sigma);
  sscanf(argv[7],"%f",&background);
  
  /* printf("%s %s %i %i %i %f %f\n",argv[1],argv[2],xinitpos,yinitpos,scanwidth,sigma,background);*/
 
  width=2*scanwidth+1;

  /* Open image file */
  data=readfits_float_clever(argv[1], axes);

  /* Allocate memory for the outdata */
  if ( (outdata=(double *)malloc(*(axes)*sizeof(double))) == NULL ||
       (tempval=(float *)malloc(width*sizeof(float))) == NULL ||
       (temppos=(float *)malloc(width*sizeof(float))) == NULL ){
    
    printf("peaktrace: Memory allocation for output data arrays failed\n");
    exit(1);
  }
  
  /* Check for sensible input */
  if( (xinitpos>*(axes)+1) || xinitpos<1 || (yinitpos>*(axes+1)+1) || 
      yinitpos<1 ){
    printf("input values are outside range of data !!\n");
    exit(1);
  }


  /* Follow trace up first. x and y following 1 based index, i,j and data 
   * following 0 based index */
  ystart=yinitpos-scanwidth; /* Change this by following new centred values */

  /* Check trace won't go off the edge of the detector
   * Assume trace may wander by 5 pixels 
   * Code me! Just stop at edge of detector! */

  if( (yinitpos+scanwidth+5>*(axes+1)) || (ystart-5<0) ){ 
   fprintf(stderr,"WARNING: yinitpos is very close to edge of detector\n");
   fprintf(stderr,"WARNING: this could result in a core dump!\n");
  }

  for(i=xinitpos-1; i<*(axes); i++){

    for(j=0; j<width; j++){ /* Looping in the y direction */
      *(temppos+j)=(float)(ystart+j); 
      *(tempval+j)=*(data+i+(ystart+j-1)**(axes));
      /* printf("x=%i y=%i %f %f\n", i+1, ystart+j, *(temppos+j),*(tempval+j)); */
    }
    
    /* Check trace levels are acceptable */
    minmax(tempval,width,&datamin,&datamax);
    if( (datamax-datamin)>sigma*background ){

      if(recovered==0){
	printf("peaktrace: recovered trace at x=%i y=%i !! \n",i+1,ystart);
      }
      *(outdata+i)=centroid(tempval, temppos, width);
      ystart=(int)(*(outdata+i)+0.5)-scanwidth;
      recovered=1;
      
    } else {

      if(recovered){
	printf("peaktrace: lost trace at x=%i y=%i !! \n",i+1,ystart);
      }
      *(outdata+i)=LOST;
      recovered=0;
      /* Don't change ystart */
     
    }

  }

  /* Follow trace down. */
  ystart=yinitpos-scanwidth; /* Change this by following new centred values */
  recovered=1;
  for(i=xinitpos-2; i>=0; i--){
 
    for(j=0; j<width; j++){ /* Looping in the y direction */
      *(temppos+j)=(float)(ystart+j); 
      *(tempval+j)=*(data+i+(ystart+j-1)**(axes));
      /* printf("x=%i y=%i %f %f\n", i+1, ystart+j,  *(temppos+j),*(tempval+j)); */
    }
    
    /* Check trace levels are acceptable */
    minmax(tempval,width,&datamin,&datamax);
    if( (datamax-datamin)>sigma*background ){
      
      if(recovered==0){
	printf("peaktrace: recovered trace at x=%i y=%i !! \n",i+1,ystart);
      }
      *(outdata+i)=centroid(tempval, temppos, width);
      /* printf("centre is %f\n",*(outdata+i));*/
      ystart=(int)(*(outdata+i)+0.5)-scanwidth;
      recovered=1;
      
    } else {
      
      if(recovered){
	printf("peaktrace: lost trace at x=%i y=%i !! \n",i+1,ystart);
      }
      *(outdata+i)=LOST;
      recovered=0;
      /* Don't change ystart */
      
    }

  }

  /* Write the data out */
  if ( !(outfile = fopen(argv[2],"w")) ){
    printf("linefind: Failed to open the output file\n");
    exit(1);
  }

  for(i=0; i<*(axes); i++){
    if(*(outdata+i)!=LOST){
      fprintf(outfile,"%i %f\n",i+1,*(outdata+i));
    }
  }
  
  fclose(outfile); 

  free(data);
  free(outdata);
  free(tempval);
  free(temppos);

  return 0;

}
