/* C Program ifudisplay
**
** Draws an array of hexagons, scaling the value of each hexagon
** by an input data set
**
** AJ Dean. 30th March, 2001
*/

/* Assumes that (int)num.x produces num irrespective of x hence the 
** round function 
*/

#include <stdlib.h>
#include <math.h>
#include "fitswrap.h"
#include "sumspec.h"
#include "specsort.h"
#include "cirpass.h"

static int hexagonx(float ypix, float rad);
static int minmax(float* values, int nvals, float* min, float* max);
static int round(float val);

int main(int argc, char* argv[])
{

  float xpix=0;           /* x values in pixel units */
  float ypix=0;           /* y values in pixel units */
  float xspace=0;         /* x pixel spacing of points on grid */
  float yspace=0;         /* y pixel spacing of points on grid */  
  float* xgrid=0;         /* x values on arbitary ifu grid */
  float* ygrid=0;         /* y values on arbitary ifu grid */
  float yoverx=0;         /* Ratio of the grid axes */
  float mingridx=0;       /* Size of the grid */ 
  float maxgridx=0;
  float mingridy=0;
  float maxgridy=0;
  float xgridsize=0;
  float ygridsize=0;
  float xgridmid=0;
  float ygridmid=0;

  float rad=0;          /* "radial" size of hexagon, pixels */
  int irad=0;           /* int of rad */
  int xlimit=0;         /* size of hexagon in x direction */
  double root3;         /* Root 3 */
  int numlenses=0;      /* Number of lenses to be displayed */
  float* pixvals=NULL;  /* data values for the pixels */
  float* value=NULL;    /* Intensity in the lens */
  float offsetx=0;      /* Pixel values for middle of display */
  float offsety=0;        
  int middle=0;         
  float maxdist=0;      /* max dist object can be from centre in terms of x */
  float mindata=0;      /* Minimum data valued in summed spectra */
  float maxdata=0;      /* Maximum data valued in summed spectra */


  int i=0;              /* Loop variables */
  int j=0;            
  int gridloop=0;

  long naxes[2];

  float** configdata=NULL;
  char* keywords[]={"YOVERX","MAXDIST"};
  float* keydata[2]; 
  int numhead=2;
  int numspec=0;        /* Number of spectra in input MEF file */
  int numcols;          /* Number of cols + rows in table extension */
  int numrows;

  char* exts[]={"SCI"}; /* Output file extensions */
  int naxis=2;          /* Number of axes in output data */
  float* data[1];       /* Place for output data */
  int start=0;          /* Pixel values of input spectra to loop over */
  int end=0; 
  
  float background=0;   /* Background signal in image */
  
  float*  specdata=NULL;
  float** lensdata=NULL;

  keydata[0]=&yoverx;
  keydata[1]=&maxdist;

 
  /* check for appropriate input */
  if ( argc < 5 ) {
    fprintf(stderr, 
	    "Usage: %s MEF-file output_image_file start_pixel end_pixel\n", 
	    argv[0]);
    exit(1);
  }
    
  sscanf(argv[3],"%i",&start);
  sscanf(argv[4],"%i",&end); 


  /* Read in the binary table data from the MEF file */  
  configdata=read_table(argv[1], 2, numhead, keywords, keydata, &numcols, 
       &numrows);
  
  /* Check input */
  if(numcols!=4){
    printf("Table header does not have 4 columns,\nhave you run addfibreinfo and addapinfo?\n");
    exit(1);
  }

  /* Read in and sum the spectra */
  specdata=sumspec(argv[1],start,end, &numspec);

  /* Sort the data into xgrid ygrid value */  
  lensdata=specsort(configdata, numrows, specdata, numspec, &numlenses);

  /* Allocate memory for the pixel values */
  if ((pixvals=(float *)malloc(DISPSIZE*DISPSIZE*sizeof(float))) == NULL ){
    
    printf("ifudisp: memory allocation for pixvals failed\n");
    exit(1);

  }
  
  xgrid=*(lensdata+0);
  ygrid=*(lensdata+1);
  value=*(lensdata+2);

  root3=sqrt(3.0);

  /* Find out the size of the grid */
  if ( minmax(xgrid,numlenses,&mingridx,&maxgridx) ){
    printf("minmax routine failed!\n");
    exit(1);
  }

  if ( minmax(ygrid,numlenses,&mingridy,&maxgridy) ){
    printf("minmax routine failed!\n");
    exit(1);
  }

  /* Find out range of input spectra so can set background accordingly */
  if ( minmax(value,numlenses,&mindata,&maxdata) ){
    printf("minmax routine failed!\n");
    exit(1);
  }
  
  /* Find out which axis is longest and set spacing accordingly */
  xgridsize=maxgridx-mingridx;
  ygridsize=maxgridy-mingridy;
  xgridmid=(maxgridx+mingridx)/2.0;
  ygridmid=(maxgridy+mingridy)/2.0;
 
  /* The .1 is a bit of extra space to account for the cases where the middle
   * of the grid falls exactly on a pixel, whereby one end will rap around 1 
   * pixel to the other side if the grid fits exactly into the full width.
   */
  if ( xgridsize > ygridsize*yoverx ){ /* This looks dimensionally wrong */
    /* But it's not! This is going _to_ pixel space */
   
    xspace=(DISPSIZE)/(xgridsize+2.1);
    yspace=xspace*yoverx;

  } else {
   
    yspace=(DISPSIZE)/(ygridsize+2.1);
    xspace=yspace/yoverx;

  }
    
  /*printf("xspace=%f, yspace=%f\n",xspace,yspace);*/

  rad=2.0*xspace/root3;/* Sets rad to fill x direction */
  irad=round(rad);
  /*printf("irad = %d\n",irad);*/
  
  middle=((int)(DISPSIZE/2.0))*DISPSIZE + round((DISPSIZE+1)/2.0) - 1;
  
  offsetx=xgridmid*xspace;
  offsety=ygridmid*yspace;
  
  /* Set background to 10% lower than minimum value */

  background=mindata-( (maxdata-mindata)*10.0/100.0 );

  for(i=0; i<DISPSIZE*DISPSIZE; i++)
    *(pixvals+i)=background;
  printf("background set to %f\n",*(pixvals));

  for(gridloop=0; gridloop<numlenses; gridloop++){

    /* i and j loop over a hexagon in pixel space */
    for(j=-irad; j<=irad; j++){

      xlimit=hexagonx((float)(j),rad);
      
      for(i=-xlimit; i<=xlimit; i++){

	xpix=i+*(xgrid+gridloop)*xspace;
	ypix=j+*(ygrid+gridloop)*yspace;
	
	if(*(value+gridloop)!=NOFIBRE){
       
	  *(pixvals + round(xpix-offsetx) + round(ypix-offsety)*DISPSIZE 
	    + middle)=*(value+gridloop); 
	
	} else {
	  
	  *(pixvals + round(xpix-offsetx) + round(ypix-offsety)*DISPSIZE 
	    + middle)=background; 
	}
      
      }
      /*printf("pixel num is %i xgrid=%f ygrid=%f\n",(round(xpix-offsetx) + round(ypix-offsety)*DISPSIZE + middle),*(xgrid+gridloop),*(ygrid+gridloop));*/
    }
  /* printf("%d %f\n",gridloop,*(value+gridloop)); */
    
  }
  
  naxes[0]=naxes[1]=DISPSIZE;
  naxis=2;
  data[0]=pixvals;

  write_cirp_copy_hdu_float(argv[1], argv[2], exts, data, 1, naxis, naxes);
  /*  writefits_float_basic(argv[2], naxes, pixvals);*/
  
  /* Clean up */
  free(pixvals);

  for(i=0; i<numcols; i++){
    free( *(configdata+i) );
  }
  free(configdata); 
  
  free(specdata);
    
  for(i=0;i<2; i++){
    free( *(lensdata+i) );
  }
  free(lensdata);
  
  return 0;

}
      
static int hexagonx(float ypix, float rad)
{
  
  float result=0.0; 

  if ( (ypix < -rad/2.0)){
  
    result=(sqrt(3.0) * (ypix+rad));

  } else if ( (ypix >= -rad/2.0) && (ypix <=rad/2.0) ){
    
    result=(rad * sqrt(3.0) / 2.0);
    
  } else {

    result=(sqrt(3.0) * (rad-ypix));
 
  }

  return round(result);

}

static int minmax(float* values, int nvals, float* min, float* max)
{

  int i=0;
  float mintemp=0.0;
  float maxtemp=0.0;

  mintemp=*(values);
  maxtemp=*(values);

  for(i=0; i<nvals; i++){
    
    if ( *(values+i) < mintemp ){
      mintemp=*(values+i);
    }
    
    if ( *(values+i) > maxtemp){
      maxtemp=*(values+i);
    }

  }

  *(min)=mintemp;
  *(max)=maxtemp;

  return 0;

}

static int round(float val)
{

  return ( val < 0.0 ? (int)(val-0.5) : (int)(val+0.5) );

}



