/* cube.c 
** c program to create 3D FITS file from input FITS files
** Written by AJ Dean. 14th June, 2001. 
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include "cirpass.h"
#include "fitswrap.h"

/* static funcion definitions */
static char* fgets_nonewline(char *s, int n, FILE *iop);
static float* get_data(const char* fn, float* data);

/* Main code */
int main(int argc, char* argv[])
{

  /* Local Variables */
  FILE *fp;               /* File pointer */
  char* inputfiles;       /* array for file names to stack */
  
  int num=0;              /* number of files to stack */
  int i=0;                /* loop variables */
  int imsize=0;           /* Number of pixel in image */
  float* scidata;         /* data in the input images */
  float gain=0;
                          /* keywords to delete from primary header */
  
  /* Variables for cfitsio */
  int status=0;                   /* Error handle for cfitsio */
  fitsfile *fptr;                 /* fits object as defined by cfitsio */
  fitsfile *fptr_first;             
  
  long naxis=3;                   /* 3D images */ 
  long axes[3];                   /* contains image dimensions */
  int nbytes=0;                   /* Number of bytes to read in */
 

  /* check for appropriate input */
  if ( argc < 4 ) {
    fprintf(stderr, 
	    "Usage: %s file:filelist numinlist outfile\n", 
	    argv[0]);
    exit(1);
  }

  /* convert the number to type int so can use it in loops */
  sscanf(argv[2],"%i",&num);
 
  /* Allocate memory for the file names */  
  if ( (inputfiles=(char *)malloc(num*FNAMELENGTH*sizeof(char))) == NULL ){
    
    printf("stack: Memory allocation for file name arrays failed!");
    exit(1);
  }
    
  /* open the input file and read in the files to cube */
  if ( (fp = fopen(argv[1],"r")) == NULL ){
    printf("Cannot open file containing list of files to be stacked\n");
    exit(1);
  }
    
  for ( i=0; i<num; i++ ){
    
    if ( fgets_nonewline(&inputfiles[i*FNAMELENGTH],FNAMELENGTH,fp) == NULL ){
      printf("Unable to read the input file list!!\n");
      exit(1);
    }
	
  }

  /* Close the input file containing the file list to be read in */
  fclose(fp); 
  
  axes[2]=(long)num;
  
  /* Find out the image size - assume all the same, copy header and
   * sort out keywords */
  if ( fits_open_file(&fptr_first, inputfiles, READONLY, &status) )
    printerror( status );
  
  if ( fits_read_key(fptr_first, TLONG, "NAXIS1", &axes[0], NULL, &status) )
    printerror ( status ) ; 

  if ( fits_read_key(fptr_first, TLONG, "NAXIS2", &axes[1], NULL, &status) )
    printerror ( status ) ;
  
  if ( fits_read_key(fptr_first, TFLOAT, "GAIN", &gain, NULL, &status) )
    printerror ( status ) ;

  /* Sort out keywords */
  naxis=3;      /* 3D data */

  if ( fits_close_file(fptr_first, &status) )
    printerror( status );
  
  /* Allocate memory for the data */  
  nbytes=(int)axes[0]*axes[1]*axes[2];
  imsize=(int)axes[0]*axes[1];

  if ( (scidata=(float *)malloc(nbytes*sizeof(float))) == NULL ){
    
    printf("stack: Memory allocation for data pointers failed!");
    exit(1);
  }

  printf("cube: reading in data\n");

  /* read in data */
  for( i=0; i<num; i++){

    get_data(&inputfiles[i*FNAMELENGTH], scidata+i*imsize);  
    /*printf("random data element is %f\n",*(scidata+i*imsize+3));*/
  }
  
  printf("cube: writing out image\n");

 /* Write out the data cube */
  if( write_3d_cube(argv[3], scidata, axes, 3) ){
    printf("makecube: write_3d_cube failed\n");
    exit(1);
  }  
  
  /* Copy the gain keyword over */
  if ( fits_open_file(&fptr, argv[3], READWRITE, &status) )
    printerror ( status ) ;
  
  if ( fits_write_key(fptr, TFLOAT, "GAIN", &gain, NULL, &status) )
      printerror ( status );

  if ( fits_close_file(fptr, &status) )
    printerror( status );
  
  /* Free memory allocated locally */
  free(inputfiles);
  free(scidata);
 
  /* succesfull return */
  return 0;

}

/* Static functions */

/* This is the same code as the standard library function fgets, 
** except for the -- in the second to last line,
** which makes it overwrite the newline with a trailing 0.
**/

char *fgets_nonewline(char *s, int n, FILE *iop)
{

  register int c=0;
  register char *cs;

  cs = s;
  while (--n > 0 && ( c = getc(iop)) !=EOF)
    if ((*cs++ = c) == '\n')
      break;
  *(--cs) = '\0';
  return ( c == EOF && cs == s ) ? NULL : s;

}

float* get_data(const char* fn, float* data)
{  

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  long nx, ny;                         /* Dimensions of image */
  int nbytes;                          /* Size of array required for data */
    
  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
    
  if ( fits_open_file(&fptr, fn, READONLY, &status) )
    
    printerror ( status ) ;
  
  /* Get info from header */
  if ( fits_read_key(fptr, TLONG, "NAXIS1", &nx, NULL, &status) ){
    printf("Failed to read naxis1 from header\n");
    printerror ( status ) ; 
  }
  
  if ( fits_read_key(fptr, TLONG, "NAXIS2", &ny, NULL, &status) ){
    printf("Failed to read naxis2 from header\n");
    printerror ( status ) ;
  }
  
  /* Find out size of array required and allocate necessary memory */
  nbytes=nx*ny*sizeof(float);
  nread=nx*ny;
  
  if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, &status) )
   printerror( status );
  
  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );
 
  return data;
 
}







