/* fitswrap.c -- high level wrapper for cfitsio FITS library */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fitswrap.h"


/* ------------------ Static function definitions ------------------------- */

static void parseheader(fitsfile* fptr, long* nx, long* ny, int* bitpix,
			int status);

/* ------------------ External functions ---------------------------------- */

/* ------------------ Reading in Data ------------------------------------- */

/*
 * read a float FITS file given as fitfile.fits[ext], 
 * return pointer to image data.
 */


extern int readkeys(const char *p_fname,long *axes, int* bitpix)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
 

  if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
    printerror ( status ) ;

 /* Get info from header */

  parseheader(fptr, axes, (axes+1), bitpix, status);
 
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}


extern float *readfits_float(const char *fn, int ext)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  long nx, ny;                          /* Dimensions of image */
  int nbytes;                          /* Size of array required for data */
  float *data;                         /* Pointer to the actual data */
  
  char* p_fname;
  char fname[FLEN_FILENAME];           /* Holds the filename and extension */
  char num[FLEN_EXT];                  /* For the extension number as a char */


  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */

  /* check strings have enough room for the file + [extension] */

  if (  ((strlen(fn)/sizeof(fn[0])) + FLEN_EXT + 2) > FLEN_FILENAME ){
    fprintf(stderr,"fitswrap: Filename character array does not have enough space\nEdit fitsio.h - FLEN_FILENAME if you require longer filenames\n");
    exit(1);
  }

  if ( ext > 999 ){
    fprintf(stderr,"fitswrap: Filename character array does not have enough space for \nmore than 3 digit extensions i.e. up to 999. \nEdit fitswrap.h - FLEN_EXT if you require more extensions\n");
    exit(1);
  }
 
  /* make up the new string name including the extension */

  p_fname=&fname[0];
  p_fname=strcpy(p_fname,fn);

  sprintf(&num[0],"%i",ext); 

  
  if ( ext > 0 ){
  
    p_fname=strcat(p_fname,"[");  
    p_fname=strcat(p_fname,&num[0]); 
    p_fname=strcat(p_fname,"]"); 
   
  }  

 /* printf("fitswrap: readfits_float, reading %s \n",fname); */  

  
 if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
   printerror ( status ) ;

 /* Get info from header */

 parseheader(fptr, &nx, &ny, &bitpix, status);
 
 /* printf("nx is %i, ny is %i, bitpix is %i\n",nx,ny,bitpix); */

 /* Check that data type is as expected */

 /* Add back to code once stack is sorted !! ******************************


 printf("%i \n", bitpix);
 
 if (bitpix !=-32) {
        fprintf(stderr, "fitswrap, readfits_: expected BITPIX == -32\n");
        exit(1);
    }
 
 */

 /* Find out size of array required and allocate necessary memory */

 nbytes=nx*ny*sizeof(float);
 nread=nx*ny;
 
 /* Read in the data */

 if ( (data=(float *)malloc(nbytes)) == NULL ){

   printf("fitswrap, readfits_float: Memory allocation failed!");
   exit(1);
 }

 if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, &status) )
    printerror( status );
     

 /* Close the file freeing memory allocated for fitsfile fptr */

 if ( fits_close_file(fptr, &status) )
      printerror ( status );

    return data;

}

extern float *readfits_float_clever(const char *fn, long* axes)
{
  
  int status=0;                   /* Error handle for cfitsio */
  fitsfile *fptr;                 /* fits object as defined by cfitsio */

  int bitpix;                     /* Bitpix in fits image */
  long nx, ny;                    /* Dimensions of image */
  int nbytes;                     /* Size of array required for data */
  float *data;                    /* Pointer to the actual data */
  int extend;                     /* (Logical) Does file contain extensions? */

  int fpixel=1;                   /* First pixel to read */
  int nread;                      /* Number of pixels to read */     
  int nullvall=0;                 /* Markers for bad pixels */
  int anynull=0;                  /* as used by cfitsio */

  
 if ( fits_open_file(&fptr, fn, READONLY, &status) )
    
   printerror ( status ) ;

 /* Find out if we have a MEF file */
 if ( fits_read_key(fptr, TLOGICAL, "EXTEND", &extend, NULL, &status) )

   printerror ( status ) ;

 if ( extend == TRUE ){ /* If have extensions move to sci 1, otherwise O.K. */

   if ( fits_movnam_hdu(fptr, ANY_HDU, "SCI", 1, &status) )

     printerror ( status );

 }

 /* Get info from header */
 parseheader(fptr, &nx, &ny, &bitpix, status);
 
 axes[0]=nx;
 axes[1]=ny;

 /* printf("nx is %i, ny is %i, bitpix is %i\n",nx,ny,bitpix); */

 /* Check that data type is as expected */

 /* Add back to code once stack is sorted !! ******************************


 printf("%i \n", bitpix);
 
 if (bitpix !=-32) {
        fprintf(stderr, "fitswrap, readfits_: expected BITPIX == -32\n");
        exit(1);
    }
 
 */

 /* Find out size of array required and allocate necessary memory */

 nbytes=nx*ny*sizeof(float);
 nread=nx*ny;
 
 /* Read in the data */

 if ( (data=(float *)malloc(nbytes)) == NULL ){

   printf("fitswrap, readfits_float: Memory allocation failed!");
   exit(1);
 }

 if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, &status) )
    printerror( status );
     

 /* Close the file freeing memory allocated for fitsfile fptr */

 if ( fits_close_file(fptr, &status) )
      printerror ( status );

    return data;

}


extern int read_float(const char *fn, float* data)
{
  /* !!!!!!!! Untested !!!!!!!!! */

  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  long nx, ny;                          /* Dimensions of image */
  int nbytes;                          /* Size of array required for data */
 
  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */

  /* check strings have enough room for the file + [extension] */

  if ( fits_open_file(&fptr, fn, READONLY, &status) )
    
    printerror ( status ) ;
  
  /* Get info from header */

  parseheader(fptr, &nx, &ny, &bitpix, status);
 
  /* printf("nx is %i, ny is %i, bitpix is %i\n",nx,ny,bitpix); */
  
  /* Check that data type is as expected */
  
  /* Add back to code once stack is sorted !! ******************************

     
     printf("%i \n", bitpix);
     
     if (bitpix !=-32) {
     fprintf(stderr, "fitswrap, readfits_: expected BITPIX == -32\n");
     exit(1);
     }
     
  */
  
  /* Find out size of array required and allocate necessary memory */
  
  nbytes=nx*ny*sizeof(float);
  nread=nx*ny;
  
  /* Read in the data */
  if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, &status) )
    printerror( status );
  
 /* Close the file freeing memory allocated for fitsfile fptr */
  
  if ( fits_close_file(fptr, &status) )
    printerror ( status );
  
  return 0;
  
}

/*
 * read a float FITS file given as fitfile.fits[extname,extver], 
 * return pointer to image data.
 */

extern float *readfits_float_extver(const char *fn, const char *p_extname, int extver)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  long nx, ny;                          /* Dimensions of image */
  int nbytes;                          /* Size of array required for data */
  float *data;                         /* Pointer to the actual data */
  
  char* p_fname;
  char fname[FLEN_FILENAME];
                                       /* Holds the filename and extension */
  char num[FLEN_EXT];                  /* For the extension number as a char */


  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */


  /* check strings have enough room for the file + [extension] */

  if (  ((strlen(fn)/sizeof(fn[0])) + FLEN_EXT + FLEN_EXTNAME + 1) > FLEN_FILENAME ){
    fprintf(stderr,"fitswrap: Filename character array does not have enough space\nEdit fitsio.h - FLEN_FILENAME if you require longer filenames\n");
    exit(1);
  }


   if ( extver > 999 ){
    fprintf(stderr,"fitswrap: Filename character array does not have enough space for \nmore than 3 digit extensions i.e. up to 999. \nEdit fitswrap.h - FLEN_EXT if you require more extensions\n");
    exit(1);
  }

  if ( ((strlen(p_extname)/sizeof(p_extname[0]))+1) > FLEN_EXTNAME ){
    fprintf(stderr,"fitswrap, readfits_float_ext: extension name is too big.\nEdit fitswrap.h - FLEN_EXTNAME if you require more space for extnesion names \n");
    exit(1);
  }

  /* make up the new string name including the extension */

  p_fname=&fname[0];
  p_fname=strcpy(p_fname,fn);
  
  sprintf(&num[0],"%i",extver); 
  
    p_fname=strcat(p_fname,"[");
    p_fname=strcat(p_fname,p_extname);
    p_fname=strcat(p_fname,",");
    p_fname=strcat(p_fname,&num[0]); 
    p_fname=strcat(p_fname,"]"); 
     

    /*  printf("fitswrap:readfits_float_extver opening, %s \n",fname); */ 

  
 if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
   printerror ( status ) ;

 /* Get info from header */

 parseheader(fptr, &nx, &ny, &bitpix, status);
 
 /* printf("nx is %i, ny is %i, bitpix is %i\n",nx,ny,bitpix); */

 /* Check that data type is as expected */

 /* Add back to code once stack is sorted !! ******************************


 printf("%i \n", bitpix);
 
 if (bitpix !=-32) {
        fprintf(stderr, "fitswrap, readfits_: expected BITPIX == -32\n");
        exit(1);
    }
 
 */

 /* Find out size of array required and allocate necessary memory */

 nbytes=nx*ny*sizeof(float);
 nread=nx*ny;
 
 /* Read in the data */

 if ( (data=(float *)malloc(nbytes)) == NULL ){

   printf("fitswrap, readfits_float: Memory allocation failed!");
   exit(1);
 }

 if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, &status) )
    printerror( status );
     

 /* Close the file freeing memory allocated for fitsfile fptr */

 if ( fits_close_file(fptr, &status) )
      printerror ( status );

    return data;

}

extern double *readfits_double_extver(const char *fn, const char *p_extname, int extver, long* nx, long* ny)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  int nbytes;                          /* Size of array required for data */
  double *data;                         /* Pointer to the actual data */
  
  char* p_fname;
  char fname[FLEN_FILENAME];
                                       /* Holds the filename and extension */
  char num[FLEN_EXT];                  /* For the extension number as a char */


  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */


  /* check strings have enough room for the file + [extension] */

  if (  ((strlen(fn)/sizeof(fn[0])) + FLEN_EXT + FLEN_EXTNAME + 1) > FLEN_FILENAME ){
    fprintf(stderr,"fitswrap: Filename character array does not have enough space\nEdit fitsio.h - FLEN_FILENAME if you require longer filenames\n");
    exit(1);
  }


   if ( extver > 999 ){
    fprintf(stderr,"fitswrap: Filename character array does not have enough space for \nmore than 3 digit extensions i.e. up to 999. \nEdit fitswrap.h - FLEN_EXT if you require more extensions\n");
    exit(1);
  }

  if ( ((strlen(p_extname)/sizeof(p_extname[0]))+1) > FLEN_EXTNAME ){
    fprintf(stderr,"fitswrap, readfits_double_ext: extension name is too big.\nEdit fitswrap.h - FLEN_EXTNAME if you require more space for extnesion names \n");
    exit(1);
  }

  /* make up the new string name including the extension */

  p_fname=&fname[0];
  p_fname=strcpy(p_fname,fn);
  
  sprintf(&num[0],"%i",extver); 
  
    p_fname=strcat(p_fname,"[");
    p_fname=strcat(p_fname,p_extname);
    p_fname=strcat(p_fname,",");
    p_fname=strcat(p_fname,&num[0]); 
    p_fname=strcat(p_fname,"]"); 
     

    /*  printf("fitswrap:readfits_double_extver opening, %s \n",fname); */ 

  
 if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
   printerror ( status ) ;

 /* Get info from header */

 parseheader(fptr, nx, ny, &bitpix, status);
 
 /* printf("nx is %i, ny is %i, bitpix is %i\n",*nx,*ny,bitpix); */

 /* Check that data type is as expected */

 /* Add back to code once stack is sorted !! ******************************


 printf("%i \n", bitpix);
 
 if (bitpix !=-32) {
        fprintf(stderr, "fitswrap, readfits_: expected BITPIX == -32\n");
        exit(1);
    }
 
 */

 /* Find out size of array required and allocate necessary memory */

 nbytes=*nx**ny*sizeof(double);
 nread=*nx**ny;
 
 /* Read in the data */

 if ( (data=(double *)malloc(nbytes)) == NULL ){

   printf("fitswrap, readfits_double: Memory allocation failed!");
   exit(1);
 }

 if ( fits_read_img(fptr, TDOUBLE, fpixel, nread, &nullvall, data, &anynull, &status) )
    printerror( status );
     

 /* Close the file freeing memory allocated for fitsfile fptr */

 if ( fits_close_file(fptr, &status) )
      printerror ( status );

    return data;

}

/* ------------------ Writing out data ----------------------------------- */

extern int writefits_float_basic(const char *outfile, long* naxes, float *data){

  fitsfile *fptr;       /* pointer to the FITS file, defined in fitsio.h */
  int status;
  int  fpixel, nelements;

  int bitpix   =  FLOAT_IMG;  /* 32 bit float pixel values*/
  long naxis=2;               /* 2D images */ 
  status = 0;         /* initialize status before calling fitsio routines */

  if (fits_create_file(&fptr, outfile, &status)) /* create new FITS file */
         printerror( status );           /* call printerror if error occurs */

  /* write the required keywords for the primary array image.     */
                                          
  if ( fits_create_img(fptr,  bitpix, naxis, naxes, &status) )
         printerror( status );          
   
  fpixel = 1;                           /* first pixel to write      */
  nelements = naxes[0] * naxes[1];      /* number of pixels to write */

  /* write the array of floats to the FITS file */

  if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data, &status) )
        printerror( status );            
   
  if ( fits_close_file(fptr, &status) )
    printerror( status );
  
  return 0;

}

extern int writefits_double_basic(const char *outfile, long* naxes, double *data){

  fitsfile *fptr;       /* pointer to the FITS file, defined in fitsio.h */
  int status;
  int  fpixel, nelements;

  int bitpix   =  DOUBLE_IMG;  /* 64 bit float pixel values*/
  long naxis=2;               /* 2D images */ 
  status = 0;         /* initialize status before calling fitsio routines */

  if (fits_create_file(&fptr, outfile, &status)) /* create new FITS file */
         printerror( status );           /* call printerror if error occurs */

  /* write the required keywords for the primary array image.     */
                                          
  if ( fits_create_img(fptr,  bitpix, naxis, naxes, &status) )
         printerror( status );          
   
  fpixel = 1;                           /* first pixel to write      */
  nelements = naxes[0] * naxes[1];      /* number of pixels to write */

  /* write the array of floats to the FITS file */

  if ( fits_write_img(fptr, TDOUBLE, fpixel, nelements, data, &status) )
        printerror( status );            
   
  if ( fits_close_file(fptr, &status) )
    printerror( status );
  
  return 0;

}

extern int writefits_float_header(const char *output_file, 
				  const char *header_file, long* naxes, 
				  int naxis, float* data)
{

  fitsfile *fptr;    
  int status=0;
  int i=0;
  int bitpix=0;
  int ver=0;
  int fpixel=0;
  int nelements=0;

  /* Make new file based on header from input header file */
  if( fits_create_template(&fptr, output_file, header_file, &status) )
    printerror( status );

  /* Sort out the keywords */
  if ( fits_update_key(fptr, TINT, "NAXIS", &naxis, NULL, &status) )
    printerror( status );
  
  if ( fits_update_key(fptr, TINT, "NAXIS1", naxes, NULL, &status) )
    printerror( status );

  if ( fits_update_key(fptr, TINT, "NAXIS2", naxes+1, NULL, &status) )
    printerror( status );
  
  bitpix=FLOAT_IMG;
  if ( fits_update_key(fptr, TINT, "BITPIX", &bitpix, NULL, &status) )
    printerror( status );

  /* Image data */
  ver=1;
  fpixel=1;
  nelements=1;

  for(i=0; i<naxis; i++){
    nelements*=*(naxes+i);
  }

  if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data, 
		      &status) )
      printerror( status );            
    
  if( fits_close_file(fptr, &status) )
    printerror( status );

  return 0;

}

extern int writefits_float_header_clever(const char *output_file, 
				  const char *header_file, long* naxes, 
				  int naxis, float* data)
{

  fitsfile *fptr;  
  fitsfile *fptr_template;
  int status=0;
  int i=0;
  int bitpix=0;
  int ver=0;
  int fpixel=0;
  int nelements=0;
  int extend=0;      
  
  /* Open the header file */
  if( fits_open_file(&fptr_template, header_file, READONLY, &status) )
    printerror ( status ); 

  /* Find out if we have a MEF file */
  if ( fits_read_key(fptr_template, TLOGICAL, "EXTEND", &extend, NULL, 
		    &status) )
    printerror ( status ) ;

  if ( extend == TRUE ){ /* If have extensions move to sci 1, otherwise O.K. */

    if ( fits_movnam_hdu(fptr_template, ANY_HDU, "SCI", 1, &status) )
     
      printerror ( status );

  }
 
  if( fits_create_file(&fptr, output_file, &status) )
    printerror( status );

  if( fits_copy_hdu(fptr_template, fptr, 0, &status) );
  
  /* Sort out the keywords */
  if ( fits_update_key(fptr, TINT, "NAXIS", &naxis, NULL, &status) )
    printerror( status );
  
  if ( fits_update_key(fptr, TINT, "NAXIS1", naxes, NULL, &status) )
    printerror( status );

  if ( fits_update_key(fptr, TINT, "NAXIS2", naxes+1, NULL, &status) )
    printerror( status );
  
  bitpix=FLOAT_IMG;
  if ( fits_update_key(fptr, TINT, "BITPIX", &bitpix, NULL, &status) )
    printerror( status );

  /* Image data */
  ver=1;
  fpixel=1;
  nelements=1;

  for(i=0; i<naxis; i++){
    nelements*=*(naxes+i);
  }

  if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data, 
		      &status) )
      printerror( status );            
    
  if( fits_close_file(fptr, &status) )
    printerror( status );

  if( fits_close_file(fptr_template, &status) )
    printerror( status );

  return 0;

}

extern int writefits_short_basic(const char *outfile, long* naxes, short *data){

  fitsfile *fptr;       /* pointer to the FITS file, defined in fitsio.h */
  int status;
  int  fpixel, nelements;

  int bitpix   =  SHORT_IMG;  /* 16 bit float pixel values*/
  long naxis=2;               /* 2D images */ 
  status = 0;         /* initialize status before calling fitsio routines */

  if (fits_create_file(&fptr, outfile, &status)) /* create new FITS file */
         printerror( status );           /* call printerror if error occurs */

  /* write the required keywords for the primary array image.     */
                                          
  if ( fits_create_img(fptr,  bitpix, naxis, naxes, &status) )
         printerror( status );          
   
  fpixel = 1;                           /* first pixel to write      */
  nelements = naxes[0] * naxes[1];      /* number of pixels to write */

  /* write the array of SHORTS to the FITS file */

  if ( fits_write_img(fptr, TSHORT, fpixel, nelements, data, &status) )
        printerror( status );            
   
  if ( fits_close_file(fptr, &status) )
    printerror( status );
  
  return 0;

}

extern int write_3d_cube(char* filename, float* data, long* axes, int naxis)
{

  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  
  long imsize;                       /* Number of pixels to write out */

  int fpixel=1;                     /* First pixel to write*/
  int i=0;                          /* Loop variable */
  int bitpix=FLOAT_IMG;
  
  printf("write_3d_cube: Have %iD data\n",naxis);
  
  imsize=1;
  for(i=0; i<naxis; i++){
    imsize *= *(axes+i);
  }
  
  /* Write out the data */
  if ( fits_create_file(&fptr, filename, &status) )    
    printerror ( status ) ;

  if ( fits_create_img(fptr, bitpix, naxis, axes, &status) )
    printerror ( status ) ;

  if ( fits_write_img(fptr, TFLOAT, fpixel, imsize, data, &status) )
    printerror ( status ) ;

  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return 0;

}




/* ----------------- Multi Extension FITS files ------------------------- */

/* ----------------- Reading data --------------------------------------- */ 

extern int readkeys_MEF(const char *p_fname,long *axes, int* nextend,
			int* bitpix)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
 
  if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
    printerror ( status ) ;
  
  if ( fits_read_key(fptr, TINT, "NEXTEND", nextend, NULL, &status) )
      
    printerror ( status ) ;
  
  /* Move to the first science header */
  if ( fits_movnam_hdu(fptr, ANY_HDU, "SCI", 1, &status) )
    
    printerror ( status ) ;

  /* Get info from header */

  parseheader(fptr, axes, (axes+1), bitpix, status);
 
 
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}


float* read_ME_FITS(char* filename, char* exttype, long* nx, long* ny, 
		    int* nextend)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  int nbytes;                          /* Size of array required for data */
  float *data;                         /* Pointer to the actual data */
  

  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
  int next=0;                          /* Number extensions to read in*/
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    
    printerror ( status ) ;

  /* Get info from header */
  if ( fits_read_key(fptr, TINT, "NEXTEND", nextend, NULL, &status) ){

    printf("fitswrap, read_ME_FITS: nextend keyword not found\n");
    printerror ( status ) ;
  }


  /* Need to move to the first extension header as the keywords 
     aren't in the primary header */
  if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, 1, &status) ){
    
    printf("fitswrap, read_ME_FITS: Could not move to extension type %s, number 1\n", exttype);
    printerror( status );
    
  }

  /*  printf("fitswrap, read_ME_FITS: Moving to extension sci 1\n"); */
  parseheader(fptr, nx, ny, &bitpix, status);
 
  next = *nextend / NIMSET;

  nbytes=(*nx)*(*ny)*next*sizeof(float);
  nread=(*nx)*(*ny);
 

  /* Allocate memory */

  if ( (data=(float *)malloc(nbytes)) == NULL ){

    printf("fitswrap, read_ME_FITS: Memory allocation failed!");
    exit(1);
  }

  /* Read in the data */

  /* Loop over the extensions */

  /* printf("fitswrap, read_ME_FITS: number of extensions to read in is %i\n",next); */

  for ( i=0; i<next; i++) {

    /* We should already be there for the first extension */

    if( i!=0 ){

      if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, i+1, &status) ){

	printf("Could not move to extension type %s, number %i\n", exttype, 
	       i+1);
	printerror( status );

      }

    }

    /*  printf("fitswrap, read_ME_FITS: reading in extension type %s, number %i\n", exttype, i+1); */
   
    if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, 
		      (data + nread*i), &anynull, &status) )
      
      printerror( status );
   
  }
 

  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return data;

}

float* read_MS_ME_FITS(char* filename, char* exttype, long* nx, long* ny, 
		    int* nextend)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  int nbytes;                          /* Size of array required for data */
  float *data;                         /* Pointer to the actual data */
  

  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
  int next=0;                          /* Number extensions to read in*/
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    
    printerror ( status ) ;

  /* Get info from header */
  if ( fits_read_key(fptr, TINT, "NEXTEND", nextend, NULL, &status) ){

    printf("fitswrap, read_MS_ME_FITS: nextend keyword not found\n");
    printerror ( status ) ;
  }


  /* Need to move to the first extension header as the keywords 
     aren't in the primary header */
  if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, 1, &status) ){
    
    printf("fitswrap, read_MS_ME_FITS: Could not move to extension type %s, number 1\n", exttype);
    printerror( status );
    
  }

  /*  printf("fitswrap, read_MS_ME_FITS: Moving to extension sci 1\n"); */
  parseheader(fptr, nx, ny, &bitpix, status);
 
  next = *nextend / MSNIMSET;

  nbytes=(*nx)*(*ny)*next*sizeof(float);
  nread=(*nx)*(*ny);
 

  /* Allocate memory */

  if ( (data=(float *)malloc(nbytes)) == NULL ){

    printf("fitswrap, read_MS_ME_FITS: Memory allocation failed!");
    exit(1);
  }

  /* Read in the data */

  /* Loop over the extensions */

  /* printf("fitswrap, read_MS_ME_FITS: number of extensions to read in is %i\n",next); */

  for ( i=0; i<next; i++) {

    /* We should already be there for the first extension */

    if( i!=0 ){

      if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, i+1, &status) ){

	printf("Could not move to extension type %s, number %i\n", exttype, 
	       i+1);
	printerror( status );

      }

    }

    /*  printf("fitswrap, read_MS_ME_FITS: reading in extension type %s, number %i\n", exttype, i+1); */
   
    if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, 
		      (data + nread*i), &anynull, &status) )
      
      printerror( status );
   
  }
 

  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return data;

}

extern float* read_cube(char* filename, long* nx, long* ny, long* nz)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  int nbytes;                          /* Size of array required for data */
  float *data;                         /* Pointer to the actual data */
  

  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
  
  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    
    printerror ( status ) ;

  /* Get info from header */
  if ( fits_read_key(fptr, TINT, "NAXIS3", nz, NULL, &status) ){

    printf("fitswrap, read_cube: NAXIS3 keyword not found\n");
    printerror ( status ) ;
  }

  parseheader(fptr, nx, ny, &bitpix, status);

  nbytes=(*nx)*(*ny)*(*nz)*sizeof(float);
  nread=(*nx)*(*ny)*(*nz);
 

  /* Allocate memory */

  if ( (data=(float *)malloc(nbytes)) == NULL ){

    printf("fitswrap, read_MS_ME_FITS: Memory allocation failed!");
    exit(1);
  }

  /* Read in the data */

  if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, data, &anynull, 
		     &status) )
    printerror( status );
  
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return data;

}

float* read_imset(char* filename, int imset_wanted, int nimset, char** names,
		  long* nx, long* ny, int* bitpix)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int nbytes;                          /* Size of array required for data */
  float *data=NULL;                    /* Pointer to the actual data */
  
  int fpixel=1;                        /* First pixel to read */
  int nread=0;                         /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
  int i=0;                             /* Loop variable */
  int headernum=0;

  printf("entering library code\n");

  if ( fits_open_file(&fptr, filename, READONLY, &status) ){
    printf("fitswrap, read_imset can't open input file\n");
    printerror ( status ) ;
  }

  /* Loop over the imset reading in the keywords and data */
  for(i=0; i<nimset; i++){
    
    printf("reading loop %d\n",i+1);

    if(i==0){
      
      /* Move to the first header of the requested set */ 
      /* +2 and not +1 as we have 0 based headers */
      
      headernum=((imset_wanted-1)*nimset)+2; 
      printf("imset_wanted: %d, nimset %d, headernum: %d\n",imset_wanted,nimset,headernum);
      
      if ( fits_movabs_hdu(fptr, headernum, NULL, &status) ){
	printf("fitswrap, read_imset: can't move to requested header\n");
	printerror ( status );
      }
  
      /* Assume this info is common to all headers */
      parseheader(fptr, nx, ny, bitpix, status);

      /* Work out how much memory we're going to need and allocate it */
      nbytes=(*nx)*(*ny)*nimset*sizeof(float);
      nread=(*nx)*(*ny);
      
      if ( (data=(float *)malloc(nbytes)) == NULL ){
	printf("fitswrap, read_ME_FITS: Memory allocation failed!");
	exit(1);
      }
      
    } else {

      /* Move one header forward */

      if ( fits_movrel_hdu(fptr, 1, NULL, &status) ){
	printf("fitswrap, read_imset: can't move to next extension header\n");
	printerror ( status );
      }

    }
    
    /* Find out the extension name */
    if ( fits_read_key(fptr, TSTRING, "EXTNAME",*(names+i), NULL, &status) ){
      printf("fitswrap, read_imset: extname keyword not found\n");
      printerror ( status ) ;
    }

    /* Read in the data */

    if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, 
		       (data + nread*i), &anynull, &status) ){
      printf("fitswrap, read_imset: unable to read in data\n");
      printerror( status );
    }
  
  }

  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
    printerror ( status );
  
  return data;

}


short* read_ME_FITS_short(char* filename, char* exttype, long* nx, long* ny, 
		    int* nextend)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  
  int bitpix;                          /* Bitpix in fits image */
  int nbytes;                          /* Size of array required for data */
  short *data;                         /* Pointer to the actual data */
  

  int fpixel=1;                        /* First pixel to read */
  int nread;                           /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */
  int next=0;                          /* Number extensions to read in*/
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    
    printerror ( status ) ;

  /* Get info from header */

  if ( fits_read_key(fptr, TINT, "NEXTEND", nextend, NULL, &status) ){

    printf("fitswrap, read_ME_FITS: nextend keyword not found\n");
    printerror ( status ) ;
  }


  /* Need to move to the first extension header as the keywords 
     aren't in the primary header */

  if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, 1, &status) ){
    
    printf("fitswrap, read_ME_FITS: Could not move to extension type %s, number 1\n", exttype);
    printerror( status );
    
  }

  /*  printf("fitswrap, read_ME_FITS: Moving to extension sci 1\n"); */
  parseheader(fptr, nx, ny, &bitpix, status);
 
  next = *nextend / NIMSET;

  nbytes=(*nx)*(*ny)*next*sizeof(short);
  nread=(*nx)*(*ny);
 

  /* Allocate memory */

  if ( (data=(short *)malloc(nbytes)) == NULL ){

    printf("fitswrap, read_ME_FITS: Memory allocation failed!");
    exit(1);
  }

  /* Read in the data */

  /* Loop over the extensions */

  /* printf("fitswrap, read_ME_FITS: number of extensions to read in is %i\n",next); */

  for ( i=0; i<next; i++) {

    /* We should already be there for the first extension */

    if( i!=0 ){

      if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, i+1, &status) ){

	printf("Could not move to extension type %s, number %i\n", exttype, 
	       i+1);
	printerror( status );

      }

    }

    /*  printf("fitswrap, read_ME_FITS: reading in extension type %s, number %i\n", exttype, i+1); */
   
    if ( fits_read_img(fptr, TSHORT, fpixel, nread, &nullvall, 
		      (data + nread*i), &anynull, &status) )
      
      printerror( status );
   
  }
 

  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return data;

}

extern float** read_table(const char *filename, int pos, int numhead,  
			  char* keywords[], float** keydata, int* numcols, 
			  int* numrows)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int hdu_type=0;                      /* Header type */
  int i=0;                             /* Loop variable */
  float** data=NULL;


  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    
    printerror ( status ) ;

  
  /* Move to the table extension at pos */

  if ( fits_movabs_hdu(fptr, pos, &hdu_type, &status) ){
    
    printf("fitswrap, read_table:Could not move to requested header\n");
    printerror( status );
    
  }

  /* Check we've got a table */

  if ( hdu_type != BINARY_TBL ){
    printf("fitswrap: read_table, this is not a table extension!!\n");
    exit(1);
  }
  
  /* Find out it's size */
  
  if ( fits_read_key(fptr, TINT, "TFIELDS", numcols, NULL, &status) ){ 
    printf("fitswrap, read_table: tfields keyword not found\n");
    printerror ( status ) ;
  }
  
  if ( fits_read_key(fptr, TINT, "NAXIS2", numrows, NULL, &status) ){
    printf("fitswrap, read_table: naxis2 keyword not found\n");
    printerror ( status ) ;
  }

  /* Read in the additional keywords required */
  for(i=0; i<numhead; i++){
    
    if ( fits_read_key(fptr, TFLOAT, *(keywords+i), *(keydata+i), NULL, &status) ){
      printf("fitswrap, read_table: %s keyword not found\n",*(keywords+i));
      printerror ( status ) ;
    }
    
    /*printf("numcols %d, numrows %d, %s=%f\n",*numcols,*numrows,*(keywords+i),
     *( *(keydata+i))); */
    
  }
  
  /* Allocate memory and Read in data */
  
  if ( ( data=(float**)malloc(*numcols*sizeof(float*)) ) == NULL ){
    printf("fitswrap, read_table: Memory allocation failed!");
    exit(1);
  }

  for(i=0; i<*numcols; i++){

     if ( ( *(data+i)=(float*)malloc(*numrows*sizeof(float)) ) == NULL ){
       printf("fitswrap, read_table: Memory allocation failed!");
       exit(1);
     }
    
     if ( fits_read_col(fptr, TFLOAT, i+1, 1, 1, *numrows, 0, *(data+i), 
			NULL, &status) )
      printerror ( status );

  }

  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );

  return data;

}

extern float** read_table_check(const char *filename, int pos, int numhead,  
				char* keywords[], float** keydata, 
				int* numcols, int* numrows)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int hdu_type=0;                      /* Header type */
  int i=0;                             /* Loop variable */
  float** data=NULL;

  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    printerror ( status ) ;
  
  /* Move to the table extension at pos */
  if ( fits_movabs_hdu(fptr, pos, &hdu_type, &status) ){
    
    printf("fitswrap, read_table_check:Could not move to requested header\n");
    printerror( status );
    
  }

  /* Check we've got a table */
  if ( hdu_type != BINARY_TBL ){
    /* No table so return with null pointer */
    return NULL;
  }
  
  /* Find out it's size */
  if ( fits_read_key(fptr, TINT, "TFIELDS", numcols, NULL, &status) ){ 
    printf("fitswrap, read_table_check: tfields keyword not found\n");
    printerror ( status ) ;
  }
  
  if ( fits_read_key(fptr, TINT, "NAXIS2", numrows, NULL, &status) ){
    printf("fitswrap, read_table_check: naxis2 keyword not found\n");
    printerror ( status ) ;
  }

  /* Read in the additional keywords required */
  for(i=0; i<numhead; i++){
  
    if ( fits_read_key(fptr, TFLOAT, *(keywords+i), *(keydata+i), NULL, 
		       &status) ){
      printf("fitswrap, read_table: %s keyword not found\n",*(keywords+i));
      printerror ( status ) ;
    }
    
    /*printf("numcols %d, numrows %d, %s=%f\n",*numcols,*numrows,*(keywords+i),
     *( *(keydata+i))); */
    
  }
  
  /* Allocate memory and Read in data */
  
  if ( ( data=(float**)malloc(*numcols*sizeof(float*)) ) == NULL ){
    printf("fitswrap, read_table: Memory allocation failed!");
    exit(1);
  }
  
  for(i=0; i<*numcols; i++){
  
     if ( ( *(data+i)=(float*)malloc(*numrows*sizeof(float)) ) == NULL ){
       printf("fitswrap, read_table: Memory allocation failed!");
       exit(1);
     }
    
     if ( fits_read_col(fptr, TFLOAT, i+1, 1, 1, *numrows, 0, *(data+i), 
			NULL, &status) )
      printerror ( status );

  }

  /* Close the file freeing memory allocated for fitsfile fptr */
  
  if ( fits_close_file(fptr, &status) )
      printerror ( status );

  return data;
  
}

extern char** read_table_column_names(const char *filename, int pos, 
				      int* numcols, int* numrows){
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int hdu_type=0;                      /* Header type */
  int i=0;                             /* Loop variable */
  char** colnames=NULL;
  
  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    printerror ( status ) ;

  /* Move to the table extension at pos */
  if ( fits_movabs_hdu(fptr, pos, &hdu_type, &status) ){
    
    printf("fitswrap, read_table_column_names:Could not move to requested header\n");
    printerror( status );
    
  }

  /* Check we've got a table */
  if ( hdu_type != BINARY_TBL ){
    /* No table so return with null pointer */
    return NULL;
  }
  
  /* Find out it's size */
  if ( fits_read_key(fptr, TINT, "TFIELDS", numcols, NULL, &status) ){ 
     printf("fitswrap, read_table_column_names: tfields keyword not found\n"); 
     printerror ( status ) ;
  }

  if ( fits_read_key(fptr, TINT, "NAXIS2", numrows, NULL, &status) ){
    printf("fitswrap, read_table_column_names: naxis2 keyword not found\n");
    printerror ( status ) ;
  }

  /* Get the column names */ 
  if ( ( colnames=(char**)malloc(*numcols*sizeof(char*)) ) == NULL ){
    printf("fitswrap, read_table_column_names: Memory allocation failed!");
    exit(1);
  }
  
  for(i=0; i<*numcols; i++){
    
    if ( ( *(colnames+i)=(char*)malloc(FLEN_CARD*sizeof(char)) ) == NULL ){
      printf("fitswrap, read_table_column_names: Memory allocation failed!");
      exit(1);
    }
    
    
    if ( fits_get_bcolparms(fptr,i+1, colnames[i], NULL, NULL, NULL, NULL, 
			    NULL, NULL, NULL, &status) ){
      printf("fitswrap, read_table_column_names: failed to read column header names\n");
      printerror ( status ) ;
    }
    
  }
  
  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return colnames;
  
}

/* ------------------ Writing data --------------------------------------- */

extern int write_ME_FITS_type(char* filename, char* exttype, float* data, 
			  long* nx, long* ny, int nwrite)
{

  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  
  int imsize;                       /* Number of pixels to write out */

  int fpixel=1;                     /* First pixel to write*/
  int i=0;                          /* Loop variable */

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )
    
    printerror ( status ) ;

  /* Write out the data */

  imsize=(*nx)*(*ny);

  /* Loop over the extensions */

   for ( i=0; i<nwrite; i++) {

   
      if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, i+1, &status) ){

	printf("fitswrap - write_ME_FITS_type: Could not move to extension type %s, number %i\n", exttype, i+1);
	printerror( status );

      }

      if ( fits_write_img(fptr, TFLOAT, fpixel, imsize, (data+i*imsize), 
			  &status) ){
	printf("fitswrap - write_ME_FITS_type: Could not write image data %s, number %i\n", exttype, i+1);
	printerror( status );
      }


   }
 
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return 0;

}

extern int write_ME_FITS_type_short(char* filename, char* exttype, 
				    short* data, long* nx, long* ny, 
				    int nwrite)
{

  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  
  int imsize;                       /* Number of pixels to write out */

  int fpixel=1;                     /* First pixel to write*/
  int i=0;                          /* Loop variable */

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )
    
    printerror ( status ) ;

  /* Write out the data */

  imsize=(*nx)*(*ny);

  /* Loop over the extensions */

   for ( i=0; i<nwrite; i++) {

   
      if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, i+1, &status) ){

	printf("fitswrap - write_ME_FITS_type: Could not move to extension type %s, number %i\n", exttype, i+1);
	printerror( status );

      }

      if ( fits_write_img(fptr, TSHORT, fpixel, imsize, (data+i*imsize), 
			  &status) ){
	printf("fitswrap - write_ME_FITS_type_short: Could not write image data %s, number %i\n", exttype, i+1);
	printerror( status );
      }


   }
 
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
      printerror ( status );
  
  return 0;

}

extern int writefits_cirp_ME_FITS(const char *outfile, float* data[], 
				  long* naxes, char* keynames, int numkeys)

{

  fitsfile *fptr;    /* pointer to the FITS file, defined in fitsio.h */
  int status=0;      /* initialize status before calling fitsio routines */
  int  fpixel, nelements;

  int bitpix   =  FLOAT_IMG;  /* 32 bit float pixel values*/
  long naxis=0;               /* Number of axes in image */
  int nextend=4;              /* Number of extensions in output image */  
  char* exts[]={"SCI","VAR","DQ","TIME"}; /* extesions to be written */
  int ver=1;                  /* Extension version */
  int i=0;                    /* Loop variable */

  if (fits_create_file(&fptr, outfile, &status)) /* create new FITS file */
    
    printerror( status );           /* call printerror if error occurs */

  /* write the required keywords for the primary header    */
  
  naxis=0;              /* used for no data in primary header */
  
  if ( fits_create_img(fptr,  bitpix, naxis, NULL , &status) )
         printerror( status );          
  
  if ( fits_update_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );
  
  fpixel = 1;                           /* first pixel to write      */
  nelements = naxes[0] * naxes[1];      /* number of pixels to write */

  /* reserve space in the header for the extra keys if needed */
  
  if ( keynames!=NULL ){

    if ( fits_set_hdrsize (fptr, numkeys, &status) )
      
      printerror ( status ) ;
    
    for ( i=0; i<numkeys; i++){
  
      if ( fits_write_record(fptr, (keynames+FLEN_CARD*i), &status) )
	
	printerror ( status ); 

    }
   

  }
  
  /* write the image data to the FITS file */
  
  naxis=2;
  
  for ( i=0; i<4; i++){

    if ( i==2 ){   /* dq array is short therefore a bit different */

      bitpix = SHORT_IMG;

      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
    
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			 &status) )
	printerror( status );      

      if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
     
      if ( fits_write_img(fptr, TSHORT, fpixel, nelements, data[i], 
			  &status) )
	printerror( status );            
    
    } else {

      bitpix=FLOAT_IMG;

      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
      
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			 &status) )
	printerror( status );      

      if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
     
      if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data[i], 
			  &status) )
	printerror( status );   
    }
  
  }         
  
  if( fits_close_file(fptr, &status) )
    printerror( status );
    
  return 0;

}

extern int write_ME_FITS(const char* outfile, const char* exts[],
			 float* data[], int nextend, long* naxes)
{

  fitsfile *fptr;    /* pointer to the FITS file, defined in fitsio.h */
  int status=0;      /* initialize status before calling fitsio routines */
  int  fpixel;

  int bitpix   =  FLOAT_IMG;  /* 32 bit float pixel values*/
  long naxis=0;               /* Number of axes in image */
  int ver=1;                  /* Extension version */
  int i=0;                    /* Loop variable */
  int nelements =0; 

  if (fits_create_file(&fptr, outfile, &status)) /* create new FITS file */
    
    printerror( status );           /* call printerror if error occurs */

  /* write the required keywords for the primary header    */
  
  naxis=0;              /* used for no data in primary header */
  
  if ( fits_create_img(fptr,  bitpix, naxis, NULL , &status) )
         printerror( status );          
  
  if ( fits_update_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );
  
  fpixel = 1;                           /* first pixel to write      */
  
  
  /* write the image data to the FITS file */
  
  naxis=2;
  nelements = naxes[0] * naxes[1];

  for ( i=0; i<nextend; i++){


      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
      
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", (char*)*(exts+i), NULL, 
	&status) )
	printerror( status );      

      if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
     
      if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, *(data+i), 
	&status) )
	printerror( status );   
  }
  
  
  
  if( fits_close_file(fptr, &status) )
    printerror( status );
    
  return 0;

}


extern int writefits_cirp_ME_FITS_head(const char *outfile, 
				       const char* headerfile, float* data[], 
				       long* naxes)
{

  /************************This routine has not been checked***************/

  /* It sort of works, problems if the header file is the same as the output
   * file
   *
   * Also multiextension files are not dealt with properly, the output file
   * always has as many extensions as the input file 
   */

  /************************************************************************/

  fitsfile *fptr;    /* pointer to the FITS file, defined in fitsio.h */
  int status=0;      /* initialize status before calling fitsio routines */
  int  fpixel, nelements;

  int bitpix   =  FLOAT_IMG;  /* 32 bit float pixel values*/
  long naxis=0;               /* Number of axes in image */
  int nextend=4;              /* Number of extensions in output image */  
  char* exts[]={"SCI","VAR","DQ","TIME"}; /* extesions to be written */
  int ver=1;                  /* Extension version */
  int i=0;

  if (fits_create_template(&fptr, outfile, headerfile, &status)) /* create new                                                                   FITS file */
         printerror( status );           /* call printerror if error occurs */

  /* write the required keywords for the primary header    */
  
  naxis=0;              /* used for no data in primary header */       
  
  if ( fits_update_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );
  
  if ( fits_update_key(fptr, TINT, "NAXIS", &naxis, NULL, &status) )
    printerror( status );

  /*

  if ( fits_create_img(fptr,  bitpix, naxis, NULL , &status) )
         printerror( status );   
  */

  fpixel = 1;                           /* first pixel to write      */
  nelements = naxes[0] * naxes[1];      /* number of pixels to write */

  /* write the image data to the FITS file */

  naxis=2;
  
  for ( i=0; i<4; i++){

    if ( i==2 ){   /* dq array is short therefore a bit different */

      bitpix = SHORT_IMG;

      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
    
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			 &status) )
	printerror( status );      

      if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
     
      if ( fits_write_img(fptr, TSHORT, fpixel, nelements, data[i], 
			  &status) )
	printerror( status );            
    
    } else {

      bitpix=FLOAT_IMG;

      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
      
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			 &status) )
	printerror( status );      

      if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
     
      if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data[i], 
			  &status) )
	printerror( status );   
    }
  
  }         
  
  if( fits_close_file(fptr, &status) )
    printerror( status );
    
  return 0;

}

extern int overwrite_ext(char* filename, float* data, long* naxes, 
			   char* exttype, int* extver)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int bitpix=0;
  int nelements=0;
  int fpixel=0;                        /* First pixel to write */

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )
    printerror ( status ) ;

  if ( fits_movnam_hdu(fptr, ANY_HDU, exttype, *extver, &status) ){

	printf("fitswrap - overwrite_ext: Could not move to extension type %s, number %i\n", exttype, *extver);
	printerror( status );

      }

  if ( fits_read_key(fptr, TINT, "BITPIX", &bitpix, NULL, &status) )
    printerror ( status ) ;

  if ( fits_update_key(fptr, TINT, "NAXIS1", naxes, NULL, &status) )
    printerror( status );

  if ( fits_update_key(fptr, TINT, "NAXIS2", naxes+1, NULL, &status) )
    printerror( status );
  
  nelements=*naxes * *(naxes+1);

  printf("Overwriting %s %i and inserting a  %i bits per pixel image\n", 
	 exttype, *extver, bitpix);

  fpixel=1;
  
  /* TFLOAT below, specifies that data is a float pointer, cfitsio
   * will automatically transform the inserted image data to that of
   * the bitpix keyword in the image header. 
   */
  if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data, 
			  &status) )
      printerror( status );   
     
  if( fits_close_file(fptr, &status) )
    printerror( status );

  return 0;
  
}

extern int insert_table(const char* filename, int pos, int numhead, 
			char* keywords[], float** keydata, int numcol, 
			char* colnames[], int numrows, float** data)
{

  int i=0;                          /* Loop variable */
  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  char** tform=NULL;
  int fieldwidth=0;
  int nextend=0;

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )  
    printerror ( status ) ;
 
  /* Sort out the nextend keyword */
  if ( fits_read_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror ( status ) ;
  
  nextend+=1;
  
  if ( fits_update_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );

  if ( fits_movabs_hdu(fptr, pos, NULL, &status) )
    printerror ( status );
  
  /* Allocate memory */
  if ( (tform=(char **)malloc(numcol*sizeof(char*))) == NULL ){

    printf("fitswrap: insert_table, memory allocation failed!\n");
    exit(1);
  
  }

  fieldwidth=3;
 
  for(i=0; i<numcol; i++){

    if (  (*(tform+i)=(char*)malloc(fieldwidth*sizeof(char)) ) == NULL ){
      printf("fitswrap: insert_table, memory allocation failed!\n");
      exit(1);
    }
    
    strcpy(*(tform+i),"1E\0");
    
  }
  
  if ( fits_insert_btbl(fptr, numrows, numcol, colnames, tform, NULL,
		       NULL, 0, &status) )
    printerror ( status );
  
  for(i=0; i<numhead; i++){
  
    if ( fits_write_key(fptr, TFLOAT, keywords[i], *(keydata+i), NULL, 
			&status) )
      printerror ( status );

  }
  
  for(i=0; i<numcol; i++){

    if ( fits_write_col(fptr, TFLOAT, i+1, 1, 1, numrows, *(data+i), &status) )
      printerror ( status );

  }
  
  if( fits_close_file(fptr, &status) )
    printerror( status );
   
  for(i=0; i<numcol; i++){
    free(*(tform+i));
  }
  free(tform);
  
  return 0;

}

extern int append_table(const char* filename, int pos, int numhead, 
			char* keywords[], float** keydata, int numcol, 
			char* colnames[], int numrows, float** data)
{

  int i=0;                          /* Loop variable */
  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  char** tform=NULL;
  int fieldwidth=0;

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )  
    printerror ( status ) ;
 
  if ( fits_movabs_hdu(fptr, pos, NULL, &status) )
    printerror ( status );
  
  /* Allocate memory */
  if ( (tform=(char **)malloc(numcol*sizeof(char*))) == NULL ){

    printf("fitswrap: append_table, memory allocation failed!\n");
    exit(1);
  
  }

  fieldwidth=3;
 
  for(i=0; i<numcol; i++){

    if (  (*(tform+i)=(char*)malloc(fieldwidth*sizeof(char)) ) == NULL ){
      printf("fitswrap: append_table, memory allocation failed!\n");
      exit(1);
    }
    
    strcpy(*(tform+i),"1E\0");
  
  }
  
  if ( fits_create_tbl(fptr, BINARY_TBL, numrows, numcol, colnames, tform, 
		       NULL, NULL, &status) )
    printerror ( status );
  
  for(i=0; i<numhead; i++){

    if ( fits_write_key(fptr, TFLOAT, keywords[i], *(keydata+i), NULL, 
			&status) )
      printerror ( status );

  }

  for(i=0; i<numcol; i++){

    if ( fits_write_col(fptr, TFLOAT, i+1, 1, 1, numrows, *(data+i), &status) )
      printerror ( status );

  }
 
  if( fits_close_file(fptr, &status) )
    printerror( status );
   
  for(i=0; i<numcol; i++){
    free(*(tform+i));
  }
  free(tform);
  
  return 0;

}

extern int add_table_column(const char* filename, int position, int nadd,
			    char* colnames[], float** coldata, int numrows)
{

  int i=0;                          /* Loop variable */
  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  char** tform=NULL;
  int fieldwidth=0;
  int numcols=0;                    /* Number of columns in the table */
  int hdu_type=0;                   /* Code for header type */

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )  
    printerror ( status ) ;
 
  if ( fits_movabs_hdu(fptr, position, &hdu_type, &status) )
    printerror ( status );
  
  /* Check we've got a table */
  if ( hdu_type != BINARY_TBL ){
    printf("fitswrap: add_table_column, this is not a table extension!!\n");
    exit(1);
  }

  /* Allocate memory */
  if ( (tform=(char **)malloc(nadd*sizeof(char*))) == NULL ){

    printf("fitswrap: add_table_column, memory allocation failed!\n");
    exit(1);  
  }

  fieldwidth=3; /* size of tform format string */
 
  for(i=0; i<nadd; i++){

    if (  (*(tform+i)=(char*)malloc(fieldwidth*sizeof(char)) ) == NULL ){
      printf("fitswrap: add_table_column, memory allocation failed!\n");
      exit(1);
    }
    
    strcpy(*(tform+i),"1E\0");
  
  }
  
  /* Find out how many columns already present */
  if ( fits_get_num_cols(fptr, &numcols, &status) )
    printerror( status );

  /* Append the new columns */
  if ( fits_insert_cols(fptr, numcols+1, nadd, colnames, tform, &status) )
    printerror( status );

  /* Write in the column data */
  for(i=0; i<nadd; i++){
    
    if ( fits_write_col(fptr, TFLOAT, numcols+i+1, 1, 1, numrows, 
			*(coldata+i), &status) )
      printerror ( status );
    
  }
  
  if( fits_close_file(fptr, &status) )
    printerror( status );
   
  for(i=0; i<nadd; i++){
    free(*(tform+i));
  }
  free(tform);
  
  return 0;

}


extern int overwrite_table_column(const char* filename, int position, 
				  int colposition, int nadd, char* colnames[],
				  float** coldata, int numrows)
{

  int i=0;                          /* Loop variable */
  int status=0;                     /* Error handle for cfitsio */
  fitsfile *fptr;                   /* fits object as defined by cfitsio */
  char** tform=NULL;
  int fieldwidth=0;
  int numcols=0;                    /* Number of columns in the table */
  int hdu_type=0;                   /* Code for header type */

  if ( fits_open_file(&fptr, filename, READWRITE, &status) )  
    printerror ( status ) ;
 
  if ( fits_movabs_hdu(fptr, position, &hdu_type, &status) )
    printerror ( status );
  
  /* Check we've got a table */
  if ( hdu_type != BINARY_TBL ){
    printf("fitswrap: add_table_column, this is not a table extension!!\n");
    exit(1);
  }

  /* Allocate memory */
  if ( (tform=(char **)malloc(nadd*sizeof(char*))) == NULL ){

    printf("fitswrap: add_table_column, memory allocation failed!\n");
    exit(1);  
  }

  fieldwidth=3; /* size of tform format string */
 
  for(i=0; i<nadd; i++){

    if (  (*(tform+i)=(char*)malloc(fieldwidth*sizeof(char)) ) == NULL ){
      printf("fitswrap: add_table_column, memory allocation failed!\n");
      exit(1);
    }
    
    strcpy(*(tform+i),"1E\0");
  
  }
  
  /* Find out how many columns already present */
  if ( fits_get_num_cols(fptr, &numcols, &status) )
    printerror( status );

  /* See if we need to insert or overwrite */

  if(numcols<colposition){ /* Need to add new columns */

    /* Append the new columns */
    if ( fits_insert_cols(fptr, numcols+1, nadd, colnames, tform, &status) )
      printerror( status );

    /* Write in the column data */
    for(i=0; i<nadd; i++){
      
      if ( fits_write_col(fptr, TFLOAT, numcols+i+1, 1, 1, numrows, 
			  *(coldata+i), &status) )
	printerror ( status );
      
    }
    
  } else {  /* Overwrite the exisiting columns */

    /* Write in the column data */
    for(i=0; i<nadd; i++){
      
      if ( fits_write_col(fptr, TFLOAT, colposition+i, 1, 1, numrows, 
			  *(coldata+i), &status) )
	printerror ( status );
      
    }
    
  }

  if( fits_close_file(fptr, &status) )
    printerror( status );
   
  for(i=0; i<nadd; i++){
    free(*(tform+i));
  }
  free(tform);
  
  return 0;

}






extern int write_cirp_copy_hdu(char* template_file, char* output_file, 
			       char* exts[], double* data[], int numdata,
			       int naxis, long* naxes)
{

  fitsfile *fptr;    
  fitsfile *fptr_template;
  int status=0;
  int numcols=0;
  int i=0;
  int bitpix=0;
  int ver=0;
  int fpixel=0;
  int nelements=0;
  int nextend=0;

  if( fits_create_file(&fptr, output_file, &status) )
    printerror( status );

  if( fits_open_file(&fptr_template, template_file, READONLY, &status) )
    printerror ( status ); 

  /*if (fits_create_template(&fptr, output_file, template_file, &status))
    printerror( status );      */

  /* Primary header */
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) );
  
  /* Sort out the nextend keyword, assuming we have a table! */
  nextend=numdata+1;
  if ( fits_update_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );

  /* Table header */
  if( fits_movrel_hdu(fptr_template, 1, NULL, &status) )
    printerror( status );
  
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) )
    printerror( status );

  /* Table data */
  if( fits_get_num_cols(fptr_template, &numcols, &status) )
    printerror( status );

  for( i=1; i<=numcols; i++ ){
    if( fits_copy_col(fptr_template, fptr, i, i, 0, &status) )
      printerror( status );
  }

  /* Actual image extensions */
  naxis=2;
  ver=1;
  fpixel=1;
  nelements=1;

  for(i=0; i<naxis; i++){
    nelements*=*(naxes+i);
  }
    
  for ( i=0; i<numdata; i++){

    bitpix=DOUBLE_IMG;
    
    if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
      printerror( status );
    
    if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			 &status) )
      printerror( status );      

    if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
    
    if ( fits_write_img(fptr, TDOUBLE, fpixel, nelements, data[i], 
			  &status) )
      printerror( status );   
     
  }         
    
  if( fits_close_file(fptr, &status) )
    printerror( status );

  if( fits_close_file(fptr_template, &status) )
    printerror( status );

  return 0;

}

extern int write_cirp_copy_hdu_float(char* template_file, char* output_file, 
			       char* exts[], float* data[], int numdata,
			       int naxis, long* naxes)
{

  fitsfile *fptr;    
  fitsfile *fptr_template;
  int status=0;
  int numcols=0;
  int i=0;
  int bitpix=0;
  int ver=0;
  int fpixel=0;
  int nelements=0;
  int nextend=0;

  if( fits_create_file(&fptr, output_file, &status) )
    printerror( status );

  if( fits_open_file(&fptr_template, template_file, READONLY, &status) )
    printerror ( status ); 

  /*if (fits_create_template(&fptr, output_file, template_file, &status))
    printerror( status );      */

  /* Primary header */
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) );

  /* Sort out the nextend keyword, assuming we have a table! */
  nextend=numdata+1;
  if ( fits_update_key(fptr, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );

  /* Table header */
  if( fits_movrel_hdu(fptr_template, 1, NULL, &status) )
    printerror( status );
  
  /* !!!!! CODE ME !!!!! */
  /* Check it's a table, copy if it is, do nothing if not */

  if( fits_copy_hdu(fptr_template, fptr, 0, &status) )
    printerror( status );

  /* Table data */
  if( fits_get_num_cols(fptr_template, &numcols, &status) )
    printerror( status );

  for( i=1; i<=numcols; i++ ){
    if( fits_copy_col(fptr_template, fptr, i, i, 0, &status) )
      printerror( status );
  }

  /* Actual image extensions */
  naxis=2;
  ver=1;
  fpixel=1;
  nelements=1;

  for(i=0; i<naxis; i++){
    nelements*=*(naxes+i);
  }
    
  for ( i=0; i<numdata; i++){

    bitpix=FLOAT_IMG;
    
    if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
      printerror( status );
    
    if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			 &status) )
      printerror( status );      

    if ( fits_update_key(fptr, TINT, "EXTVER", &ver, NULL, &status) )
	printerror( status ); 
    
    if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, data[i], 
			  &status) )
      printerror( status );   
     
  }         
    
  if( fits_close_file(fptr, &status) )
    printerror( status );

  if( fits_close_file(fptr_template, &status) )
    printerror( status );

  return 0;

}

extern int write_cirp_mef(char* template_file, char* output_file, 
			  char* exts[], float* data[], int numdata, int naxis,
			  long* naxes, int next)
{

  fitsfile *fptr;    
  fitsfile *fptr_template;
  int status=0;
  int numcols=0;
  int i=0;
  int j=0;
  int bitpix=0;
  int fpixel=0;
  int nelements=0;
  long nextend=0;

  if( fits_create_file(&fptr, output_file, &status) )
    printerror( status );

  if( fits_open_file(&fptr_template, template_file, READONLY, &status) )
    printerror ( status ); 

  /*if (fits_create_template(&fptr, output_file, template_file, &status))
    printerror( status );      */

  /* Primary header */
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) );
   
  /* Update the keywords, assuming we have a table here! */
  nextend=next*NIMSET+1;
  if( fits_update_key(fptr, TLONG, "NEXTEND", &nextend, NULL, &status) )
      printerror( status );  

  /* Table header */
  if( fits_movrel_hdu(fptr_template, 1, NULL, &status) )
    printerror( status );
  
  /* !!!!! CODE ME !!!!! */
  /* Check it's a table, copy if it is, do nothing if not */

  if( fits_copy_hdu(fptr_template, fptr, 0, &status) )
    printerror( status );

  /* Table data */
  if( fits_get_num_cols(fptr_template, &numcols, &status) )
    printerror( status );

  for( i=1; i<=numcols; i++ ){
    if( fits_copy_col(fptr_template, fptr, i, i, 0, &status) )
      printerror( status );
  }

  /* Actual image extensions */
  naxis=2;
  fpixel=1;
  nelements=1;

  for(i=0; i<naxis; i++){
    nelements*=*(naxes+i);
  }
    

  for(j=1; j<=next; j++){

    for (i=0; i<numdata; i++){
      
      bitpix=FLOAT_IMG;
      /* printf("Writing %s %i\n",exts[i],j); */
      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
      
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			   &status) )
	printerror( status );      
      
      if ( fits_update_key(fptr, TINT, "EXTVER", &j, NULL, &status) )
	printerror( status ); 
      
      if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, 
			  (*(data+i)+nelements*(j-1)), &status) )
	printerror( status );   
      
    }         
    
  }

  if( fits_close_file(fptr, &status) )
    printerror( status );

  if( fits_close_file(fptr_template, &status) )
    printerror( status );

  return 0;

}

extern int write_cirp_ms_mef(char* template_file, char* output_file, 
			     char* exts[], float* data[], int numdata, 
			     int naxis, long* naxes, int next)
{

  fitsfile *fptr;    
  fitsfile *fptr_template;
  int status=0;
  int numcols=0;
  int i=0;
  int j=0;
  int bitpix=0;
  int fpixel=0;
  int nelements=0;
  long nextend=0;

  if( fits_create_file(&fptr, output_file, &status) )
    printerror( status );

  if( fits_open_file(&fptr_template, template_file, READONLY, &status) )
    printerror ( status ); 

  /*if (fits_create_template(&fptr, output_file, template_file, &status))
    printerror( status );      */

  /* Primary header */
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) );
   
  /* Update the keywords, assuming we have a table here! */
  nextend=next*MSNIMSET+1;
  if( fits_update_key(fptr, TLONG, "NEXTEND", &nextend, NULL, &status) )
      printerror( status );  

  /* Table header */
  if( fits_movrel_hdu(fptr_template, 1, NULL, &status) )
    printerror( status );
  
  /* !!!!! CODE ME !!!!! */
  /* Check it's a table, copy if it is, do nothing if not */

  if( fits_copy_hdu(fptr_template, fptr, 0, &status) )
    printerror( status );

  /* Table data */
  if( fits_get_num_cols(fptr_template, &numcols, &status) )
    printerror( status );

  for( i=1; i<=numcols; i++ ){
    if( fits_copy_col(fptr_template, fptr, i, i, 0, &status) )
      printerror( status );
  }

  /* Actual image extensions */
  naxis=2;
  fpixel=1;
  nelements=1;

  for(i=0; i<naxis; i++){
    nelements*=*(naxes+i);
  }
    

  for(j=1; j<=next; j++){

    for (i=0; i<numdata; i++){
      
      bitpix=FLOAT_IMG;
      /* printf("Writing %s %i\n",exts[i],j); */
      if ( fits_create_img(fptr, bitpix, naxis, naxes, &status) )
	printerror( status );
      
      if ( fits_update_key(fptr, TSTRING, "EXTNAME", exts[i], NULL, 
			   &status) )
	printerror( status );      
      
      if ( fits_update_key(fptr, TINT, "EXTVER", &j, NULL, &status) )
	printerror( status ); 
      
      if ( fits_write_img(fptr, TFLOAT, fpixel, nelements, 
			  (*(data+i)+nelements*(j-1)), &status) )
	printerror( status );   
      
    }         
    
  }

  if( fits_close_file(fptr, &status) )
    printerror( status );

  if( fits_close_file(fptr_template, &status) )
    printerror( status );

  return 0;

}

/* ------------------ Utility routines ----------------------------------- */

void printerror(int status)
{
  if (status)
    { 
      fits_report_error(stderr, status);
      exit( status );
    }
  return;
}

extern int read_float_keys(char* p_fname, char* keynames[], float* data, 
			   int numtoread)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */
 
  if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
    printerror ( status ) ;

  for(i=0; i<numtoread; i++){

    if ( fits_read_key(fptr, TFLOAT, keynames[i], data+i, NULL, &status) )
      
      printerror ( status ) ; 

  }


  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;


}

extern int read_int_keys(char* p_fname, char* keynames[], int* data, 
			   int numtoread)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */
 
  if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
    printerror ( status ) ;
  
  for(i=0; i<numtoread; i++){

    if ( fits_read_key(fptr, TINT, keynames[i], data+i, NULL, &status) )
      
      printerror ( status ) ; 
   
  }


  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;


}

extern char* get_key_names(const char* p_fname, int* numfound)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int keynum=0;                        /* header keyword index */
  int i=0;                             /* Loop variable */
  char* keynames=NULL;                 /* Array for the keynames */
 
  if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
    printerror ( status ) ;

  /* Get info from header */
  
  if ( fits_get_hdrspace(fptr, numfound, NULL, &status) )
    
    printerror ( status ) ;
  
  /* Allocate memory for the keynames */
  
  if (  (keynames=(char *)malloc(*(numfound)*FLEN_CARD*sizeof(char))) == NULL ){
    
    printf("get_key_names: memory allocation for key name array failed\n");
    exit(1);

  }
  
  /* Move to the first header keyword */
  
  keynum = 0;
  
  if ( fits_read_record(fptr, keynum, keynames, &status) )

    printerror ( status );
  
  /* Loop over and read all the keys */
  
  for ( i=0; i<*(numfound); i++){
  
    if ( fits_read_record(fptr, i+1, (keynames+FLEN_CARD*i), &status) )

      printerror ( status );
      
    /*    printf("%s\n", (keynames+FLEN_CARD*i)); */
    
  
  }
  
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return keynames;

}

extern int addkeys_int(const char* p_fname, char* keynames[], int* keydata,
		   int numkeys, int headernum)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, p_fname, READWRITE, &status) )  
    printerror ( status ) ;

  /* Move to the required header (using 0 based index) */
  if ( fits_movabs_hdu(fptr, headernum+1, NULL, &status) )
    printerror ( status ) ;
  
  /* Loop over and write all the keys */
  for ( i=0; i<numkeys; i++){
        
    if ( fits_write_key(fptr, TINT, keynames[i], keydata+i, NULL, &status) )
      printerror ( status ); 
    
  }
   
  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}

extern int addkeys_double(const char* p_fname, char* keynames[], 
			  double* keydata, int numkeys, int headernum)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, p_fname, READWRITE, &status) )  
    printerror ( status ) ;

  /* Move to the required header (using 0 based index) */
  if ( fits_movabs_hdu(fptr, headernum+1, NULL, &status) )
    printerror ( status ) ;

  /* Loop over and write all the keys */
  for ( i=0; i<numkeys; i++){
        
    if ( fits_write_key(fptr, TDOUBLE, keynames[i], keydata+i, NULL, &status) )
      printerror ( status ); 
    
  }
   
  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}

extern int addkeys_float(const char* p_fname, char* keynames[], float* keydata,
			 int numkeys, int headernum)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, p_fname, READWRITE, &status) )  
    printerror ( status ) ;

  /* Move to the required header (using 0 based index) */
  if ( fits_movabs_hdu(fptr, headernum+1, NULL, &status) )
    printerror ( status ) ;

  /* Loop over and write all the keys */
  for ( i=0; i<numkeys; i++){
        
    if ( fits_write_key(fptr, TFLOAT, keynames[i], keydata+i, NULL, &status) )
      printerror ( status ); 
    
  }
   
  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}

extern int addkeys_string(const char* p_fname, char* keynames[], 
			  char* keydata[], int numkeys, int headernum)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */

  if ( fits_open_file(&fptr, p_fname, READWRITE, &status) )  
    printerror ( status ) ;

  /* Move to the required header (using 0 based index) */
  if ( fits_movabs_hdu(fptr, headernum+1, NULL, &status) )
    printerror ( status ) ;
  
  /* Loop over and write all the keys */
  for ( i=0; i<numkeys; i++){
        
    if ( fits_write_key(fptr, TSTRING, keynames[i], keydata[i], NULL, 
			&status) )
      printerror ( status ); 
    
  }
   
  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}


extern int write_key_names(const char* p_fname, char* keynames, int numkeys)
{
  
  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */
  int keysalready=0;
  int morekeys=0;

  if ( fits_open_file(&fptr, p_fname, READWRITE, &status) )
    
    printerror ( status ) ;

  /* Move to the first header keyword */
  
  /* see if need to do this!!!!!!! */
  
  /* Loop over and write all the keys */
  
  printf("numkeys is %i\n",numkeys);
  
  if ( fits_get_hdrspace(fptr, &keysalready, &morekeys, &status) )

    printerror (status ) ;

  if ( numkeys > morekeys ){

    printf("fitswrap, write_key_names: Trying to add in %i keys when there is only space for %i\n", numkeys, morekeys);
    exit(1);

  }

  printf("numkeys is %i, space is %i\n",numkeys,morekeys);
  for ( i=0; i<numkeys; i++){
  
    if ( fits_write_record(fptr, (keynames+FLEN_CARD*i), &status) )

      printerror ( status ); 

    /*  printf("Wrote record %i which is %s\n",i,keynames+FLEN_CARD*i);  */

  }
   
  /* Close the file freeing memory allocated for fitsfile fptr */

  if ( fits_close_file(fptr, &status) )
    printerror ( status );

  return 0;

}

extern int datatype(int* bitpix, int* dtype)
{

  if(*bitpix==8){
    *dtype=TBYTE;
  } else if (*bitpix==16) {
    *dtype=TSHORT;
  } else if (*bitpix==32) {
    *dtype=TLONG;
  } else if (*bitpix==-32) {
    *dtype=TFLOAT;
  } else if (*bitpix==-64) {
    *dtype=TDOUBLE;
  } else {
    printf("fitswrap, datatype: unknown value for bitpix!!\n");
    exit(1);
  }

  return 0;
 
}
     




/* ------------------ Static Functions ----------------------------------- */

void parseheader(fitsfile* fptr, long* nx, long* ny, int* bitpix, int status)
{

 char comment[FLEN_COMMENT];          /* Fits header comment */
 
 if ( fits_read_key(fptr, TLONG, "NAXIS1", nx, &comment[0], &status) )
      
   printerror ( status ) ; 

 if ( fits_read_key(fptr, TLONG, "NAXIS2", ny, &comment[0], &status) )
      
   printerror ( status ) ;
 
 if ( fits_read_key(fptr, TINT, "BITPIX", bitpix, &comment[0], &status) )
      
   printerror ( status ) ;

}



 








