/* plotpix.c -- plots the pixel as it integrates from
** a stacked cirpass data cube,
**
** Andrew J. Dean. 4th Aug, 2000.   
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>         /* for sqrt function */
#include "fitswrap.h"     /* for printerror and cfitsio */
#include "cpgplot.h"      /* for the graphics output */

/* Start of main code */

int main(int argc, char* argv[])
{

  /* Define parameters for fitsio */
  int status=0;                  /* Error handle for cfitsio */
  fitsfile *fptr;                /* fits input object as defined by cfitsio */
 
  int fpixel=1;                  /* First pixel to read */
  int nread=0;                   /* Number of pixels to read */     
  int nullvall=0;                /* Markers for bad pixels */
  int anynull=0; 
  char comment[FLEN_COMMENT];    /* Fits header comment */
  
  /* Local variables */
  float* data;                   /* Big array for whole image data */
  float* sci;                    /* Fitting arrays for cosmic ray routines */
  float* time;                   /* and data quality */
	
  int xpix=0;                    /* xpixel to examine */
  int ypix=0;                    /* ypixel to examine */

  int nx=0;                         /* Dimensions of image */
  int ny=0;
  int nloops=0;                     /* Number of loops in exposure */
  int nreads=0;                     /* Number of reads in each loop */
  int nsci=0;                       /* Number of science extensions in file */
  int nbytes;                       /* bytes needed for data */
  int i=0;                          /* Loop variables */
  int j=0;
  int k=0;
  short nsamp=0;                      

  FILE *dataoutfile; 

  /* Check for appropriate input */
  
  if (argc < 5) {
    fprintf(stderr, 
            "Usage: %s scidata.fits outfile xpix ypix\n", 
            argv[0]);
    exit(1);
  }
 
  sscanf(argv[3],"%i",&xpix);
  sscanf(argv[4],"%i",&ypix);   

  /* Open the input file */
  if ( fits_open_file(&fptr, argv[1], READONLY, &status) ){
    printf("Failed to open file\n");
    printerror ( status ) ;
  }

  /* Read dimensions of image, nx and ny, and find number of loops and reads */
  if ( fits_read_key(fptr, TINT, "NLOOPS", &nloops, &comment[0], &status) )
      
    printerror ( status ) ; 

  if ( fits_read_key(fptr, TINT, "NREADS", &nreads, &comment[0], &status) )
      
    printerror ( status ) ;

  
  /* Move to the first science extension to get the image dimensions */
  fits_movnam_hdu(fptr, ANY_HDU, "sci", 1, &status);
 
  if ( fits_read_key(fptr, TINT, "NAXIS1", &nx, &comment[0], &status) )
      
    printerror ( status ) ; 

  if ( fits_read_key(fptr, TINT, "NAXIS2", &ny, &comment[0], &status) )
      
    printerror ( status ) ; 
  
  printf("nx is %i, ny is %i, nloops is %i, nreads is %i \n", nx, ny, 
	   nloops, nreads);

  nsci=nloops*nreads; /* Number of science extensions in image */
  
  if ( nsci <=1 ){
    printf( "plotpix: Not enough data to make a sensible plot!\n" );
    exit(1);
  }

  nread=nx*ny; /* Number of pixels to read in */
  
  nbytes=nx*ny*nsci*sizeof(float); /* Memory needed for the data array */

 /* Allocate memory for local fitting arrays */
  if ( (sci=(float *)malloc(nsci*sizeof(float))) == NULL ){

    printf("cosmicray: Memory allocation for sci fitting arrays failed!");
    exit(1);
  }

  if ( (time=(float *)malloc(nsci*sizeof(float))) == NULL ){

    printf("cosmicray: Memory allocation for time fitting arrays failed!");
    exit(1);
  }
  
  /* Use lots of memory and read the data in! */
  if ( (data=(float *)malloc(nbytes)) == NULL ){

   printf("cosmicray: Memory allocation for data failed!");
   exit(1);
  }
  
  fpixel=1;
  nsamp=nsci-1;

  /* Open the output file for writing */
  
  if ( !(dataoutfile = fopen(argv[2],"w")) ){
    
    printf("Failed to open output file for writing\n");
    exit(1);
    
  }
  
  /* Loop over the science extensions in the input data and
  ** place into float data array
  */
  for(i=0;i<nsci;i++){

    /* Read in the science data, this is in datanumbers */

    fits_movnam_hdu(fptr, ANY_HDU, "sci", i+1, &status);
    
    if ( fits_read_img(fptr, TFLOAT, fpixel, nread, &nullvall, 
		       &data[i*nread], &anynull, &status) )

    printerror( status );

    /* Read in the times of the science data, use nsamp to reorder
    ** to count up 
    */

    if ( fits_read_key(fptr, TFLOAT, "SAMPTIME", &time[nsamp], 
		       &comment[0], &status) )
      
     printerror ( status ) ; 
    
    /* PixCel outputs the time in milliseconds so convert to seconds */
    time[nsamp]/=1000;
       
    nsamp--;

  }

  /* Loop over image array, assembling the data
  **
  */
 
  i=ypix-1;
  j=xpix-1;

  /*
  for( i=0; i<ny; i++ ){
    
    for ( j=0; j<nx; j++ ){
  */
      /* create fitting arrays for this pixel reordering to integrate up*/
      
      nsamp=0;

      /* Fill the fitting arrays with all the data, reordering to integrate up
      */

      for ( k=nsci-1; k>=0; k-- ){

	sci[nsamp]=data[k*nx*ny + i*nx + j]; 
	                                          /* indexing is zero based */
	
	/* Write out a file containing the x and y pixel data asked for */

	/* if (i==(ypix-1) && (j==xpix-1)){

	 
	  printf("pixel %i,%i in science extension %i has sci %f and time %f.\n",xpix,ypix,nsci-nsamp,sci[nsamp],time[nsamp]);
	  
	  fprintf(dataoutfile,"%f\t%f\n",sci[nsamp],time[nsamp]); 
	
	  }	
	 
	*/
	
	nsamp++;
 
      }

      /* Need to avoid reads below and including the zeroth read so move all
      ** values down the appropriate amount 
      */

      nsamp=nsci-nreads;
   
      for ( k=0; k<nsamp; k++ ){
	
	if ((i==ypix-1) && (j==xpix-1)){
	  
	  sci[k]=sci[k+nreads];
	  time[k]=time[k+nreads];  

	  /*  printf("pixel %i,%i in science extension %i has sci %f and time %f.\n",xpix,ypix,nsci-k-nreads,sci[k],time[k]);
	 	
	  fprintf(dataoutfile,"%f\t%f\n",sci[k],time[k]);   
	  */
	}
	
      }
     
      /* Reorder again to only use the last read of the loop */

      nsamp = nsamp/nreads;
   
      for( k=0; k<nsamp; k++ ){
	
	if ((i==ypix-1) && (j==xpix-1)){

	  sci[k]=sci[k*nreads+nreads-1];
	  time[k]=time[k*nreads+nreads-1];

	printf("pixel %i,%i in science extension %i has time %f and sci %f.\n",xpix,ypix,1+k*nreads,time[k],sci[k]);
	 	
	fprintf(dataoutfile,"%f\t%f\n",time[k],sci[k]); 
	
	}

      }

      if  ((i==ypix-1) && (j==xpix-1)){

	/* plot the output on screen */

	/* Open graphics device. */
  
	if(cpgbeg(0, "/xwindow", 1, 1) != 1)
	  return 1; 
  
	/* Define coordinate range of graph                       
	 * (0 < x < max time, 0 < y < max val), and draw axes. 
	 */
  
	cpgenv(time[0], time[nsamp-1], sci[0], sci[nsamp-1], 0, 0);
  
	/* Label the axes (note use of \\u and \\d for raising exponent). */
  
	cpglab("Time (s)", "Counts", "PGPLOT Graph: Pixel integration with time");
	
	/*  Plot the line graph. */
   
	cpgline(nsamp, time, sci);
  
	/* Close the graphics device */
  
	cpgend();

      }
      /*
    }

  }
      */
 
  /* Free memory allocated locally */
  
  free (sci);
  free (time);
    
  if ( fits_close_file(fptr, &status) )
    printerror( status );
   
  fclose(dataoutfile); 
  
  free (data);
  
  /* successful return */

  return 0;

}






