
/* badpix.c - produce a image with badpix set to BADPIX all others 
** to zero.
**
** The input flatfield image is divided by its mode (normalized) to produce
** a gain map.  A pixel is considered bad if the pixel gain < MINGAIN 
** or gain > MAXGAIN or gain > NSIG sigma deviant from the local background 
** in the gain map.
**/

#include <stdio.h>
#include <stdlib.h>
#include "fitswrap.h"
#include "median.h"
#include "grid.h"

#define MINGAIN 0.7    /* pixels with sensitivity < MINGAIN are assumed bad */
#define MAXGAIN 1.3    /* pixels with sensitivity > MAXGAIN are assumed bad */

#define NSIG 5.0    /* pixels bad if sensitivity > NSIG sigma from local bkg */

#define NXBLOCK 16     /* image size should be multiple of block size */
#define NYBLOCK 16

static float buf[NXBLOCK * NYBLOCK];

int main(int argc, char *argv[])
{
    int i, j, k, l, nx, ny, nbad = 0;
    float scale, mode, med, sig, lo, hi;
    float *bpm, *gain, nsig;
    float *flat;

    if (argc < 3) {
        fprintf(stderr, "Usage: %s flat.fits gainmap.fits [nsig]\n", argv[0]);
        exit(1);
    }

    nsig = (argc == 4) ? (nsig = (float)atof(argv[3])) : NSIG;

    flat = readfits(argv[1], 0);

    mode = bkgcalc(flat, nx, ny, -1, -1, NULL);

    if (mode <= 0.0) {
        fprintf(stderr, "%s: mode <= 0\n", argv[0]);
        exit(1);
    }

    if ((gain = (float *)malloc(nx * ny * sizeof(float))) == NULL) {
        fprintf(stderr, "gainmap: malloc fail\n");
        exit(1);
    }

    if ((bpm = (float *)malloc(nx * ny * sizeof(float))) == NULL) {
        fprintf(stderr, "gainmap: malloc fail\n");
        exit(1);
    }

    scale = 1.0 / mode;

    for (i = 0; i < nx * ny; i++) {
        gain[i] = (float)flat[i] * scale;         /* scale flatfield by mode */

        if (gain[i] < MINGAIN || gain[i] > MAXGAIN) {      /* bad pixel? */
            gain[i] = 0.0;
            nbad++;
        }
    }

    for (i = 0; i < ny; i += NYBLOCK) {
        for (j = 0; j < nx; j += NXBLOCK) {          /* foreach image block */
            int n = 0;

            for (k = i; k < i + NYBLOCK; k++)
                for (l = j; l < j + NXBLOCK; l++)
                    if (gain[k*nx+l] > 0.0)             /* if good pixel */
                        buf[n++] = gain[k*nx+l];

            med = (n > 0) ? median_float(buf, n) : 0.0;   /* calc. median */
              
            for (k = i; k < i + NYBLOCK; k++)
                for (l = j; l < j + NXBLOCK; l++)
                    bpm[k*nx+l] = gain[k*nx+l] - med;    /* subtract median */
        }
    }

    med = median_float(bpm, nx * ny);
    sig = 1.5 * median_absdev(bpm, med, nx * ny);
    lo  = med - nsig * sig;
    hi  = med + nsig * sig;

    for (i = 0; i < nx * ny; i++)
        if (bpm[i] < lo || bpm[i] > hi) {
            gain[i] = 0.0;
            nbad++;
        }

    writefits_copyhdr_float(argv[2], argv[1], gain);

    put_keyword(argv[2], "IRDR_BKG", 1.0);
    put_keyword(argv[2], "IRDR_SIG", sig);

    fprintf(stderr, "\n%s processed: %s\n", argv[0], argv[2]);
    fprintf(stderr, "  bad pixel count: %d\n", nbad);
    fprintf(stderr, "  sigma threshold: %f\n", nsig);
    fprintf(stderr, "  min gain threshold: %f\n", MINGAIN);
    fprintf(stderr, "  max gain threshold: %f\n", MAXGAIN);
    fprintf(stderr, "  block size in x: %d\n", NXBLOCK);
    fprintf(stderr, "  block size in y: %d\n", NYBLOCK);

    return 0;
}

