/* cosmicray.c -- finds cosmic rays in a stacked cirpass data cube,
** removes them and produces an output cosmic ray cleaned image,
** and a pixel list of cosmic ray hits.
**
** fitsamps and actual cosmic ray rejection algorithm taken from
** stsdas nicmos data reduction n_cridcalc.c
** (locally: /data/star/iraf/extern/stsdas/pkg/hst_calib/nicmos/calnica)
**
** Andrew J. Dean. 7th June, 2000.   Version 1
**
*/

/* *************  Required fixes / known problems ************************/

/* Hot pixels can sometimes get flagged as CR Hits if they saturate.
** Integration slope reduces with time, so first integration can seem
** abnormally large (depending on sigma), so it gets flagged as a CR hit
** which usually results in subsequent results been flagged too
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>       /* for strcat */
#include <math.h>         /* for sqrt function */
#include "cirpass.h"      /* for CR_HIT definitions e.t.c */
#include "cosmicray.h"
#include "fitswrap.h"     /* for printerror and cfitsio */

/* Static funcion definitions */
static void fitsamps(short nsamp, float *sci, float *err, short *dq, 
		     float *time, float thresh, float *out_sci, 
		     float *out_sci_final, float *out_err, short *out_samp, 
		     float *out_time, short i, short j);

static void linfit (float *x, float *y, float *sig, short *mask, short ndata, 
		    float *a, float *b, float *siga, float *sigb);

static void RejSpikes (float *tsci, float *terr, short *dq, float *diff, 
		       short nsamp, float thresh, int *nrej);

static void RejCRs (short *dq, float *diff, short nsamp, float thresh,
		    int *nrej);

/* Start of main code */

extern int cosmicray(const char* p_fname, const char* p_outfname, float thresh)
{

  /* Define parameters for fitsio */
  int status=0;                  /* Error handle for cfitsio */
  fitsfile *fptr;                /* fits input object as defined by cfitsio */
 
  int fpixel=1;                  /* First pixel to read */
  int ntoread=0;                 /* Number of pixels to read */     
  int nullvall=0;                /* Markers for bad pixels */
  int anynull=0; 
  char comment[FLEN_COMMENT];    /* Fits header comment */
  
                                 /* output file parameters */
  long axes[2];                  /* contains image dimensions */

  int numfound=0;
  int keynum=0;
  char* keynames=NULL;

  /* Local variables */
  float* data;           /* Big array for whole image data */
  float* errors;         /* Big array for whole error image data */
  short* dqdata;         /* Big array for whole dq data */
  float* sci;            /* Fitting arrays for cosmic ray routines */
  float* err;            /* For science, error, time */
  float* time;           /* and data quality */
  short* dq;
  float* cntspersec;     /* Output cnts/s cosmic rejected image */
  float* errout;         /* Output error data */
  float* timeout;        /* Output time data */
  short* dqout;          /* Array for output dq image */ 
  float* outexts[4];     /* contains pointers to output data (sci, err ... ) */
  int    hits=0;         /* Number of cosmic ray hits found */
   
  int nx=0;                         /* Dimensions of image */
  int ny=0;
  /* int nloops=0;   */             /* Number of loops in exposure */
  /* int nreads=0;   */             /* Number of reads in each loop */
  int nsci=0;                       /* Number of science extensions in file */
  int nextend=0;                    /* Number of extensions in image */
  int nimset=NIMSET;                /* Number of images per read */
  int nbytes;                       /* bytes needed for data */
  int i=0;                          /* Loop variables */
  int j=0;
  int k=0;
  int ii=0;
  short nsamp=0;                      

  float out_sci;                    /* output science value (slope of fit)*/
  float out_sci_final;              /* output sci value (final point of fit)*/
  float out_err;                    /* output error value */
  short out_dq;                     /* output data quality value */   
  short out_samp;                   /* output samp value */ 
  float out_time;                   /* output time value */    

  int return_slope=1;     /* Return fitted slope (1), or last point (0) */

  /* Variables for the table data */
  float** tabledata=NULL;
  char** colnames=NULL;
  char* keywords[]={"YOVERX","MAXDIST"};
  float yoverx=0;
  float maxdist=0;
  float* keydata[2]; 
  int pos=2;            /* Position of table in file */
  int numhead=2;
  int numcols;          /* Number of cols + rows in table extension */
  int numrows;


  /* Debug variables */  

# if defined(DEBUG_COSMIC)

  long debug_axes[2];
  long* debug_naxes;
 
  char* tempout;
  char fred[80];
  tempout=&fred[0];
  
  debug_naxes=&debug_axes[0];      
  debug_naxes[0]=debug_naxes[1]=512;
 
# endif /* DEBUG_COSMIC */

  /* Open the input file */
  if ( fits_open_file(&fptr, p_fname, READONLY, &status) )
    
    printerror ( status ) ;


  /* Read in the header keywords so they can be copied to the output file 
   * This is basically fitswrap:get_key_names
   */   
  if ( fits_get_hdrspace(fptr, &numfound, NULL, &status) )
    
    printerror ( status ) ;
  
  /* Allocate memory for the keynames */
  if (  (keynames=(char *)malloc(numfound*FLEN_CARD*sizeof(char))) == NULL ){
    
    printf("cosmicray: memory allocation for key name array failed\n");
    exit(1);

  }
 
  /* Move to the first header keyword */
  keynum = 0;
  
  if ( fits_read_record(fptr, keynum, keynames, &status) )

    printerror ( status );
  
  /* Loop over and read all the keys */
  for ( i=0; i<numfound; i++){
  
    if ( fits_read_record(fptr, i+1, (keynames+FLEN_CARD*i), &status) )
      
      printerror ( status );
    /*printf("%s\n", (keynames+FLEN_CARD*i)); */

  }
  
  if ( fits_read_key(fptr, TINT, "NEXTEND", &nextend, &comment[0], &status) )
      
    printerror ( status ) ;

  
  /* Move to the first science extension to get the image dimensions */
  fits_movnam_hdu(fptr, ANY_HDU, "sci", 1, &status);
 
  if ( fits_read_key(fptr, TINT, "NAXIS1", &nx, &comment[0], &status) )
      
    printerror ( status ) ; 

  if ( fits_read_key(fptr, TINT, "NAXIS2", &ny, &comment[0], &status) )
      
    printerror ( status ) ; 
  
  /* nx=DETECTOR_X;
     ny=DETECTOR_Y; */

# if defined(DEBUG_COSMIC)

    printf("nx is %i, ny is %i \n", nx, ny);

# endif /* DEBUG_COSMIC */

  nsci=nextend/nimset; /* Number of science extensions in image */
 
  if ( nsci <=1 ){
    printf( "cosmic: Not enough data to run the cosmic ray extraction!\n" );
    exit(1);
  }

  ntoread=nx*ny; /* Number of pixels to read in */
  
  nbytes=nx*ny*nsci*sizeof(float); /* Memory needed for the data arrays */

  /* Allocate lots of memory for input data arrays! */
  if ( (data=(float *)malloc(nbytes))    == NULL ||
       (errors=(float *)malloc(nbytes))  == NULL ||
       (dqdata=(short *)malloc(nbytes))  == NULL ){
    
   printf("cosmicray: Memory allocation for input data arrays failed\n");
   exit(1);
  }

  /* Allocate memory for ouput data arrays */
  if ( (cntspersec=(float *)malloc(ntoread*sizeof(float))) == NULL ||
       (errout=(float *)malloc(ntoread*sizeof(float)))     == NULL ||
       (dqout=(short *)malloc(ntoread*sizeof(short)))      == NULL ||
       (timeout=(float *)malloc(ntoread*sizeof(float)))    == NULL ){

    printf("cosmicray: Memory allocation for output arrays failes!\n");
    exit(1);
  
  }

  /* Allocate memory for local fitting arrays */
  if ( (sci=(float *)malloc(nsci*sizeof(float)))  == NULL || 
       (err=(float *)malloc(nsci*sizeof(float)))  == NULL ||
       (time=(float *)malloc(nsci*sizeof(float))) == NULL ||
       (dq=(short *)malloc(nsci*sizeof(short)))   == NULL ){

    printf("cosmicray: Memory allocation for fitting arrays failed!\n");
    exit(1);
  
  }
   
  fpixel=1;
  nsamp=nsci-1;

  /* Loop over the extensions in the input fits file and
  ** place data into appropriate arrays
  */
  for(i=0;i<nsci;i++){

    /* Read in the science data, this is in datanumbers */
    fits_movnam_hdu(fptr, ANY_HDU, "sci", i+1, &status);
    
    if ( fits_read_img(fptr, TFLOAT, fpixel, ntoread, &nullvall, 
		       &data[i*ntoread], &anynull, &status) )

    printerror( status );

    /* Read in the times of the science data, use nsamp to reorder
    ** to count up */
    if ( fits_read_key(fptr, TFLOAT, "SAMPTIME", &time[nsamp], 
		       &comment[0], &status) )
      
      printerror ( status ) ; 

    /* PixCel outputs the time in milliseconds so convert to seconds */
    time[nsamp]/=1000;

    /* Read in the error (var) data, this is in datanumbers (squared) */
    fits_movnam_hdu(fptr, ANY_HDU, "VAR", i+1, &status);
    
    if ( fits_read_img(fptr, TFLOAT, fpixel, ntoread, &nullvall, 
		       &errors[i*ntoread], &anynull, &status) )

      printerror( status );

    /* Read in the data quality data */
    fits_movnam_hdu(fptr, ANY_HDU, "DQ", i+1, &status);
    
    if ( fits_read_img(fptr, TSHORT, fpixel, ntoread, &nullvall, 
		       &dqdata[i*ntoread], &anynull, &status) )

      printerror( status );

     
# if defined(DEBUG_COSMIC)  
# if defined(DEBUG_COSMIC_LOTS)

    sprintf(tempout,"out%i.fits",i);
    
    writefits_float_basic(tempout,debug_naxes,&data[i*ntoread]);

# endif /* DEBUG_COSMIC_LOTS */
# endif /* DEBUG_COSMIC */

    nsamp--;

  }

  /* Got all the data so close the input file */    
  if ( fits_close_file(fptr, &status) )
    printerror( status );
 
  /* check if user set a cosmic ray rejection threshold
  ** use default if not
  */
  if ( thresh == 0.0 )
    thresh=COSMIC_THRESH;
  printf("cosmicray: Using a cosmic ray rejection threshold of %f sigma\n",
	 thresh);

  /* Initialise output image data quality value */
  out_dq=0;


  /* Loop over image array, computing mean countrate at each pixel and
  ** rejecting cosmic rays
  */
  for( j=0; j<ny; j++ ){
    
    for ( i=0; i<nx; i++ ){

      /* create fitting arrays for this pixel reordering to integrate up*/
      nsamp=0;
 
      /* Fill the fitting arrays with all the data, reordering to integrate up
      */
      for ( k=nsci-1; k>=0; k-- ){

	sci[nsamp]=data[k*nx*ny + j*nx + i]; /* indexing is zero based */
	
	err[nsamp]= sqrt( errors[k*nx*ny + j*nx + i] ); /* have variance */

	dq[nsamp]=dqdata[k*nx*ny + j*nx + i];

# if defined(DEBUG_COSMIC)
	
	if (i==0 && j==0)
	  printf("pixel 1,1 in science extension %i has sci %f and time %f.\n"
		 ,k+1,sci[nsamp],time[nsamp]);

# endif /* DEBUG_COSMIC */    
	
	nsamp++;

      }

      /* Need to avoid reads below and including the zeroth read so move all
      ** values down the appropriate amount, time arrays only need to be fixed
      ** once though
      */
      /* No longer need this. The one in everything below was nreads, 
       * if you stack them all and want to avoid the grot in the other 
       * reads.
       */ 
      /*
      nsamp=nsci-1;  
      
      for ( k=0; k<nsamp; k++ ){

	sci[k]=sci[k+1];
	err[k]=err[k+1];
	dq[k]=dq[k+1];
	if (i==0 && j==0)
	  time[k]=time[k+1];

# if defined(DEBUG_COSMIC) 

	if (i==2 && j==2)
	  printf("pixel 3,3 in science extension %i has sci %f and time %f.\n",nsci-k-nreads,sci[k],time[k]);
    
# endif    

      }
      */
      /* Reorder again to only use the last read of the loop */
      /*
      nsamp=nsamp/nreads;

      for ( k=0; k<nsamp; k++ ){

	sci[k]=sci[k*nreads+nreads-1];
	err[k]=err[k*nreads+nreads-1];
	dq[k]=dq[k*nreads+nreads-1];
	if (i==0 && j==0)
	  time[k]=time[k*nreads+nreads-1];

      }
      */
      /* Do iterative rejection and computation of slope */
      fitsamps (nsamp, sci, err, dq, time, thresh, &out_sci, &out_sci_final, 
		&out_err, &out_samp, &out_time, i, j);
 
      /* Put the science data into the output array */

      if (return_slope==1) { 
	
	cntspersec[j*nx+i]=out_sci;
	
      } else {
	
	cntspersec[j*nx+i]=out_sci_final / out_time;
      
      }
      errout[j*nx+i]=out_err * out_err;  /* want variance again */
      timeout[j*nx+i]=out_time;          /* out_err is error on slope so is */
                                         /* in DN/sec */

      /* Look for pixels with a cosmic ray hit */
      dqout[j*nx+i]=0;
      
      for (ii=0;ii<nsamp;ii++)
	dqout[j*nx+i]= dqout[j*nx+i] | dq[ii]; 

      /* Count the number of cosmic ray hits */
      if ( (dqout[j*nx+i] & CR_HIT) == CR_HIT )
	hits++;

      /* printf("%f  %f\n",out_time,time[nsci-1]); */
      /* printf("dq is %d\n",dq[8]); */
      /* printf("dq is %d\n",dqout[i*nx+j]); */

    }

  }

  /* write out the rejected cosmic ray image */
  axes[0]=nx;
  axes[1]=ny;

  outexts[0]=cntspersec;
  outexts[1]=errout;   
  outexts[2]=(float*)dqout; /* out science is DN / sec */
  outexts[3]=timeout;

  /* don't want the first 7 header keywords as they are only appropriate
   * for the original input image and not the different dimension
   * MEF output image
   */
  
  /* See if we have a table and read it in */
  keydata[0]=&yoverx;
  keydata[1]=&maxdist;

  /* Here rather than after writefits as we need to read the table
   * information before the input file may be overwritten
   */
  tabledata=read_table_check(p_fname, pos, numhead, keywords, keydata, 
			     &numcols,  &numrows);

  if(tabledata!=NULL){
    colnames=read_table_column_names(p_fname, pos, &numcols, &numrows);
  }

  writefits_cirp_ME_FITS(p_outfname, outexts, axes, (keynames+FLEN_CARD*8), 
			 numfound-8);
    
  if(tabledata!=NULL){
    
    /* Don't know why it's pos-1, both insert and read do a mov_abs_hdu, but
     * clearly one absolute is different to another, also other code
     * uses those routines and works so can't change them 
     */

    if( *p_outfname=='!' ){
      
      if( insert_table(p_outfname+1, pos-1, numhead, keywords, keydata, 
		       numcols, colnames, numrows, tabledata) ){
      printf("cosmic: insert table failed\n");
      exit(1);
      }
    
    } else {
      
      if( insert_table(p_outfname, pos-1, numhead, keywords, keydata, numcols,
		       colnames, numrows, tabledata) ){
	printf("cosmic: insert table failed\n");
	exit(1);
      }
    
    }

  }
  
  printf("cosmicray: Found %i pixels with a cosmic ray hit\n",hits);

  /*
  writefits_float_basic("final_samples.fits",&axes[0],cntspersec);
  writefits_float_basic("dnpersec.fits",&axes[0],errout);
  writefits_short_basic("dqarray.fits",axes,dqout);
  */

  /* Free memory allocated locally */
  free (sci);
  free (err);
  free (time);
  free (dq);
  free (dqout);
  free (cntspersec);
  free (timeout);
  free (errout);
  free (data);
  free (errors);
  free (dqdata);
  free (keynames);
  
  for(i=0; i<numcols; i++){
    free( *(tabledata+i) );
    free( *(colnames+i) );
  }
  free(tabledata); 
  free(colnames);
  
  /* successful return */
  return 0;

}

/*********  All following code taken directly from n_cridcalc.c *************/

/* Some modifications:

 * Do not need to mask zeroth read
 * Also returns final point as well as slope of line fit

*/


/* FITSAMPS: Fit accumulating counts vs. time to compute mean countrate,
** iteratively rejecting CR hits and refitting until no new samples are
** rejected. The rejection test is based on the input error value for
** each sample.
*/

static void fitsamps (short nsamp, float *sci, float *err, short *dq,
		      float *time, float thresh, float *out_sci, 
		      float *out_sci_final, float *out_err, short *out_samp, 
		      float *out_time, short i, short j) {

	/* Local variables */
	int k;			/* loop index */
	int nrej;		/* number of rejected samples */
	float *tsci;		/* list of cleaned fluxes */
	float *terr;		/* list of cleaned errors */
	float *ttime;		/* list of cleaned times */
	float bad_sci;		/* sci  value of bad sample(s) */
	float bad_time;		/* time value of bad sample(s) */
	int   bad_samp;		/* number of bad samples */
	float a, siga;		/* linear fit zeropoint and error */
	float b, sigb;		/* linear fit slope and error */
	float *diff;		/* sample differences relative to fit */

	/* Initialize the output values */
	*out_sci  = 0;
	*out_err  = 0;
	*out_samp = 0;
	*out_time = 0;
	*out_sci_final = 0;

	/* Allocate memory for local arrays */
	tsci  = NULL;
	terr  = NULL;
	ttime = NULL;
	diff  = NULL;
	tsci   = (float *) calloc(nsamp, sizeof(float));
	terr   = (float *) calloc(nsamp, sizeof(float));
	ttime  = (float *) calloc(nsamp, sizeof(float));
	diff   = (float *) calloc(nsamp, sizeof(float));

	/* Fit and test samples for this pixel until no new samples
	** are rejected */
	nrej = 1;
	while (nrej) {

		/* Copy non-rejected sample values to tmp arrays */
		bad_samp = 0;
		bad_time = 0;
		bad_sci  = 0;
		for (k = 0; k < nsamp; k++) {

		     /* Temporarily flag zeroth read so it's not used in fit */
		     if (k == 0) {
			 tsci[k] = sci[k];
			 terr[k] = err[k];

			 /* We don't need this, read 1 is not the zeroth read
			 ** therefore removed AJD 
			 dq[k] = dq[k] | BADPIX;
			 */
			 
			 dq[k] = dq[k];
			 ttime[k] = time[k];

		     /* Copy unflagged samples to tmp arrays; if a previous
		     ** sample was flagged, subtract its values from
		     ** accumulated counts and time, but use original errs. */
		     } else if (dq[k] == 0) {
			 if (bad_samp == 0) {
			     tsci[k]  = sci[k];
			     ttime[k] = time[k];
			 } else {
			     tsci[k]  = sci[k]  - bad_sci;
			     ttime[k] = time[k] - bad_time;
			 }
			 terr[k] = err[k];

		     /* Accumulate bad sci and time values for flagged samples,
		     ** but only if they're not flagged as BADPIX; BADPIX is
		     ** used to flag electronic noise which does not affect
		     ** later samples. */
		     } else if (!(dq[k] & BADPIX)) {
			 bad_sci  += sci[k]  - sci[k-1];
			 bad_time += time[k] - time[k-1];
			 bad_samp++;
		     }

		     if (DEBUG && ((i==X1-1 && j==Y1-1)||(i==X2-1 && j==Y2-1))){
		         printf (" time=%g, sci=%g, err=%g, dq=%d\n", ttime[k],
			           tsci[k], terr[k], dq[k]);
		         fflush (stdout);
		     }
		}

		/* Compute mean countrate using linear fit */
		linfit (ttime, tsci, terr, dq, nsamp, &a, &b, &siga, &sigb);
		nrej = 0;

		if (DEBUG && ((i==X1-1 && j==Y1-1)||(i==X2-1 && j==Y2-1))) {
		    printf (" slope=%g, sigma=%g, intercept=%g\n", b,sigb,a);
		    fflush (stdout);
		}

		/* Compute difference of each sample from the fit */
		for (k = 0; k < nsamp; k++) {
		     if (terr[k] != 0 && dq[k] == 0)
			 diff[k] = (tsci[k]-(a+b*ttime[k])) / terr[k];
		     else
			 diff[k] = 0;

		     if (DEBUG && ((i==X1-1 && j==Y1-1)||(i==X2-1 && j==Y2-1))){
			 printf (" diff[%d] = %g\n", k, diff[k]);
			 fflush (stdout);
		     }
		}

		/* Reject electronic noise spikes */
		RejSpikes (tsci, terr, dq, diff, nsamp, thresh, &nrej);

		/* If a spike was found,
		** reiterate the fit before looking for CR hits */
		if (nrej > 0)
		    continue;

		/* Reject CR hits */
		RejCRs (dq, diff, nsamp, thresh, &nrej);

	} /* iterate the fit */

	/* Set output SCI and ERR values */
	*(out_sci) = b;
	*(out_err) = sigb;

	/* Compute output SAMP and TIME values from non-rejected samples */
	for (k = 0; k < nsamp; k++) {
	     if (dq[k] == 0) {
		 (*out_samp)++;
		 (*out_time) = ttime[k];
                 (*out_sci_final) = tsci[k];
	     }
	}

	/* Remove BADPIX flag that was temporarily set for the zeroth read */
	
	/* We're not using this !!! AJD
        if (dq[0] & BADPIX)
	    dq[0] -= BADPIX;
	*/

	/* Free local memory */
	free (tsci);
	free (terr);
	free (ttime);
	free (diff);

}

/* LINFIT: Compute a linear fit of the form y = a + bx to a set of
** data points x, y with individual standard deviations sig. Returned
** are a, b and their respective probable uncertainties siga and sigb.
** Taken from Numerical Recipes (with modifications to handle masked data).
*/

static void linfit (float *x, float *y, float *sig, short *mask, short ndata,
		    float *a, float *b, float *siga, float *sigb) {

	/* Local variables */
	int k;
	int ngood;
	int goodpt=0;
	float ss, sx, sy, st2, t, wt, sxoss;

	/* Initialize accumulators and results */
	ss = 0; sx = 0; sy = 0; st2 = 0;
	*a = 0; *b = 0; *siga = 0; *sigb = 0;

	/* Count number of non-masked points */
	ngood = 0;
	for (k = 0; k < ndata; k++) {
	     if (mask[k] == 0) {
		 goodpt = k;
		 ngood++;
	     }
	}

	/* Check for trivial cases */
	if (ngood == 0)
	    return;
	else if (ngood == 1 && x[goodpt] != 0) {
	    (*b)    = y[goodpt] / x[goodpt];
	    (*sigb) = sig[goodpt] / x[goodpt];
	    return;
	}

	/* Accumulate sums */
	for (k = 0; k < ndata; k++) {
	     if (sig[k] != 0 && mask[k] == 0) {
		 wt = 1.0 / (sig[k]*sig[k]);
		 ss += wt;
		 sx += x[k]*wt;
		 sy += y[k]*wt;
	     }
	}

	if (ss != 0)
	    sxoss = sx / ss;
	else
	    sxoss = 0;

	for (k = 0; k < ndata; k++) {
	     if (sig[k] != 0 && mask[k] == 0) {
		 t = (x[k] - sxoss) / sig[k];
		 st2  += t*t;
		 (*b) += t*y[k]/sig[k];
	     }
	}

	/* Solve for b and sigb */
	if (st2 != 0) {
	    (*b) = (*b) / st2;
	    (*sigb) = sqrt ( 1.0 / st2);
	} else {
	    (*b) = 0;
	    (*sigb) = 0;
	}

	/* Solve for a and siga */
	if (ss != 0 && st2 != 0) {
	    (*a) = (sy - sx*(*b)) / ss;
	    (*siga) = sqrt ( (1.0+sx*sx/(ss*st2)) / ss);
	} else {
	    (*a) = 0;
	    (*siga) = 0;
	}

}

/* REJSPIKES: Find and flag electronic noise spikes. These show up as
** large positive or negative deviations for a single sample relative to
** the samples on either side of it.
*/

static void RejSpikes (float *tsci, float*terr, short *dq, float *diff,
		       short nsamp, float thresh, int *nrej) {

	/* Local variables */
	int   k;		/* loop index */
	float max_diff; 	/* max sample deviation */
	int   max_samp;		/* sample with max deviation */

	max_diff = 0.0; max_samp = 0;

	/* Loop over samples */
	for (k = 1; k < nsamp; k++) {

	     /* Regular samples */
	     if (k < nsamp-1) {
		 if (dq[k-1] == 0 && dq[k] == 0 && dq[k+1] == 0) {

		     /* Check for positive spikes */
		     if (diff[k]-diff[k-1] > thresh &&
			 diff[k]-diff[k+1] > thresh &&
			 tsci[k]-tsci[k-1] > thresh*terr[k-1] &&
			 tsci[k]-tsci[k+1] > thresh*terr[k+1]) {
			 if (tsci[k]-tsci[k-1] > max_diff) {
			     max_diff = tsci[k]-tsci[k-1];
			     max_samp = k;
			 }
			 (*nrej)++;
		     }

		     /* Check for negative spikes */
		     if (diff[k-1]-diff[k] > thresh &&
			 diff[k+1]-diff[k] > thresh &&
			 tsci[k-1]-tsci[k] > thresh*terr[k-1] &&
			 tsci[k+1]-tsci[k] > thresh*terr[k+1]) {
			 if (tsci[k+1]-tsci[k] > max_diff) {
			     max_diff = tsci[k+1]-tsci[k];
			     max_samp = k;
			 }
			 (*nrej)++;
		     }
		 }

	     /* Handle last sample in a special way */
	     } else {
		 if (dq[k-1] == 0 && dq[k] == 0) {

		     /* Check for positive spikes */
		     if (diff[k]-diff[k-1] > thresh &&
			 tsci[k]-tsci[k-1] > thresh*terr[k-1]) {
			 if (tsci[k]-tsci[k-1] > max_diff) {
			     max_diff = tsci[k]-tsci[k-1];
			     max_samp = k;
			 }
			 (*nrej)++;
		     }

		     /* Check for negative spikes */
		     if (diff[k-1]-diff[k] > thresh &&
			 tsci[k-1]-tsci[k] > thresh*terr[k-1]) {
			 if (tsci[k-1]-tsci[k] > max_diff) {
			     max_diff = tsci[k-1]-tsci[k];
			     max_samp = k;
			 }
			 (*nrej)++;
		     }
		 }
	     }
	}

	/* If a spike was found, flag the sample that has the max deviation */
	if (*nrej > 0)
	    dq[max_samp] = dq[max_samp] | BADPIX;

}

/* REJCRS: Find and flag Cosmic Ray hits. This is done by looking for
** large negative-to-positive going changes in the sample values from one
** sample to the next.
*/

static void RejCRs (short *dq, float *diff, short nsamp, float thresh,
		    int *nrej) {

	/* Local variables */
	int k;			/* loop index */
	float prev_diff;	/* last useable difference value */

	prev_diff = 0;

	/* Loop over samples */
	for (k = 1; k < nsamp; k++) {
	     if (dq[k-1] == 0)
		 prev_diff = diff[k-1];
	     if (dq[k] == 0) {
		 if (diff[k]-prev_diff > thresh) { 
		     dq[k] = dq[k] | CR_HIT;
		     prev_diff = diff[k];
		     (*nrej)++;
		 }
	     }
	}
}

