/* linefind.c
**
** Finds the positions of lines on the detector - picks out
** the fibre positions
**
** A. J. Dean, 16th July, 2001.
**
*/

#include <stdlib.h>
#include "fitswrap.h"
#include "mathutils.h"

static int findpeaks(float* section, int* peaklist, int imsize, int* numfound,
		    float back, float thresh);

int main(int argc, char* argv[])
{

  int line=0;
  float noise=0;               /* Noise in image */
  float sigma=0;               /* Sigma away which represents a line */
  float background=0;          /* Background in image */  
  float* data=NULL;            /* Image data */
  float* section=NULL;         /* Cut through image */
  long axes[2]={0};            /* Image dimensions */
  int* peaklist=NULL;          /* List of peaks in image */
  int numfound=0;              /* Number of peaks found */
  float thresh=0;              /* Counts above back which represents a line */
  int linewidth=10;            /* Number of pixel around line to median */
  float* tempdata=0;           /* Line data to be medianed */ 

  int i=0;
  int j=0;

  FILE* outfile;

  /* check for appropriate input */
  if ( argc < 6 ) {
    fprintf(stderr, 
            "Usage: %s image_file line noise sigma output_file\n", 
            argv[0]);
    exit(1);
  }

  sscanf(argv[2],"%i",&line);
  sscanf(argv[3],"%f",&noise); 
  sscanf(argv[4],"%f",&sigma);

  data=readfits_float_clever(argv[1], axes); /* Gives sci,1 from MEF files
						or data from plain FITS */
  
  /* Allocate memory */
  if ( (section=(float *)malloc(axes[1]*sizeof(float))) == NULL || 
       (peaklist=(int *)malloc(axes[1]*sizeof(float))) == NULL  ||
       (tempdata=(float *)malloc(linewidth*sizeof(float))) == NULL ){
    
    printf("linefine: Memory allocation failed!");
    exit(1);
  }

  /* Median around line and copy into section */  
  for(i=0; i<axes[1]; i++){

    for(j=0; j<linewidth; j++){

      *(tempdata+j)=*(data + line - (linewidth/2) + j + i*axes[0]);
    
    }

    *(section + i)=median(tempdata,linewidth);

  }

  /* Median find the background */
  background=median(data, axes[0]*axes[1]);
  /*printf("using a background of %f\n",background);*/
  
  thresh=sigma*noise;
  findpeaks(section, peaklist, axes[1], &numfound, background, thresh);

  if( numfound==0 ){
    printf("peakfind: Didn't find any peaks!!\n");
  }

  /* Write the data out */
  if ( !(outfile = fopen(argv[5],"w")) ){
    printf("linefind: Failed to open the output file\n");
    exit(1);
  }

  for(i=0; i<numfound; i++){

    fprintf(outfile,"%i 1.000\n",*(peaklist+i));

  }
  
  fclose(outfile); 

  free(section);
  free(data);
  free(tempdata);

  return 0;

}

static int findpeaks(float* section, int* peaklist, int imsize, int* numfound,
		    float back, float thresh)
{

  int pixel=0;

  *numfound=0;

  /* Code here for absorption lines too */
  /* !!! CODE ME !!! */
  
  do{

    /* Look for a line by seeing if pixel is above background by thresh */
    if( *(section+pixel) - back > thresh ){ 

      (*numfound)++;
      pixel++;

      while( *(section+pixel) > *(section+pixel-1) ){
	pixel++;
	if( pixel == imsize-1 )
	  break;
      }
      
      *(peaklist+*(numfound)-1) = pixel; /* Iraf list is 1 based so no - 1 */
      /*printf("%d 1.000\n",pixel);*/

      /* Follow pixel down */
      while( *(section+pixel) < *(section+pixel-1) ||
	     *(section+pixel) - back > thresh ){
	pixel++;
	if( pixel == imsize-1 )
	  break;
      }

    } else {
      
      pixel++;
      if( pixel == imsize-2 )
	break;
    }
  
  } while (pixel < imsize-2);

  return 0;

}



