/* sumspec.c
**
** Code to read in a cirpass reduced MEF spectra file (similar to
** .ms format)
** Returns the fibre position on the slit and the sum of it's
** spectra in the range requested 
**
** A J Dean. 9th April, 2001
**
*/

#include <stdlib.h>
#include "fitswrap.h"
#include "sumspec.h"

extern float* sumspec(const char* filename, int start, int end, int* numfound)
{

  int status=0;                        /* Error handle for cfitsio */
  fitsfile *fptr;                      /* fits object as defined by cfitsio */
  int i=0;                             /* Loop variable */
  int j=0;
  float* data=NULL;                    /* Returned data array */
  float* tempdata=NULL;                /* Image data */
  float tempsum=0;           
  long nx, ny;                         /* Dimensions of image */

  int nbytes;                          /* Size of array required for data */
  int fpixel=1;                        /* First pixel to read */
  int ntoread;                         /* Number of pixels to read */     
  int nullvall=0;                      /* Markers for bad pixels */
  int anynull=0;                       /* as used by cfitsio */

  
  /* Open the input file and find out it's size */
  if ( fits_open_file(&fptr, filename, READONLY, &status) )
    printerror ( status ) ;

  if ( fits_movnam_hdu(fptr, ANY_HDU, "SCI", 1, &status) )
    printerror( status );

  if ( fits_read_key(fptr, TINT, "NAXIS1", &nx, NULL, &status) )
    printerror ( status ) ;
  
  if ( fits_read_key(fptr, TINT, "NAXIS2", &ny, NULL, &status) )
    printerror ( status ) ;
  
  *numfound=ny;

  nbytes=nx*ny*sizeof(float);
  ntoread=nx*ny;

  if ( (tempdata=(float *)malloc(nbytes)) == NULL ){
    
   printf("sumspec: Memory allocation failed!");
   exit(1);
  }

  /* Allocate memory for summed spectra */   
  if ( ( data=(float*)malloc(ny*sizeof(float)) ) == NULL ){
    printf("sumspec: Memory allocation failed!");
    exit(1);
  }

  /* Read in the image */
  if ( fits_read_img(fptr, TFLOAT, fpixel, ntoread, &nullvall, tempdata, 
		     &anynull, &status) )
    printerror( status );

  /* Sum the input spectra over the range required */
  if ( (start < 1) || (end > nx) ){

    printf("sumspec: Range requested is not in input data.\n");
    printf("sumspec: So will sum entire spectrum instead.\n");

    for( j=0; j<ny; j++ ){    /* Sum over apertures */

      tempsum=0.0;

      for( i=0; i<nx; i++ ){   /* Sum over spectra */

	tempsum+=*( tempdata + i + j*nx);

      }

      /* *( data + j ) = tempsum / (end-start+1) ; */
      *( data + j ) = tempsum;
     
    }

  } else {

    for( j=0; j<ny; j++ ){    /* Sum over apertures */

      tempsum=0.0;
      
      for( i=start-1; i<end-1; i++ ){   /* Sum over spectra */

	tempsum+=*( tempdata + i + j*nx);
		
      }

      /* *( data + j ) = tempsum / (end-start+1) ; */
      *( data + j ) = tempsum;
      
    }
 
  }

  /* Close the file freeing memory allocated for fitsfile fptr */
  if ( fits_close_file(fptr, &status) )
    printerror ( status );
  
 
  free(tempdata);
 
  return data;

}


