/* C Program ifudisp
**
** Draws an array of hexagons, scaling the value of each hexagon
** by an input data set
**
** AJ Dean. 30th March, 2000
*/

/* Assumes that (int)num.x produces num irrespective of x hence the 
** round function 
*/

#include <stdlib.h>
#include <math.h>
#include "fitswrap.h"

#define DISPSIZE 1003

static int hexagonx(float ypix, float rad);
static int minmax(float* values, int nvals, float* min, float* max);
static int round(float val);

int main(int argc, char* argv[])
{

  float xpix=0;           /* x values in pixel units */
  float ypix=0;           /* y values in pixel units */
  float xspace=0;         /* x pixel spacing of points on grid */
  float yspace=0;         /* y pixel spacing of points on grid */  
  float* xgrid=0;         /* x values on arbitary ifu grid */
  float* ygrid=0;         /* y values on arbitary ifu grid */
  float yoverx=0;         /* Ratio of the grid axes */
  float mingridx=0;       /* Size of the grid */ 
  float maxgridx=0;
  float mingridy=0;
  float maxgridy=0;
  float xgridsize=0;
  float ygridsize=0;
  float xgridmid=0;
  float ygridmid=0;

  float rad=0;          /* "radial" size of hexagon, pixels */
  int irad=0;           /* int of rad */
  int xlimit=0;         /* size of hexagon in x direction */
  double root3;         /* Root 3 */
  int numlenses=0;      /* Number of lenses to be displayed */
  float* pixvals=NULL;  /* data values for the pixels */
  float* value=NULL;    /* Intensity in the lens */
  float offsetx=0;        /* Pixel values for middle of display */
  float offsety=0;        
  int middle=0;         
  float maxdist=0;      /* max dist object can be from centre in terms of x */

  int i=0;              /* Loop variables */
  int j=0;            
  int gridloop=0;

  FILE *fp;             /* For config file */
  long naxes[2];

  /* check for appropriate input */
  if ( argc < 2 ) {
    fprintf(stderr, 
	    "Usage: %s config_file \n", 
	    argv[0]);
    exit(1);
  }
    
  /* Allocate memory for the pixel values */
  if ((pixvals=(float *)malloc(DISPSIZE*DISPSIZE*sizeof(float))) == NULL ){
    
    printf("ifudisp: memory allocation for pixvals failed\n");
    exit(1);

  }
  
  /* open the config file */
  if ( (fp = fopen(argv[1],"r")) == NULL ){
    printf("Cannot open the configuration file\n");
    exit(1);
  }
    
  
  fscanf(fp,"%d %f %f",&numlenses,&yoverx,&maxdist);    
   
  /* Allocate memory for the ifu grid + values*/
  
  if ((xgrid=(float *)malloc(numlenses*sizeof(float))) == NULL ||
      (ygrid=(float *)malloc(numlenses*sizeof(float))) == NULL ||
      (value=(float *)malloc(numlenses*sizeof(float))) == NULL ){
    
    printf("ifudisp: memory allocation for grid arrays failed\n");
    exit(1);
    
  }  
 
  for(i=0; i<numlenses; i++){
    fscanf(fp,"%f %f %f",xgrid+i,ygrid+i,value+i);
    /*printf("%f %f %f\n",*(xgrid+i),*(ygrid+i),*(value+i));*/
  }

  /* Close the input file containing the file list to be read in */
  fclose(fp); 

  root3=sqrt(3);

  /* Find out the size of the grid */

  if ( minmax(xgrid,numlenses,&mingridx,&maxgridx) ){
    printf("minmax routine failed!\n");
    exit(1);
  }

  if ( minmax(ygrid,numlenses,&mingridy,&maxgridy) ){
    printf("minmax routine failed!\n");
    exit(1);
  }
  
  /* Find out which axis is longest and set spacing accordingly */

  xgridsize=maxgridx-mingridx;
  ygridsize=maxgridy-mingridy;
  xgridmid=(maxgridx+mingridx)/2.0;
  ygridmid=(maxgridy+mingridy)/2.0;

  /* The .1 is a bit of extra space to account for the cases where the middle
   * of the grid falls exactly on a pixel, whereby one end will rap around 1 
   * pixel to the other side if the grid fits exactly into the full width.
   */
  if ( xgridsize*yoverx > ygridsize ){
   
    xspace=(DISPSIZE)/(xgridsize+2.1);
    yspace=xspace*yoverx;

  } else {
    
    yspace=(DISPSIZE)/(ygridsize+2.1);
    xspace=yspace/yoverx;

  }
    
  /*printf("xspace=%f, yspace=%f\n",xspace,yspace);*/

  rad=2.0*xspace/root3;/* Sets rad to fill x direction */
  irad=round(rad);
  
  /*printf("irad = %d\n",irad);*/
  
  middle=((int)(DISPSIZE/2.0))*DISPSIZE + round((DISPSIZE+1)/2.0) - 1;
  offsetx=xgridmid*xspace;
  offsety=ygridmid*yspace;
  
  for(i=0; i<DISPSIZE*DISPSIZE; i++)
    *(pixvals+i)=0;

  for(gridloop=0; gridloop<numlenses; gridloop++){

    /* i and j loop over a hexagon in pixel space */

    for(j=-irad; j<=irad; j++){

      xlimit=hexagonx((float)(j),rad);
      
      for(i=-xlimit; i<=xlimit; i++){

	xpix=i+*(xgrid+gridloop)*xspace;
	ypix=j+*(ygrid+gridloop)*yspace;
       
	*(pixvals + round(xpix-offsetx) + round(ypix-offsety)*DISPSIZE 
	  + middle)=*(value+gridloop); 
	
      }
      
    }

  }


  naxes[0]=naxes[1]=DISPSIZE;
  writefits_float_basic("!testout.fits", naxes, pixvals);
  

  free(xgrid);
  free(ygrid);
  free(pixvals);
  free(value);

  return 0;

}
      
static int hexagonx(float ypix, float rad)
{
  
  float result=0.0; 

  if ( (ypix < -rad/2.0)){
  
    result=(sqrt(3) * (ypix+rad));

  } else if ( (ypix >= -rad/2.0) && (ypix <=rad/2.0) ){
    
    result=(rad * sqrt(3) / 2.0);
    
  } else {

    result=(sqrt(3) * (rad-ypix));
 
  }

  return round(result);

}

static int minmax(float* values, int nvals, float* min, float* max)
{

  int i=0;
  float mintemp=0;
  float maxtemp=0;

  mintemp=*(values);
  maxtemp=*(values);

  for(i=0; i<nvals; i++){
    
    if ( *(values+i) < mintemp ){
      mintemp=*(values+i);
    }
    
    if ( *(values+i) > maxtemp){
      maxtemp=*(values+i);
    }

  }

  *(min)=mintemp;
  *(max)=maxtemp;

  return 0;

}

static int round(float val)
{

  return ( val < 0.0 ? (int)(val-0.5) : (int)(val+0.5) );

}



