/* makemef3.c
 *
 * Makes a CIRPASS MEF file based on a template header file
 * Improved version of makemef2 so that all the headers are correctly 
 * copied across and the template just has its science extension replaced
 *
 * Output file can contain more than 4 image extensions
 * 
 * Assumes that the new file should have a header
 * and a table
 *
 * A J Dean, 17th Dec, 2002.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include "fitswrap.h"

static void copy_header_and_data(fitsfile *source, fitsfile* destination, char* name, int ver);

int main(int argc, char* argv[])
{

  fitsfile *fptr;    
  fitsfile *fptr_template;
  fitsfile *fptr_insert;
  int status=0;
  int numcols=0;
  int nextend=0;
  int i=0;
  int ver=0;
  char name[]="SCI";

  /* Check for appropriate input */
  if (argc < 4) {
    fprintf(stderr, 
            "Usage: %s reffile.fits infile.fits outputfile.fits\n", 
            argv[0]);
    exit(1);
  }
  
  if( fits_create_file(&fptr, argv[3], &status) )
    printerror( status );

  if( fits_open_file(&fptr_template, argv[1], READONLY, &status) )
    printerror ( status ); 

  if( fits_open_file(&fptr_insert, argv[2], READONLY, &status) )
    printerror ( status );

  /* Find out number of extensions in template file */
  if ( fits_read_key(fptr_template, TINT, "NEXTEND", &nextend, NULL, &status) )
      
    printerror ( status ) ;

  /* Primary header */
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) );
   
  /* Table header */
  if( fits_movrel_hdu(fptr_template, 1, NULL, &status) )
    printerror( status );
  
  if( fits_copy_hdu(fptr_template, fptr, 0, &status) )
    printerror( status );

  /* Table data */
  if( fits_get_num_cols(fptr_template, &numcols, &status) )
    printerror( status );

  for( i=1; i<=numcols; i++ ){
    if( fits_copy_col(fptr_template, fptr, i, i, 0, &status) )
      printerror( status );
  }

  /* Actual image extensions */
  ver=1;

  copy_header_and_data(fptr_insert, fptr, name, 1);
   
  /* Do remaining arrays */
  /* Not doing an absolute move to try to keep the code execution
   * reasonably fast */

  for(i=4; i<=nextend+1; i++){

    if(i==4){
      if( fits_movabs_hdu(fptr_template, 4, NULL, &status) )
	printerror( status );
    } else {

      if( fits_movrel_hdu(fptr_template, 1, NULL, &status) )
	printerror( status );

    }
    
    copy_header_and_data(fptr_template, fptr, NULL, 1);

  }

  /* Tidy up */
  if( fits_close_file(fptr, &status) )
    printerror( status );
  
  if( fits_close_file(fptr_template, &status) )
    printerror( status );

  if( fits_close_file(fptr_insert, &status) )
    printerror( status );
   


  return 0;

}

void copy_header_and_data(fitsfile *source, fitsfile* destination, char* name, int ver)
{

  int fpixel=0;
  int nelements=0;
  int bitpix=0;
 
  long nx=0;
  long ny=0;
  long axes[2]={0};
  int naxis=2;
  int nbytes=0;
  float* data=NULL;
  int status=0;

  naxis=2;
  fpixel=1;
  nelements=1;

  /* Copy the header over from the file to be inserted */
  if( fits_copy_hdu(source, destination, 0, &status) )
    printerror( status );
  
  /* Update the extension keywords*/
  if(name!=NULL){
    if ( fits_update_key(destination, TSTRING, "EXTNAME", name, NULL, 
			 &status) )
      printerror( status );      

    if ( fits_update_key(destination, TINT, "EXTVER", &ver, NULL, &status) )
      printerror( status ); 
  }

  /* Find out the image size */
  if ( fits_read_key(source, TLONG, "NAXIS1", &nx, NULL, &status) )     
    printerror ( status ) ; 
  
  if ( fits_read_key(source, TLONG, "NAXIS2", &ny, NULL, &status) )
    printerror ( status ) ;

  if ( fits_read_key(source, TINT, "BITPIX", &bitpix, NULL, &status) )
    printerror ( status ) ;

  axes[0]=nx;
  axes[1]=ny;

  /* Find out size of array required and allocate necessary memory */
  nbytes=nx*ny*sizeof(float);
  nelements=nx*ny;
  
  /* Read in the data */
  if ( (data=(float *)malloc(nbytes)) == NULL ){
    printf("fitswrap, readfits_float: Memory allocation failed!");
    exit(1);
  }
  
  if ( fits_read_img(source, TFLOAT, fpixel, nelements, NULL, data, NULL, &status) )
    printerror( status );
  
  /* And write it out to the new file */
  if ( fits_write_img(destination, TFLOAT, fpixel, nelements, data, &status) )
        printerror( status ); 
  
  free(data);

}
