/* c_get_imsets.c
** c program to extract imsets from a multi-extension fits file
**
** AJ Dean. 27th March, 2000
**
*/

#include <stdlib.h>
#include "fitswrap.h"

/* ------------------ Static function definitions ------------------------- */

static void parseheader(fitsfile* fptr, long* nx, long* ny, int* bitpix,
			int status);

/* ------------------         Main code           ------------------------- */


int main(int argc, char* argv[])
{

  int nimset=0;                   /* nimset! */
  int start_ext=0;                /* where to start copying from */
  int end_ext=0;                  /* where to finish copying from */
  int step=1;                     /* Copying step */
  int count=0;                    /* accumulator */
  int start_header_num=0;         /* header to start copying from */
  int end_header_num=0;           /* header to finish copying from */

  
  float* data=NULL;               /* Contains data */
  int i=0;                        /* Loop variables */ 
  int j=0; 
  long nx=0;                      /* Image dimensions */
  long ny=0;
  int bitpix=0;                   /* Bits per pixel */
  int nextend=0;                  /* Number of extensions in output file */

  int status=0;                   /* Error handle for cfitsio */
  fitsfile *fptr_in;              /* fits object as defined by cfitsio */
  fitsfile *fptr_out;     
  int fpixel=1;
  int nbytes=0;
  int nread=0;                    /* Number of pixels to read */     
  int nullvall=0;                 /* Markers for bad pixels */
  int anynull=0;                  /* as used by cfitsio */
  int dtype=0;                    /* code for data type */

  /* check for appropriate input */
  if ( argc < 5) {
    fprintf(stderr, 
	    "Usage: %s inputfile outputfile nimset start_ext [end_ext] [step]\n", 
	    argv[0]);
    exit(1);
  }

  /* convert chars to type int*/
  sscanf(argv[3],"%i",&nimset);
  sscanf(argv[4],"%i",&start_ext);
  
  if(argc==5){
    end_ext=start_ext;
    step=1;
  }
  
  if(argc==6){
    sscanf(argv[5],"%i",&end_ext);
    step=1;
  }

  if(argc==7){
    sscanf(argv[5],"%i",&end_ext); 
    sscanf(argv[6],"%i",&step); 
  }

  /* Open the output file for writing and sort out it's primary header */

  /* Create the multi-extension fits file */
  if ( fits_create_file(&fptr_out, argv[2], &status) ) 
    printerror( status );       

  /* Create the primary header */
  
  /* by copying over the header from the input file*/

  if ( fits_open_file(&fptr_in, argv[1], READONLY, &status) )
    printerror( status );
  
  if ( fits_copy_hdu(fptr_in, fptr_out, 0, &status) )
    printerror( status );

  /* Sort out the extension numbering */

  nextend=( (start_ext-end_ext+1) / step ) * nimset;

  if ( fits_update_key(fptr_out, TINT, "NEXTEND", &nextend, NULL, &status) )
    printerror( status );

  start_header_num=((start_ext-1)*nimset)+2; /* Zero based indexing */
  end_header_num=((end_ext-1)*nimset)+2;

  count=1;

  for(j=start_header_num; j<=end_header_num; j+=step*nimset){

    for(i=0; i<nimset; i++){

      if ( fits_movabs_hdu(fptr_in, j+i, NULL, &status) ){
	printf("fitswrap, read_imset: can't move to requested header\n");
	printerror ( status );
      }
      

      parseheader(fptr_in, &nx, &ny, &bitpix, status);

      datatype(&bitpix,&dtype);

      /* Work out how much memory we're going to need and allocate it */
      nbytes=(nx)*(ny)*sizeof(float);
      nread=(nx)*(ny);
            
      if ( (data=(float *)malloc(nbytes)) == NULL ){
	printf("fitswrap, read_ME_FITS: Memory allocation failed!");
	exit(1);
      }
      
      /* Read in the data */
      
      if ( fits_read_img(fptr_in, dtype, fpixel, nread, &nullvall, 
			  data, &anynull, &status) ){
	printf("fitswrap, read_imset: unable to read in data\n");
	printerror( status );
      }
 
      /* Write out the data */
      
      if ( fits_copy_hdu(fptr_in, fptr_out, 0, &status) )
	printerror( status );
      
      if ( fits_update_key(fptr_out, TINT, "EXTVER", &count, NULL, 
			   &status) )
	printerror( status );
     
      if ( fits_write_img(fptr_out, dtype, fpixel, nread, data, 
			  &status) )
	printerror( status );  

      free(data);
     
    } 
  
    count++;
      
  }

  
  if ( fits_close_file(fptr_in, &status) )
    printerror( status );

  if ( fits_close_file(fptr_out, &status) )
    printerror( status );  
  
  return 0;

}

/* ------------------ Static Functions ----------------------------------- */

void parseheader(fitsfile* fptr, long* nx, long* ny, int* bitpix, int status)
{

 char comment[FLEN_COMMENT];          /* Fits header comment */
 
 if ( fits_read_key(fptr, TLONG, "NAXIS1", nx, &comment[0], &status) )
      
   printerror ( status ) ; 

 if ( fits_read_key(fptr, TLONG, "NAXIS2", ny, &comment[0], &status) )
      
   printerror ( status ) ;
 
 if ( fits_read_key(fptr, TINT, "BITPIX", bitpix, &comment[0], &status) )
      
   printerror ( status ) ;

}







