/* cirparith.c
 *
 * Does image arithmetic on cirpass MEF fits files
 *
 * AJ Dean, 16th September, 2002
 *
 */

#include <stdlib.h>
#include <string.h>
#include "fitswrap.h" 

int main(int argc, char* argv[])
{

  char* exttypes[]={"SCI","VAR","DQ","TIME"};
  int   numexts=4;
  int   i,j;
  int   nextend,next;
  int   nextend2;
  long  nx,ny;
  float* operand1[4]; /* [] indexes exts above */
  float* operand2[4];
  float* scidata;
  float* vardata;
  float* dqdata;
  float* timedata;
  float* data[4];
  float number;

  int imsize=0;
  int naxis=2;
  long naxes[2];
  
  /* Check for appropriate input */
  if (argc < 5) {
  
      fprintf(stderr, 
          "Usage: %s operand1.fits operator[+|-|x|/] operand2.fits outfile.fits\n", 
           argv[0]);
        exit(1);
  
  } 

  /* find out if second operand is image or number */
  if ( strstr (argv[3], ".fits" ) == NULL &&  
       strstr (argv[3], ".FITS" ) == NULL ){
  
    
    /* No .fits i.e. we were given a number */
    if(sscanf(argv[3],"%f",&number)==0){
      printf("cirparith: you didn't give me a number or a fits file!\n");
      exit(0);
    }

    /* Read in the data */
    for(i=0; i<numexts; i++){
      
      operand1[i]=read_ME_FITS(argv[1], exttypes[i], &nx, &ny, &nextend);  

    }

    next=nextend/NIMSET;
    imsize=nx*ny;
    naxes[0]=nx;
    naxes[1]=ny;
    
    /* Allocate memory */
    if ( (scidata=(float *)malloc(sizeof(float)*imsize*next))    == NULL ||
	 (vardata=(float *)malloc(sizeof(float)*imsize*next))  == NULL ||
	 (dqdata=(float *)malloc(sizeof(float)*imsize*next))  == NULL ||
	 (timedata=(float *)malloc(sizeof(float)*imsize*next))  == NULL ){
      printf("cirparith: Memory allocation failed\n");
      exit(1);
    }
    
    /* Still not checked but no need to tell the world!
    printf("cirparith: Warning, var arithmetic coding not checked!\n");*/
    printf("cirparith: Read in data, doing the arithmetic\n");
    
    if( strcmp(argv[2],"+")==0 ){
      
      for(j=0; j<imsize*next; j++){
      
	*(scidata+j)=*(operand1[0]+j)+number;
	*(vardata+j)=*(operand1[1]+j);
	*(dqdata+j)=*(operand1[2]+j);
	*(timedata+j)=*(operand1[3]+j);
	
      }
      
    } else if( strcmp(argv[2],"-")==0 ){
      
      for(j=0; j<imsize*next; j++){
	
	*(scidata+j)=*(operand1[0]+j)-number;
	*(vardata+j)=*(operand1[1]+j);
	*(dqdata+j)=*(operand1[2]+j);
	*(timedata+j)=*(operand1[3]+j);
      
      }
  
    } else if( strcmp(argv[2],"*")==0 ){
      
      for(j=0; j<imsize*next; j++){
	
	*(scidata+j)=*(operand1[0]+j)*number;
	*(vardata+j)=*(operand1[1]+j)*number; 
	*(dqdata+j)=*(operand1[2]+j);
	*(timedata+j)=*(operand1[3]+j);
      
      }
    
    } else if( strcmp(argv[2],"/")==0 ){
    
      for(j=0; j<imsize*next; j++){
      
	*(scidata+j)=*(operand1[0]+j) / number;
	*(vardata+j)=*(operand1[1]+j) / number;
	*(dqdata+j)=*(operand1[2]+j);
	*(timedata+j)=*(operand1[3]+j);
	
      }
    
    } else {
    
      printf("cirparith: ERROR! Unrecognised operator!\n");
      exit(1);
    }
    
    data[0]=scidata;
    data[1]=vardata;
    data[2]=dqdata;
    data[3]=timedata;
    
    /* Write out the new image, using the header and table from the 
       original file */
    printf("cirparith: Writing out new image\n");
    write_cirp_mef(argv[1], argv[4], exttypes, data, numexts,
		   naxis, naxes, next);
		    
    /* free all dynamically allocated data */
    for(i=0; i<numexts; i++){
      free(operand1[i]);
    }
    
  } else {
    
    /* Read in the data */
    for(i=0; i<numexts; i++){
      
      operand1[i]=read_ME_FITS(argv[1], exttypes[i], &nx, &ny, &nextend);
      
      operand2[i]=read_ME_FITS(argv[3], exttypes[i], &nx, &ny, &nextend2);
   
    }
  
    /* Check the files have the same number of extensions */
    if(nextend!=nextend2){
      printf("cirparith: ** Warning! Input files do not have the same number of extensions! **\ncirparith: ** Will just use the first four image extensions. **\n");
      nextend=4;
      nextend2=4;
    }

    next=nextend/NIMSET;
    imsize=nx*ny;
    naxes[0]=nx;
    naxes[1]=ny;
    
    /* Allocate memory */
    if ( (scidata=(float *)malloc(sizeof(float)*imsize*next))    == NULL ||
	 (vardata=(float *)malloc(sizeof(float)*imsize*next))  == NULL ||
	 (dqdata=(float *)malloc(sizeof(float)*imsize*next))  == NULL ||
	 (timedata=(float *)malloc(sizeof(float)*imsize*next))  == NULL ){
      printf("cirparith: Memory allocation failed\n");
      exit(1);
    }
    
    /* Still not checked but no need to tell the world!
    printf("cirparith: Warning, var arithmetic coding not checked!\n");*/
    printf("cirparith: Read in data, doing the arithmetic\n");
    
    if( strcmp(argv[2],"+")==0 ){
      
      for(j=0; j<imsize*next; j++){
      
	*(scidata+j)=*(operand1[0]+j)+*(operand2[0]+j);
	*(vardata+j)=*(operand1[1]+j)+*(operand2[1]+j);
	*(dqdata+j)=*(operand1[2]+j)+*(operand2[2]+j);
	*(timedata+j)=*(operand1[3]+j);
	
      }
      
    } else if( strcmp(argv[2],"-")==0 ){
      
      for(j=0; j<imsize*next; j++){
	
	*(scidata+j)=*(operand1[0]+j)-*(operand2[0]+j);
	*(vardata+j)=*(operand1[1]+j)+*(operand2[1]+j);
	*(dqdata+j)=*(operand1[2]+j)+*(operand2[2]+j);
	*(timedata+j)=*(operand1[3]+j);
      
      }
  
    } else if( strcmp(argv[2],"*")==0 ){
      
      for(j=0; j<imsize*next; j++){
	
	*(scidata+j)=*(operand1[0]+j)**(operand2[0]+j);
	*(vardata+j)=(*(operand1[0]+j) * *(operand2[0]+j)) * 
	           (*(operand1[0]+j)**(operand2[0]+j)) *
	           (*(operand1[1]+j) / *(operand1[0]+j) / *(operand1[0]+j) + 
		    *(operand2[1]+j) / *(operand2[0]+j) / *(operand2[0]+j));
	*(dqdata+j)=*(operand1[2]+j)+*(operand2[2]+j);
	*(timedata+j)=*(operand1[3]+j);
      
      }
    
    } else if( strcmp(argv[2],"/")==0 ){
    
      for(j=0; j<imsize*next; j++){
      
	*(scidata+j)=*(operand1[0]+j) / *(operand2[0]+j);
	*(vardata+j)=(*(operand1[0]+j) / *(operand2[0]+j)) * 
                     (*(operand1[0]+j)**(operand2[0]+j)) *
	             (*(operand1[1]+j) / *(operand1[0]+j) / *(operand1[0]+j) + 
	             *(operand2[1]+j) / *(operand2[0]+j) / *(operand2[0]+j));
	*(dqdata+j)=*(operand1[2]+j)+*(operand2[2]+j);
	*(timedata+j)=*(operand1[3]+j);
	
      }
    
    } else {
    
      printf("cirparith: ERROR! Unrecognised operator!\n");
      exit(1);
    }
    
    data[0]=scidata;
    data[1]=vardata;
    data[2]=dqdata;
    data[3]=timedata;
    
    /* Write out the new image, using the header and table from the 
       original file */
    printf("cirparith: Writing out new image\n");
    write_cirp_mef(argv[1], argv[4], exttypes, data, numexts,
		   naxis, naxes, next);
		    
    /* free all dynamically allocated data */
    for(i=0; i<numexts; i++){
      free(operand1[i]);
      free(operand2[i]);
    }
		    	    
  }	  
  
  free(scidata);
  free(vardata);
  free(dqdata);
  free(timedata);

  return 0;

}
